/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lakeformation.model.AccessDeniedException;
import software.amazon.awssdk.services.lakeformation.model.AddLfTagsToResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.AddLfTagsToResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.AlreadyExistsException;
import software.amazon.awssdk.services.lakeformation.model.BatchGrantPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.BatchGrantPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.BatchRevokePermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.BatchRevokePermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.ConcurrentModificationException;
import software.amazon.awssdk.services.lakeformation.model.CreateLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.CreateLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.DeleteLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.DeleteLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.DeregisterResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.DeregisterResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.DescribeResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.DescribeResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.EntityNotFoundException;
import software.amazon.awssdk.services.lakeformation.model.GetDataLakeSettingsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetDataLakeSettingsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest;
import software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathResponse;
import software.amazon.awssdk.services.lakeformation.model.GetLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.GetLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.GetResourceLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetResourceLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.GlueEncryptionException;
import software.amazon.awssdk.services.lakeformation.model.GrantPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.GrantPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.InternalServiceException;
import software.amazon.awssdk.services.lakeformation.model.InvalidInputException;
import software.amazon.awssdk.services.lakeformation.model.LakeFormationException;
import software.amazon.awssdk.services.lakeformation.model.LakeFormationRequest;
import software.amazon.awssdk.services.lakeformation.model.ListLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest;
import software.amazon.awssdk.services.lakeformation.model.ListResourcesResponse;
import software.amazon.awssdk.services.lakeformation.model.OperationTimeoutException;
import software.amazon.awssdk.services.lakeformation.model.PutDataLakeSettingsRequest;
import software.amazon.awssdk.services.lakeformation.model.PutDataLakeSettingsResponse;
import software.amazon.awssdk.services.lakeformation.model.RegisterResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.RegisterResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.RemoveLfTagsFromResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.RemoveLfTagsFromResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.ResourceNumberLimitExceededException;
import software.amazon.awssdk.services.lakeformation.model.RevokePermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.RevokePermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.SearchDatabasesByLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.SearchDatabasesByLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.SearchTablesByLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.SearchTablesByLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateResourceResponse;
import software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathPublisher;
import software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsPublisher;
import software.amazon.awssdk.services.lakeformation.paginators.ListResourcesPublisher;
import software.amazon.awssdk.services.lakeformation.transform.AddLfTagsToResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.BatchGrantPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.BatchRevokePermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.CreateLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeleteLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeregisterResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DescribeResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetDataLakeSettingsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetEffectivePermissionsForPathRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetResourceLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GrantPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.PutDataLakeSettingsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RegisterResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RemoveLfTagsFromResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RevokePermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.SearchDatabasesByLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.SearchTablesByLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LakeFormationAsyncClient}.
 *
 * @see LakeFormationAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLakeFormationAsyncClient implements LakeFormationAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLakeFormationAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLakeFormationAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Attaches one or more tags to an existing resource.
     * </p>
     *
     * @param addLfTagsToResourceRequest
     * @return A Java Future containing the result of the AddLFTagsToResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.AddLFTagsToResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/AddLFTagsToResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddLfTagsToResourceResponse> addLFTagsToResource(
            AddLfTagsToResourceRequest addLfTagsToResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addLfTagsToResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddLFTagsToResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddLfTagsToResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddLfTagsToResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddLfTagsToResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddLfTagsToResourceRequest, AddLfTagsToResourceResponse>()
                            .withOperationName("AddLFTagsToResource")
                            .withMarshaller(new AddLfTagsToResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addLfTagsToResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addLfTagsToResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AddLfTagsToResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Batch operation to grant permissions to the principal.
     * </p>
     *
     * @param batchGrantPermissionsRequest
     * @return A Java Future containing the result of the BatchGrantPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.BatchGrantPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/BatchGrantPermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGrantPermissionsResponse> batchGrantPermissions(
            BatchGrantPermissionsRequest batchGrantPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGrantPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGrantPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGrantPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGrantPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGrantPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGrantPermissionsRequest, BatchGrantPermissionsResponse>()
                            .withOperationName("BatchGrantPermissions")
                            .withMarshaller(new BatchGrantPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGrantPermissionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchGrantPermissionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<BatchGrantPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Batch operation to revoke permissions from the principal.
     * </p>
     *
     * @param batchRevokePermissionsRequest
     * @return A Java Future containing the result of the BatchRevokePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.BatchRevokePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/BatchRevokePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchRevokePermissionsResponse> batchRevokePermissions(
            BatchRevokePermissionsRequest batchRevokePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchRevokePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchRevokePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchRevokePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchRevokePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchRevokePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchRevokePermissionsRequest, BatchRevokePermissionsResponse>()
                            .withOperationName("BatchRevokePermissions")
                            .withMarshaller(new BatchRevokePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchRevokePermissionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchRevokePermissionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<BatchRevokePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a tag with the specified name and values.
     * </p>
     *
     * @param createLfTagRequest
     * @return A Java Future containing the result of the CreateLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>ResourceNumberLimitExceededException A resource numerical limit was exceeded.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.CreateLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/CreateLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLfTagResponse> createLFTag(CreateLfTagRequest createLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLfTagRequest, CreateLfTagResponse>()
                            .withOperationName("CreateLFTag").withMarshaller(new CreateLfTagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLfTagRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLfTagRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified tag key name. If the attribute key does not exist or the tag does not exist, then the
     * operation will not do anything. If the attribute key exists, then the operation checks if any resources are
     * tagged with this attribute key, if yes, the API throws a 400 Exception with the message "Delete not allowed" as
     * the tag key is still attached with resources. You can consider untagging resources with this tag key.
     * </p>
     *
     * @param deleteLfTagRequest
     * @return A Java Future containing the result of the DeleteLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DeleteLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeleteLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLfTagResponse> deleteLFTag(DeleteLfTagRequest deleteLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLfTagRequest, DeleteLfTagResponse>()
                            .withOperationName("DeleteLFTag").withMarshaller(new DeleteLfTagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLfTagRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLfTagRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters the resource as managed by the Data Catalog.
     * </p>
     * <p>
     * When you deregister a path, Lake Formation removes the path from the inline policy attached to your
     * service-linked role.
     * </p>
     *
     * @param deregisterResourceRequest
     * @return A Java Future containing the result of the DeregisterResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DeregisterResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeregisterResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterResourceResponse> deregisterResource(DeregisterResourceRequest deregisterResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterResourceRequest, DeregisterResourceResponse>()
                            .withOperationName("DeregisterResource")
                            .withMarshaller(new DeregisterResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deregisterResourceRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeregisterResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current data access role for the given resource registered in AWS Lake Formation.
     * </p>
     *
     * @param describeResourceRequest
     * @return A Java Future containing the result of the DescribeResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DescribeResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DescribeResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeResourceResponse> describeResource(DescribeResourceRequest describeResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeResourceRequest, DescribeResourceResponse>()
                            .withOperationName("DescribeResource")
                            .withMarshaller(new DescribeResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of the data lake administrators of a Lake Formation-managed data lake.
     * </p>
     *
     * @param getDataLakeSettingsRequest
     * @return A Java Future containing the result of the GetDataLakeSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetDataLakeSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetDataLakeSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataLakeSettingsResponse> getDataLakeSettings(
            GetDataLakeSettingsRequest getDataLakeSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataLakeSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataLakeSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataLakeSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataLakeSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeSettingsRequest, GetDataLakeSettingsResponse>()
                            .withOperationName("GetDataLakeSettings")
                            .withMarshaller(new GetDataLakeSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDataLakeSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDataLakeSettingsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetDataLakeSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the Lake Formation permissions for a specified table or database resource located at a path in Amazon S3.
     * <code>GetEffectivePermissionsForPath</code> will not return databases and tables if the catalog is encrypted.
     * </p>
     *
     * @param getEffectivePermissionsForPathRequest
     * @return A Java Future containing the result of the GetEffectivePermissionsForPath operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetEffectivePermissionsForPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetEffectivePermissionsForPath"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEffectivePermissionsForPathResponse> getEffectivePermissionsForPath(
            GetEffectivePermissionsForPathRequest getEffectivePermissionsForPathRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEffectivePermissionsForPathRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEffectivePermissionsForPath");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEffectivePermissionsForPathResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEffectivePermissionsForPathResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEffectivePermissionsForPathResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEffectivePermissionsForPathRequest, GetEffectivePermissionsForPathResponse>()
                            .withOperationName("GetEffectivePermissionsForPath")
                            .withMarshaller(new GetEffectivePermissionsForPathRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEffectivePermissionsForPathRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEffectivePermissionsForPathRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetEffectivePermissionsForPathResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the Lake Formation permissions for a specified table or database resource located at a path in Amazon S3.
     * <code>GetEffectivePermissionsForPath</code> will not return databases and tables if the catalog is encrypted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getEffectivePermissionsForPath(software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathPublisher publisher = client.getEffectivePermissionsForPathPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathPublisher publisher = client.getEffectivePermissionsForPathPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getEffectivePermissionsForPath(software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest)}
     * operation.</b>
     * </p>
     *
     * @param getEffectivePermissionsForPathRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetEffectivePermissionsForPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetEffectivePermissionsForPath"
     *      target="_top">AWS API Documentation</a>
     */
    public GetEffectivePermissionsForPathPublisher getEffectivePermissionsForPathPaginator(
            GetEffectivePermissionsForPathRequest getEffectivePermissionsForPathRequest) {
        return new GetEffectivePermissionsForPathPublisher(this, applyPaginatorUserAgent(getEffectivePermissionsForPathRequest));
    }

    /**
     * <p>
     * Returns a tag definition.
     * </p>
     *
     * @param getLfTagRequest
     * @return A Java Future containing the result of the GetLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLfTagResponse> getLFTag(GetLfTagRequest getLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLfTagRequest, GetLfTagResponse>().withOperationName("GetLFTag")
                            .withMarshaller(new GetLfTagRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLfTagRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLfTagRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the tags applied to a resource.
     * </p>
     *
     * @param getResourceLfTagsRequest
     * @return A Java Future containing the result of the GetResourceLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetResourceLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetResourceLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceLfTagsResponse> getResourceLFTags(GetResourceLfTagsRequest getResourceLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceLfTagsRequest, GetResourceLfTagsResponse>()
                            .withOperationName("GetResourceLFTags")
                            .withMarshaller(new GetResourceLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourceLfTagsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResourceLfTagsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetResourceLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permissions to the principal to access metadata in the Data Catalog and data organized in underlying data
     * storage such as Amazon S3.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     *
     * @param grantPermissionsRequest
     * @return A Java Future containing the result of the GrantPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GrantPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GrantPermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GrantPermissionsResponse> grantPermissions(GrantPermissionsRequest grantPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, grantPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GrantPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GrantPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GrantPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GrantPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GrantPermissionsRequest, GrantPermissionsResponse>()
                            .withOperationName("GrantPermissions")
                            .withMarshaller(new GrantPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(grantPermissionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = grantPermissionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GrantPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags that the requester has permission to view.
     * </p>
     *
     * @param listLfTagsRequest
     * @return A Java Future containing the result of the ListLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListLFTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLfTagsResponse> listLFTags(ListLfTagsRequest listLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLfTagsRequest, ListLfTagsResponse>().withOperationName("ListLFTags")
                            .withMarshaller(new ListLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLfTagsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLfTagsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the principal permissions on the resource, filtered by the permissions of the caller. For
     * example, if you are granted an ALTER permission, you are able to see only the principal permissions for ALTER.
     * </p>
     * <p>
     * This operation returns only those permissions that have been explicitly granted.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     *
     * @param listPermissionsRequest
     * @return A Java Future containing the result of the ListPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionsResponse> listPermissions(ListPermissionsRequest listPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                            .withOperationName("ListPermissions")
                            .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPermissionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPermissionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the principal permissions on the resource, filtered by the permissions of the caller. For
     * example, if you are granted an ALTER permission, you are able to see only the principal permissions for ALTER.
     * </p>
     * <p>
     * This operation returns only those permissions that have been explicitly granted.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPermissions(software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsPublisher publisher = client.listPermissionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsPublisher publisher = client.listPermissionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lakeformation.model.ListPermissionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lakeformation.model.ListPermissionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPermissions(software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPermissionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    public ListPermissionsPublisher listPermissionsPaginator(ListPermissionsRequest listPermissionsRequest) {
        return new ListPermissionsPublisher(this, applyPaginatorUserAgent(listPermissionsRequest));
    }

    /**
     * <p>
     * Lists the resources registered to be managed by the Data Catalog.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                            .withOperationName("ListResources")
                            .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourcesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResourcesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources registered to be managed by the Data Catalog.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResources(software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lakeformation.model.ListResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lakeformation.model.ListResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    public ListResourcesPublisher listResourcesPaginator(ListResourcesRequest listResourcesRequest) {
        return new ListResourcesPublisher(this, applyPaginatorUserAgent(listResourcesRequest));
    }

    /**
     * <p>
     * Sets the list of data lake administrators who have admin privileges on all resources managed by Lake Formation.
     * For more information on admin privileges, see <a
     * href="https://docs.aws.amazon.com/lake-formation/latest/dg/lake-formation-permissions.html">Granting Lake
     * Formation Permissions</a>.
     * </p>
     * <p>
     * This API replaces the current list of data lake admins with the new list being passed. To add an admin, fetch the
     * current list and add the new admin to that list and pass that list in this API.
     * </p>
     *
     * @param putDataLakeSettingsRequest
     * @return A Java Future containing the result of the PutDataLakeSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.PutDataLakeSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/PutDataLakeSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDataLakeSettingsResponse> putDataLakeSettings(
            PutDataLakeSettingsRequest putDataLakeSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDataLakeSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDataLakeSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDataLakeSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDataLakeSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDataLakeSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDataLakeSettingsRequest, PutDataLakeSettingsResponse>()
                            .withOperationName("PutDataLakeSettings")
                            .withMarshaller(new PutDataLakeSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putDataLakeSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putDataLakeSettingsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutDataLakeSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers the resource as managed by the Data Catalog.
     * </p>
     * <p>
     * To add or update data, Lake Formation needs read/write access to the chosen Amazon S3 path. Choose a role that
     * you know has permission to do this, or choose the AWSServiceRoleForLakeFormationDataAccess service-linked role.
     * When you register the first Amazon S3 path, the service-linked role and a new inline policy are created on your
     * behalf. Lake Formation adds the first path to the inline policy and attaches it to the service-linked role. When
     * you register subsequent paths, Lake Formation adds the path to the existing policy.
     * </p>
     * <p>
     * The following request registers a new location and gives AWS Lake Formation permission to use the service-linked
     * role to access that location.
     * </p>
     * <p>
     * <code>ResourceArn = arn:aws:s3:::my-bucket UseServiceLinkedRole = true</code>
     * </p>
     * <p>
     * If <code>UseServiceLinkedRole</code> is not set to true, you must provide or set the <code>RoleArn</code>:
     * </p>
     * <p>
     * <code>arn:aws:iam::12345:role/my-data-access-role</code>
     * </p>
     *
     * @param registerResourceRequest
     * @return A Java Future containing the result of the RegisterResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AlreadyExistsException A resource to be created or added already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.RegisterResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RegisterResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterResourceResponse> registerResource(RegisterResourceRequest registerResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterResourceRequest, RegisterResourceResponse>()
                            .withOperationName("RegisterResource")
                            .withMarshaller(new RegisterResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = registerResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RegisterResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from the resource. Only database, table, or tableWithColumns resource are allowed. To tag columns,
     * use the column inclusion list in <code>tableWithColumns</code> to specify column input.
     * </p>
     *
     * @param removeLfTagsFromResourceRequest
     * @return A Java Future containing the result of the RemoveLFTagsFromResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.RemoveLFTagsFromResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RemoveLFTagsFromResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveLfTagsFromResourceResponse> removeLFTagsFromResource(
            RemoveLfTagsFromResourceRequest removeLfTagsFromResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeLfTagsFromResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveLFTagsFromResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveLfTagsFromResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveLfTagsFromResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveLfTagsFromResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveLfTagsFromResourceRequest, RemoveLfTagsFromResourceResponse>()
                            .withOperationName("RemoveLFTagsFromResource")
                            .withMarshaller(new RemoveLfTagsFromResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeLfTagsFromResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeLfTagsFromResourceRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RemoveLfTagsFromResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes permissions to the principal to access metadata in the Data Catalog and data organized in underlying data
     * storage such as Amazon S3.
     * </p>
     *
     * @param revokePermissionsRequest
     * @return A Java Future containing the result of the RevokePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.RevokePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RevokePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokePermissionsResponse> revokePermissions(RevokePermissionsRequest revokePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokePermissionsRequest, RevokePermissionsResponse>()
                            .withOperationName("RevokePermissions")
                            .withMarshaller(new RevokePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(revokePermissionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = revokePermissionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RevokePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation allows a search on <code>DATABASE</code> resources by <code>TagCondition</code>. This operation is
     * used by admins who want to grant user permissions on certain <code>TagConditions</code>. Before making a grant,
     * the admin can use <code>SearchDatabasesByTags</code> to find all resources where the given
     * <code>TagConditions</code> are valid to verify whether the returned resources can be shared.
     * </p>
     *
     * @param searchDatabasesByLfTagsRequest
     * @return A Java Future containing the result of the SearchDatabasesByLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.SearchDatabasesByLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/SearchDatabasesByLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchDatabasesByLfTagsResponse> searchDatabasesByLFTags(
            SearchDatabasesByLfTagsRequest searchDatabasesByLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchDatabasesByLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchDatabasesByLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchDatabasesByLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchDatabasesByLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchDatabasesByLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchDatabasesByLfTagsRequest, SearchDatabasesByLfTagsResponse>()
                            .withOperationName("SearchDatabasesByLFTags")
                            .withMarshaller(new SearchDatabasesByLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchDatabasesByLfTagsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchDatabasesByLfTagsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SearchDatabasesByLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation allows a search on <code>TABLE</code> resources by <code>LFTag</code>s. This will be used by
     * admins who want to grant user permissions on certain LFTags. Before making a grant, the admin can use
     * <code>SearchTablesByLFTags</code> to find all resources where the given <code>LFTag</code>s are valid to verify
     * whether the returned resources can be shared.
     * </p>
     *
     * @param searchTablesByLfTagsRequest
     * @return A Java Future containing the result of the SearchTablesByLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.SearchTablesByLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/SearchTablesByLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchTablesByLfTagsResponse> searchTablesByLFTags(
            SearchTablesByLfTagsRequest searchTablesByLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchTablesByLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchTablesByLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchTablesByLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchTablesByLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchTablesByLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchTablesByLfTagsRequest, SearchTablesByLfTagsResponse>()
                            .withOperationName("SearchTablesByLFTags")
                            .withMarshaller(new SearchTablesByLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchTablesByLfTagsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchTablesByLfTagsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<SearchTablesByLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the list of possible values for the specified tag key. If the tag does not exist, the operation throws an
     * EntityNotFoundException. The values in the delete key values will be deleted from list of possible values. If any
     * value in the delete key values is attached to a resource, then API errors out with a 400 Exception -
     * "Update not allowed". Untag the attribute before deleting the tag key's value.
     * </p>
     *
     * @param updateLfTagRequest
     * @return A Java Future containing the result of the UpdateLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLfTagResponse> updateLFTag(UpdateLfTagRequest updateLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLfTagRequest, UpdateLfTagResponse>()
                            .withOperationName("UpdateLFTag").withMarshaller(new UpdateLfTagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLfTagRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateLfTagRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the data access role used for vending access to the given (registered) resource in AWS Lake Formation.
     * </p>
     *
     * @param updateResourceRequest
     * @return A Java Future containing the result of the UpdateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceResponse> updateResource(UpdateResourceRequest updateResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceRequest, UpdateResourceResponse>()
                            .withOperationName("UpdateResource")
                            .withMarshaller(new UpdateResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LakeFormationException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNumberLimitExceededException")
                                .exceptionBuilderSupplier(ResourceNumberLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityNotFoundException")
                                .exceptionBuilderSupplier(EntityNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlueEncryptionException")
                                .exceptionBuilderSupplier(GlueEncryptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationTimeoutException")
                                .exceptionBuilderSupplier(OperationTimeoutException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LakeFormationRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
