/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lakeformation.model.AccessDeniedException;
import software.amazon.awssdk.services.lakeformation.model.AddLfTagsToResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.AddLfTagsToResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.AlreadyExistsException;
import software.amazon.awssdk.services.lakeformation.model.BatchGrantPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.BatchGrantPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.BatchRevokePermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.BatchRevokePermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.ConcurrentModificationException;
import software.amazon.awssdk.services.lakeformation.model.CreateLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.CreateLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.DeleteLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.DeleteLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.DeregisterResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.DeregisterResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.DescribeResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.DescribeResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.EntityNotFoundException;
import software.amazon.awssdk.services.lakeformation.model.GetDataLakeSettingsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetDataLakeSettingsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest;
import software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathResponse;
import software.amazon.awssdk.services.lakeformation.model.GetLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.GetLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.GetResourceLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetResourceLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.GlueEncryptionException;
import software.amazon.awssdk.services.lakeformation.model.GrantPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.GrantPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.InternalServiceException;
import software.amazon.awssdk.services.lakeformation.model.InvalidInputException;
import software.amazon.awssdk.services.lakeformation.model.LakeFormationException;
import software.amazon.awssdk.services.lakeformation.model.LakeFormationRequest;
import software.amazon.awssdk.services.lakeformation.model.ListLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest;
import software.amazon.awssdk.services.lakeformation.model.ListResourcesResponse;
import software.amazon.awssdk.services.lakeformation.model.OperationTimeoutException;
import software.amazon.awssdk.services.lakeformation.model.PutDataLakeSettingsRequest;
import software.amazon.awssdk.services.lakeformation.model.PutDataLakeSettingsResponse;
import software.amazon.awssdk.services.lakeformation.model.RegisterResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.RegisterResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.RemoveLfTagsFromResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.RemoveLfTagsFromResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.ResourceNumberLimitExceededException;
import software.amazon.awssdk.services.lakeformation.model.RevokePermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.RevokePermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.SearchDatabasesByLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.SearchDatabasesByLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.SearchTablesByLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.SearchTablesByLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateResourceResponse;
import software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathIterable;
import software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsIterable;
import software.amazon.awssdk.services.lakeformation.paginators.ListResourcesIterable;
import software.amazon.awssdk.services.lakeformation.transform.AddLfTagsToResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.BatchGrantPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.BatchRevokePermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.CreateLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeleteLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeregisterResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DescribeResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetDataLakeSettingsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetEffectivePermissionsForPathRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetResourceLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GrantPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.PutDataLakeSettingsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RegisterResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RemoveLfTagsFromResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RevokePermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.SearchDatabasesByLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.SearchTablesByLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LakeFormationClient}.
 *
 * @see LakeFormationClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLakeFormationClient implements LakeFormationClient {
    private static final Logger log = Logger.loggerFor(DefaultLakeFormationClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLakeFormationClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Attaches one or more tags to an existing resource.
     * </p>
     *
     * @param addLfTagsToResourceRequest
     * @return Result of the AddLFTagsToResource operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws ConcurrentModificationException
     *         Two processes are trying to modify a resource simultaneously.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.AddLFTagsToResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/AddLFTagsToResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddLfTagsToResourceResponse addLFTagsToResource(AddLfTagsToResourceRequest addLfTagsToResourceRequest)
            throws EntityNotFoundException, InvalidInputException, InternalServiceException, OperationTimeoutException,
            AccessDeniedException, ConcurrentModificationException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddLfTagsToResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddLfTagsToResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addLfTagsToResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddLFTagsToResource");

            return clientHandler.execute(new ClientExecutionParams<AddLfTagsToResourceRequest, AddLfTagsToResourceResponse>()
                    .withOperationName("AddLFTagsToResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addLfTagsToResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddLfTagsToResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Batch operation to grant permissions to the principal.
     * </p>
     *
     * @param batchGrantPermissionsRequest
     * @return Result of the BatchGrantPermissions operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.BatchGrantPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/BatchGrantPermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGrantPermissionsResponse batchGrantPermissions(BatchGrantPermissionsRequest batchGrantPermissionsRequest)
            throws InvalidInputException, OperationTimeoutException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGrantPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchGrantPermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGrantPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGrantPermissions");

            return clientHandler.execute(new ClientExecutionParams<BatchGrantPermissionsRequest, BatchGrantPermissionsResponse>()
                    .withOperationName("BatchGrantPermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchGrantPermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGrantPermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Batch operation to revoke permissions from the principal.
     * </p>
     *
     * @param batchRevokePermissionsRequest
     * @return Result of the BatchRevokePermissions operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.BatchRevokePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/BatchRevokePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchRevokePermissionsResponse batchRevokePermissions(BatchRevokePermissionsRequest batchRevokePermissionsRequest)
            throws InvalidInputException, OperationTimeoutException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchRevokePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchRevokePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchRevokePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchRevokePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchRevokePermissionsRequest, BatchRevokePermissionsResponse>()
                            .withOperationName("BatchRevokePermissions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(batchRevokePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchRevokePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a tag with the specified name and values.
     * </p>
     *
     * @param createLfTagRequest
     * @return Result of the CreateLFTag operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws ResourceNumberLimitExceededException
     *         A resource numerical limit was exceeded.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.CreateLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/CreateLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLfTagResponse createLFTag(CreateLfTagRequest createLfTagRequest) throws EntityNotFoundException,
            InvalidInputException, ResourceNumberLimitExceededException, InternalServiceException, OperationTimeoutException,
            AccessDeniedException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLfTagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLFTag");

            return clientHandler.execute(new ClientExecutionParams<CreateLfTagRequest, CreateLfTagResponse>()
                    .withOperationName("CreateLFTag").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLfTagRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLfTagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified tag key name. If the attribute key does not exist or the tag does not exist, then the
     * operation will not do anything. If the attribute key exists, then the operation checks if any resources are
     * tagged with this attribute key, if yes, the API throws a 400 Exception with the message "Delete not allowed" as
     * the tag key is still attached with resources. You can consider untagging resources with this tag key.
     * </p>
     *
     * @param deleteLfTagRequest
     * @return Result of the DeleteLFTag operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.DeleteLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeleteLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLfTagResponse deleteLFTag(DeleteLfTagRequest deleteLfTagRequest) throws EntityNotFoundException,
            InvalidInputException, InternalServiceException, OperationTimeoutException, AccessDeniedException,
            AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLfTagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLFTag");

            return clientHandler.execute(new ClientExecutionParams<DeleteLfTagRequest, DeleteLfTagResponse>()
                    .withOperationName("DeleteLFTag").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLfTagRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLfTagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters the resource as managed by the Data Catalog.
     * </p>
     * <p>
     * When you deregister a path, Lake Formation removes the path from the inline policy attached to your
     * service-linked role.
     * </p>
     *
     * @param deregisterResourceRequest
     * @return Result of the DeregisterResource operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.DeregisterResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeregisterResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterResourceResponse deregisterResource(DeregisterResourceRequest deregisterResourceRequest)
            throws InvalidInputException, InternalServiceException, OperationTimeoutException, EntityNotFoundException,
            AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterResource");

            return clientHandler.execute(new ClientExecutionParams<DeregisterResourceRequest, DeregisterResourceResponse>()
                    .withOperationName("DeregisterResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deregisterResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeregisterResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the current data access role for the given resource registered in AWS Lake Formation.
     * </p>
     *
     * @param describeResourceRequest
     * @return Result of the DescribeResource operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.DescribeResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DescribeResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeResourceResponse describeResource(DescribeResourceRequest describeResourceRequest)
            throws InvalidInputException, InternalServiceException, OperationTimeoutException, EntityNotFoundException,
            AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResource");

            return clientHandler.execute(new ClientExecutionParams<DescribeResourceRequest, DescribeResourceResponse>()
                    .withOperationName("DescribeResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the list of the data lake administrators of a Lake Formation-managed data lake.
     * </p>
     *
     * @param getDataLakeSettingsRequest
     * @return Result of the GetDataLakeSettings operation returned by the service.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.GetDataLakeSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetDataLakeSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataLakeSettingsResponse getDataLakeSettings(GetDataLakeSettingsRequest getDataLakeSettingsRequest)
            throws InternalServiceException, InvalidInputException, EntityNotFoundException, AwsServiceException,
            SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataLakeSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataLakeSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataLakeSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeSettings");

            return clientHandler.execute(new ClientExecutionParams<GetDataLakeSettingsRequest, GetDataLakeSettingsResponse>()
                    .withOperationName("GetDataLakeSettings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDataLakeSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDataLakeSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the Lake Formation permissions for a specified table or database resource located at a path in Amazon S3.
     * <code>GetEffectivePermissionsForPath</code> will not return databases and tables if the catalog is encrypted.
     * </p>
     *
     * @param getEffectivePermissionsForPathRequest
     * @return Result of the GetEffectivePermissionsForPath operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.GetEffectivePermissionsForPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetEffectivePermissionsForPath"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEffectivePermissionsForPathResponse getEffectivePermissionsForPath(
            GetEffectivePermissionsForPathRequest getEffectivePermissionsForPathRequest) throws InvalidInputException,
            EntityNotFoundException, OperationTimeoutException, InternalServiceException, AwsServiceException,
            SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEffectivePermissionsForPathResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEffectivePermissionsForPathResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEffectivePermissionsForPathRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEffectivePermissionsForPath");

            return clientHandler
                    .execute(new ClientExecutionParams<GetEffectivePermissionsForPathRequest, GetEffectivePermissionsForPathResponse>()
                            .withOperationName("GetEffectivePermissionsForPath").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getEffectivePermissionsForPathRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetEffectivePermissionsForPathRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the Lake Formation permissions for a specified table or database resource located at a path in Amazon S3.
     * <code>GetEffectivePermissionsForPath</code> will not return databases and tables if the catalog is encrypted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getEffectivePermissionsForPath(software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathIterable responses = client.getEffectivePermissionsForPathPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathIterable responses = client
     *             .getEffectivePermissionsForPathPaginator(request);
     *     for (software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.GetEffectivePermissionsForPathIterable responses = client.getEffectivePermissionsForPathPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getEffectivePermissionsForPath(software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest)}
     * operation.</b>
     * </p>
     *
     * @param getEffectivePermissionsForPathRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.GetEffectivePermissionsForPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetEffectivePermissionsForPath"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEffectivePermissionsForPathIterable getEffectivePermissionsForPathPaginator(
            GetEffectivePermissionsForPathRequest getEffectivePermissionsForPathRequest) throws InvalidInputException,
            EntityNotFoundException, OperationTimeoutException, InternalServiceException, AwsServiceException,
            SdkClientException, LakeFormationException {
        return new GetEffectivePermissionsForPathIterable(this, applyPaginatorUserAgent(getEffectivePermissionsForPathRequest));
    }

    /**
     * <p>
     * Returns a tag definition.
     * </p>
     *
     * @param getLfTagRequest
     * @return Result of the GetLFTag operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.GetLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLfTagResponse getLFTag(GetLfTagRequest getLfTagRequest) throws EntityNotFoundException, InvalidInputException,
            InternalServiceException, OperationTimeoutException, AccessDeniedException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLfTagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLFTag");

            return clientHandler.execute(new ClientExecutionParams<GetLfTagRequest, GetLfTagResponse>()
                    .withOperationName("GetLFTag").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLfTagRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetLfTagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the tags applied to a resource.
     * </p>
     *
     * @param getResourceLfTagsRequest
     * @return Result of the GetResourceLFTags operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws GlueEncryptionException
     *         An encryption operation failed.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.GetResourceLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetResourceLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceLfTagsResponse getResourceLFTags(GetResourceLfTagsRequest getResourceLfTagsRequest)
            throws EntityNotFoundException, InvalidInputException, InternalServiceException, OperationTimeoutException,
            GlueEncryptionException, AccessDeniedException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceLfTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceLFTags");

            return clientHandler.execute(new ClientExecutionParams<GetResourceLfTagsRequest, GetResourceLfTagsResponse>()
                    .withOperationName("GetResourceLFTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceLfTagsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceLfTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permissions to the principal to access metadata in the Data Catalog and data organized in underlying data
     * storage such as Amazon S3.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     *
     * @param grantPermissionsRequest
     * @return Result of the GrantPermissions operation returned by the service.
     * @throws ConcurrentModificationException
     *         Two processes are trying to modify a resource simultaneously.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.GrantPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GrantPermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GrantPermissionsResponse grantPermissions(GrantPermissionsRequest grantPermissionsRequest)
            throws ConcurrentModificationException, EntityNotFoundException, InvalidInputException, AwsServiceException,
            SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GrantPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GrantPermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, grantPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GrantPermissions");

            return clientHandler.execute(new ClientExecutionParams<GrantPermissionsRequest, GrantPermissionsResponse>()
                    .withOperationName("GrantPermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(grantPermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GrantPermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags that the requester has permission to view.
     * </p>
     *
     * @param listLfTagsRequest
     * @return Result of the ListLFTags operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.ListLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListLFTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLfTagsResponse listLFTags(ListLfTagsRequest listLfTagsRequest) throws EntityNotFoundException,
            InvalidInputException, InternalServiceException, OperationTimeoutException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLfTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLFTags");

            return clientHandler
                    .execute(new ClientExecutionParams<ListLfTagsRequest, ListLfTagsResponse>().withOperationName("ListLFTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listLfTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListLfTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the principal permissions on the resource, filtered by the permissions of the caller. For
     * example, if you are granted an ALTER permission, you are able to see only the principal permissions for ALTER.
     * </p>
     * <p>
     * This operation returns only those permissions that have been explicitly granted.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     *
     * @param listPermissionsRequest
     * @return Result of the ListPermissions operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPermissionsResponse listPermissions(ListPermissionsRequest listPermissionsRequest) throws InvalidInputException,
            OperationTimeoutException, InternalServiceException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");

            return clientHandler.execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                    .withOperationName("ListPermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the principal permissions on the resource, filtered by the permissions of the caller. For
     * example, if you are granted an ALTER permission, you are able to see only the principal permissions for ALTER.
     * </p>
     * <p>
     * This operation returns only those permissions that have been explicitly granted.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPermissions(software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsIterable responses = client.listPermissionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsIterable responses = client
     *             .listPermissionsPaginator(request);
     *     for (software.amazon.awssdk.services.lakeformation.model.ListPermissionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListPermissionsIterable responses = client.listPermissionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPermissions(software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPermissionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPermissionsIterable listPermissionsPaginator(ListPermissionsRequest listPermissionsRequest)
            throws InvalidInputException, OperationTimeoutException, InternalServiceException, AwsServiceException,
            SdkClientException, LakeFormationException {
        return new ListPermissionsIterable(this, applyPaginatorUserAgent(listPermissionsRequest));
    }

    /**
     * <p>
     * Lists the resources registered to be managed by the Data Catalog.
     * </p>
     *
     * @param listResourcesRequest
     * @return Result of the ListResources operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListResourcesResponse listResources(ListResourcesRequest listResourcesRequest) throws InvalidInputException,
            InternalServiceException, OperationTimeoutException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");

            return clientHandler.execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                    .withOperationName("ListResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the resources registered to be managed by the Data Catalog.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResources(software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.lakeformation.paginators.ListResourcesIterable responses = client
     *             .listResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.lakeformation.model.ListResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lakeformation.paginators.ListResourcesIterable responses = client.listResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListResourcesIterable listResourcesPaginator(ListResourcesRequest listResourcesRequest) throws InvalidInputException,
            InternalServiceException, OperationTimeoutException, AwsServiceException, SdkClientException, LakeFormationException {
        return new ListResourcesIterable(this, applyPaginatorUserAgent(listResourcesRequest));
    }

    /**
     * <p>
     * Sets the list of data lake administrators who have admin privileges on all resources managed by Lake Formation.
     * For more information on admin privileges, see <a
     * href="https://docs.aws.amazon.com/lake-formation/latest/dg/lake-formation-permissions.html">Granting Lake
     * Formation Permissions</a>.
     * </p>
     * <p>
     * This API replaces the current list of data lake admins with the new list being passed. To add an admin, fetch the
     * current list and add the new admin to that list and pass that list in this API.
     * </p>
     *
     * @param putDataLakeSettingsRequest
     * @return Result of the PutDataLakeSettings operation returned by the service.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.PutDataLakeSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/PutDataLakeSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutDataLakeSettingsResponse putDataLakeSettings(PutDataLakeSettingsRequest putDataLakeSettingsRequest)
            throws InternalServiceException, InvalidInputException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutDataLakeSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutDataLakeSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDataLakeSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDataLakeSettings");

            return clientHandler.execute(new ClientExecutionParams<PutDataLakeSettingsRequest, PutDataLakeSettingsResponse>()
                    .withOperationName("PutDataLakeSettings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putDataLakeSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutDataLakeSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers the resource as managed by the Data Catalog.
     * </p>
     * <p>
     * To add or update data, Lake Formation needs read/write access to the chosen Amazon S3 path. Choose a role that
     * you know has permission to do this, or choose the AWSServiceRoleForLakeFormationDataAccess service-linked role.
     * When you register the first Amazon S3 path, the service-linked role and a new inline policy are created on your
     * behalf. Lake Formation adds the first path to the inline policy and attaches it to the service-linked role. When
     * you register subsequent paths, Lake Formation adds the path to the existing policy.
     * </p>
     * <p>
     * The following request registers a new location and gives AWS Lake Formation permission to use the service-linked
     * role to access that location.
     * </p>
     * <p>
     * <code>ResourceArn = arn:aws:s3:::my-bucket UseServiceLinkedRole = true</code>
     * </p>
     * <p>
     * If <code>UseServiceLinkedRole</code> is not set to true, you must provide or set the <code>RoleArn</code>:
     * </p>
     * <p>
     * <code>arn:aws:iam::12345:role/my-data-access-role</code>
     * </p>
     *
     * @param registerResourceRequest
     * @return Result of the RegisterResource operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws AlreadyExistsException
     *         A resource to be created or added already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.RegisterResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RegisterResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterResourceResponse registerResource(RegisterResourceRequest registerResourceRequest)
            throws InvalidInputException, InternalServiceException, OperationTimeoutException, AlreadyExistsException,
            AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterResource");

            return clientHandler.execute(new ClientExecutionParams<RegisterResourceRequest, RegisterResourceResponse>()
                    .withOperationName("RegisterResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from the resource. Only database, table, or tableWithColumns resource are allowed. To tag columns,
     * use the column inclusion list in <code>tableWithColumns</code> to specify column input.
     * </p>
     *
     * @param removeLfTagsFromResourceRequest
     * @return Result of the RemoveLFTagsFromResource operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws GlueEncryptionException
     *         An encryption operation failed.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws ConcurrentModificationException
     *         Two processes are trying to modify a resource simultaneously.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.RemoveLFTagsFromResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RemoveLFTagsFromResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveLfTagsFromResourceResponse removeLFTagsFromResource(
            RemoveLfTagsFromResourceRequest removeLfTagsFromResourceRequest) throws EntityNotFoundException,
            InvalidInputException, InternalServiceException, OperationTimeoutException, GlueEncryptionException,
            AccessDeniedException, ConcurrentModificationException, AwsServiceException, SdkClientException,
            LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveLfTagsFromResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveLfTagsFromResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeLfTagsFromResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveLFTagsFromResource");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveLfTagsFromResourceRequest, RemoveLfTagsFromResourceResponse>()
                            .withOperationName("RemoveLFTagsFromResource").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeLfTagsFromResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveLfTagsFromResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Revokes permissions to the principal to access metadata in the Data Catalog and data organized in underlying data
     * storage such as Amazon S3.
     * </p>
     *
     * @param revokePermissionsRequest
     * @return Result of the RevokePermissions operation returned by the service.
     * @throws ConcurrentModificationException
     *         Two processes are trying to modify a resource simultaneously.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.RevokePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RevokePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RevokePermissionsResponse revokePermissions(RevokePermissionsRequest revokePermissionsRequest)
            throws ConcurrentModificationException, EntityNotFoundException, InvalidInputException, AwsServiceException,
            SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RevokePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RevokePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokePermissions");

            return clientHandler.execute(new ClientExecutionParams<RevokePermissionsRequest, RevokePermissionsResponse>()
                    .withOperationName("RevokePermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(revokePermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RevokePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation allows a search on <code>DATABASE</code> resources by <code>TagCondition</code>. This operation is
     * used by admins who want to grant user permissions on certain <code>TagConditions</code>. Before making a grant,
     * the admin can use <code>SearchDatabasesByTags</code> to find all resources where the given
     * <code>TagConditions</code> are valid to verify whether the returned resources can be shared.
     * </p>
     *
     * @param searchDatabasesByLfTagsRequest
     * @return Result of the SearchDatabasesByLFTags operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws GlueEncryptionException
     *         An encryption operation failed.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.SearchDatabasesByLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/SearchDatabasesByLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchDatabasesByLfTagsResponse searchDatabasesByLFTags(SearchDatabasesByLfTagsRequest searchDatabasesByLfTagsRequest)
            throws EntityNotFoundException, InternalServiceException, InvalidInputException, OperationTimeoutException,
            GlueEncryptionException, AccessDeniedException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchDatabasesByLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchDatabasesByLfTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchDatabasesByLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchDatabasesByLFTags");

            return clientHandler
                    .execute(new ClientExecutionParams<SearchDatabasesByLfTagsRequest, SearchDatabasesByLfTagsResponse>()
                            .withOperationName("SearchDatabasesByLFTags").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(searchDatabasesByLfTagsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SearchDatabasesByLfTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation allows a search on <code>TABLE</code> resources by <code>LFTag</code>s. This will be used by
     * admins who want to grant user permissions on certain LFTags. Before making a grant, the admin can use
     * <code>SearchTablesByLFTags</code> to find all resources where the given <code>LFTag</code>s are valid to verify
     * whether the returned resources can be shared.
     * </p>
     *
     * @param searchTablesByLfTagsRequest
     * @return Result of the SearchTablesByLFTags operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws GlueEncryptionException
     *         An encryption operation failed.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.SearchTablesByLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/SearchTablesByLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchTablesByLfTagsResponse searchTablesByLFTags(SearchTablesByLfTagsRequest searchTablesByLfTagsRequest)
            throws EntityNotFoundException, InternalServiceException, InvalidInputException, OperationTimeoutException,
            GlueEncryptionException, AccessDeniedException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchTablesByLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchTablesByLfTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchTablesByLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchTablesByLFTags");

            return clientHandler.execute(new ClientExecutionParams<SearchTablesByLfTagsRequest, SearchTablesByLfTagsResponse>()
                    .withOperationName("SearchTablesByLFTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchTablesByLfTagsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchTablesByLfTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the list of possible values for the specified tag key. If the tag does not exist, the operation throws an
     * EntityNotFoundException. The values in the delete key values will be deleted from list of possible values. If any
     * value in the delete key values is attached to a resource, then API errors out with a 400 Exception -
     * "Update not allowed". Untag the attribute before deleting the tag key's value.
     * </p>
     *
     * @param updateLfTagRequest
     * @return Result of the UpdateLFTag operation returned by the service.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws ConcurrentModificationException
     *         Two processes are trying to modify a resource simultaneously.
     * @throws AccessDeniedException
     *         Access to a resource was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.UpdateLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLfTagResponse updateLFTag(UpdateLfTagRequest updateLfTagRequest) throws EntityNotFoundException,
            InvalidInputException, InternalServiceException, OperationTimeoutException, ConcurrentModificationException,
            AccessDeniedException, AwsServiceException, SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLfTagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLFTag");

            return clientHandler.execute(new ClientExecutionParams<UpdateLfTagRequest, UpdateLfTagResponse>()
                    .withOperationName("UpdateLFTag").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLfTagRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLfTagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the data access role used for vending access to the given (registered) resource in AWS Lake Formation.
     * </p>
     *
     * @param updateResourceRequest
     * @return Result of the UpdateResource operation returned by the service.
     * @throws InvalidInputException
     *         The input provided was not valid.
     * @throws InternalServiceException
     *         An internal service error occurred.
     * @throws OperationTimeoutException
     *         The operation timed out.
     * @throws EntityNotFoundException
     *         A specified entity does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LakeFormationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LakeFormationClient.UpdateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateResourceResponse updateResource(UpdateResourceRequest updateResourceRequest) throws InvalidInputException,
            InternalServiceException, OperationTimeoutException, EntityNotFoundException, AwsServiceException,
            SdkClientException, LakeFormationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResource");

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceRequest, UpdateResourceResponse>()
                    .withOperationName("UpdateResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LakeFormationException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNumberLimitExceededException")
                                .exceptionBuilderSupplier(ResourceNumberLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityNotFoundException")
                                .exceptionBuilderSupplier(EntityNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlueEncryptionException")
                                .exceptionBuilderSupplier(GlueEncryptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationTimeoutException")
                                .exceptionBuilderSupplier(OperationTimeoutException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends LakeFormationRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
