/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lakeformation.model.AccessDeniedException;
import software.amazon.awssdk.services.lakeformation.model.AddLfTagsToResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.AddLfTagsToResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.AlreadyExistsException;
import software.amazon.awssdk.services.lakeformation.model.AssumeDecoratedRoleWithSamlRequest;
import software.amazon.awssdk.services.lakeformation.model.AssumeDecoratedRoleWithSamlResponse;
import software.amazon.awssdk.services.lakeformation.model.BatchGrantPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.BatchGrantPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.BatchRevokePermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.BatchRevokePermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.CancelTransactionRequest;
import software.amazon.awssdk.services.lakeformation.model.CancelTransactionResponse;
import software.amazon.awssdk.services.lakeformation.model.CommitTransactionRequest;
import software.amazon.awssdk.services.lakeformation.model.CommitTransactionResponse;
import software.amazon.awssdk.services.lakeformation.model.ConcurrentModificationException;
import software.amazon.awssdk.services.lakeformation.model.CreateDataCellsFilterRequest;
import software.amazon.awssdk.services.lakeformation.model.CreateDataCellsFilterResponse;
import software.amazon.awssdk.services.lakeformation.model.CreateLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.CreateLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.DeleteDataCellsFilterRequest;
import software.amazon.awssdk.services.lakeformation.model.DeleteDataCellsFilterResponse;
import software.amazon.awssdk.services.lakeformation.model.DeleteLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.DeleteLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.DeleteObjectsOnCancelRequest;
import software.amazon.awssdk.services.lakeformation.model.DeleteObjectsOnCancelResponse;
import software.amazon.awssdk.services.lakeformation.model.DeregisterResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.DeregisterResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.DescribeResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.DescribeResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.DescribeTransactionRequest;
import software.amazon.awssdk.services.lakeformation.model.DescribeTransactionResponse;
import software.amazon.awssdk.services.lakeformation.model.EntityNotFoundException;
import software.amazon.awssdk.services.lakeformation.model.ExpiredException;
import software.amazon.awssdk.services.lakeformation.model.ExtendTransactionRequest;
import software.amazon.awssdk.services.lakeformation.model.ExtendTransactionResponse;
import software.amazon.awssdk.services.lakeformation.model.GetDataCellsFilterRequest;
import software.amazon.awssdk.services.lakeformation.model.GetDataCellsFilterResponse;
import software.amazon.awssdk.services.lakeformation.model.GetDataLakeSettingsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetDataLakeSettingsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathRequest;
import software.amazon.awssdk.services.lakeformation.model.GetEffectivePermissionsForPathResponse;
import software.amazon.awssdk.services.lakeformation.model.GetLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.GetLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.GetQueryStateRequest;
import software.amazon.awssdk.services.lakeformation.model.GetQueryStateResponse;
import software.amazon.awssdk.services.lakeformation.model.GetQueryStatisticsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetQueryStatisticsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetResourceLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetResourceLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetTableObjectsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetTableObjectsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetTemporaryGluePartitionCredentialsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetTemporaryGluePartitionCredentialsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetTemporaryGlueTableCredentialsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetTemporaryGlueTableCredentialsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetWorkUnitResultsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetWorkUnitResultsResponse;
import software.amazon.awssdk.services.lakeformation.model.GetWorkUnitsRequest;
import software.amazon.awssdk.services.lakeformation.model.GetWorkUnitsResponse;
import software.amazon.awssdk.services.lakeformation.model.GlueEncryptionException;
import software.amazon.awssdk.services.lakeformation.model.GrantPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.GrantPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.InternalServiceException;
import software.amazon.awssdk.services.lakeformation.model.InvalidInputException;
import software.amazon.awssdk.services.lakeformation.model.LakeFormationException;
import software.amazon.awssdk.services.lakeformation.model.ListDataCellsFilterRequest;
import software.amazon.awssdk.services.lakeformation.model.ListDataCellsFilterResponse;
import software.amazon.awssdk.services.lakeformation.model.ListLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.ListPermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListPermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.ListResourcesRequest;
import software.amazon.awssdk.services.lakeformation.model.ListResourcesResponse;
import software.amazon.awssdk.services.lakeformation.model.ListTableStorageOptimizersRequest;
import software.amazon.awssdk.services.lakeformation.model.ListTableStorageOptimizersResponse;
import software.amazon.awssdk.services.lakeformation.model.ListTransactionsRequest;
import software.amazon.awssdk.services.lakeformation.model.ListTransactionsResponse;
import software.amazon.awssdk.services.lakeformation.model.OperationTimeoutException;
import software.amazon.awssdk.services.lakeformation.model.PermissionTypeMismatchException;
import software.amazon.awssdk.services.lakeformation.model.PutDataLakeSettingsRequest;
import software.amazon.awssdk.services.lakeformation.model.PutDataLakeSettingsResponse;
import software.amazon.awssdk.services.lakeformation.model.RegisterResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.RegisterResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.RemoveLfTagsFromResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.RemoveLfTagsFromResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.ResourceNotReadyException;
import software.amazon.awssdk.services.lakeformation.model.ResourceNumberLimitExceededException;
import software.amazon.awssdk.services.lakeformation.model.RevokePermissionsRequest;
import software.amazon.awssdk.services.lakeformation.model.RevokePermissionsResponse;
import software.amazon.awssdk.services.lakeformation.model.SearchDatabasesByLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.SearchDatabasesByLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.SearchTablesByLfTagsRequest;
import software.amazon.awssdk.services.lakeformation.model.SearchTablesByLfTagsResponse;
import software.amazon.awssdk.services.lakeformation.model.StartQueryPlanningRequest;
import software.amazon.awssdk.services.lakeformation.model.StartQueryPlanningResponse;
import software.amazon.awssdk.services.lakeformation.model.StartTransactionRequest;
import software.amazon.awssdk.services.lakeformation.model.StartTransactionResponse;
import software.amazon.awssdk.services.lakeformation.model.StatisticsNotReadyYetException;
import software.amazon.awssdk.services.lakeformation.model.ThrottledException;
import software.amazon.awssdk.services.lakeformation.model.TransactionCanceledException;
import software.amazon.awssdk.services.lakeformation.model.TransactionCommitInProgressException;
import software.amazon.awssdk.services.lakeformation.model.TransactionCommittedException;
import software.amazon.awssdk.services.lakeformation.model.UpdateDataCellsFilterRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateDataCellsFilterResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateLfTagRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateLfTagResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateResourceRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateResourceResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateTableObjectsRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateTableObjectsResponse;
import software.amazon.awssdk.services.lakeformation.model.UpdateTableStorageOptimizerRequest;
import software.amazon.awssdk.services.lakeformation.model.UpdateTableStorageOptimizerResponse;
import software.amazon.awssdk.services.lakeformation.model.WorkUnitsNotReadyYetException;
import software.amazon.awssdk.services.lakeformation.transform.AddLfTagsToResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.AssumeDecoratedRoleWithSamlRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.BatchGrantPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.BatchRevokePermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.CancelTransactionRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.CommitTransactionRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.CreateDataCellsFilterRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.CreateLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeleteDataCellsFilterRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeleteLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeleteObjectsOnCancelRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DeregisterResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DescribeResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.DescribeTransactionRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ExtendTransactionRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetDataCellsFilterRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetDataLakeSettingsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetEffectivePermissionsForPathRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetQueryStateRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetQueryStatisticsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetResourceLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetTableObjectsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetTemporaryGluePartitionCredentialsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetTemporaryGlueTableCredentialsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetWorkUnitResultsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GetWorkUnitsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.GrantPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListDataCellsFilterRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListTableStorageOptimizersRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.ListTransactionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.PutDataLakeSettingsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RegisterResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RemoveLfTagsFromResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.RevokePermissionsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.SearchDatabasesByLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.SearchTablesByLfTagsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.StartQueryPlanningRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.StartTransactionRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateDataCellsFilterRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateLfTagRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateResourceRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateTableObjectsRequestMarshaller;
import software.amazon.awssdk.services.lakeformation.transform.UpdateTableStorageOptimizerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link LakeFormationAsyncClient}.
 *
 * @see LakeFormationAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLakeFormationAsyncClient implements LakeFormationAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLakeFormationAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LakeFormationServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLakeFormationAsyncClient(LakeFormationServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Attaches one or more LF-tags to an existing resource.
     * </p>
     *
     * @param addLfTagsToResourceRequest
     * @return A Java Future containing the result of the AddLFTagsToResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.AddLFTagsToResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/AddLFTagsToResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddLfTagsToResourceResponse> addLFTagsToResource(
            AddLfTagsToResourceRequest addLfTagsToResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addLfTagsToResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddLFTagsToResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddLfTagsToResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddLfTagsToResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddLfTagsToResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddLfTagsToResourceRequest, AddLfTagsToResourceResponse>()
                            .withOperationName("AddLFTagsToResource")
                            .withMarshaller(new AddLfTagsToResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addLfTagsToResourceRequest));
            CompletableFuture<AddLfTagsToResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows a caller to assume an IAM role decorated as the SAML user specified in the SAML assertion included in the
     * request. This decoration allows Lake Formation to enforce access policies against the SAML users and groups. This
     * API operation requires SAML federation setup in the caller’s account as it can only be called with valid SAML
     * assertions. Lake Formation does not scope down the permission of the assumed role. All permissions attached to
     * the role via the SAML federation setup will be included in the role session.
     * </p>
     * <p>
     * This decorated role is expected to access data in Amazon S3 by getting temporary access from Lake Formation which
     * is authorized via the virtual API <code>GetDataAccess</code>. Therefore, all SAML roles that can be assumed via
     * <code>AssumeDecoratedRoleWithSAML</code> must at a minimum include <code>lakeformation:GetDataAccess</code> in
     * their role policies. A typical IAM policy attached to such a role would look as follows:
     * </p>
     *
     * @param assumeDecoratedRoleWithSamlRequest
     * @return A Java Future containing the result of the AssumeDecoratedRoleWithSAML operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.AssumeDecoratedRoleWithSAML
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/AssumeDecoratedRoleWithSAML"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssumeDecoratedRoleWithSamlResponse> assumeDecoratedRoleWithSAML(
            AssumeDecoratedRoleWithSamlRequest assumeDecoratedRoleWithSamlRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, assumeDecoratedRoleWithSamlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssumeDecoratedRoleWithSAML");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssumeDecoratedRoleWithSamlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssumeDecoratedRoleWithSamlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssumeDecoratedRoleWithSamlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssumeDecoratedRoleWithSamlRequest, AssumeDecoratedRoleWithSamlResponse>()
                            .withOperationName("AssumeDecoratedRoleWithSAML")
                            .withMarshaller(new AssumeDecoratedRoleWithSamlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(assumeDecoratedRoleWithSamlRequest));
            CompletableFuture<AssumeDecoratedRoleWithSamlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Batch operation to grant permissions to the principal.
     * </p>
     *
     * @param batchGrantPermissionsRequest
     * @return A Java Future containing the result of the BatchGrantPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.BatchGrantPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/BatchGrantPermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGrantPermissionsResponse> batchGrantPermissions(
            BatchGrantPermissionsRequest batchGrantPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGrantPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGrantPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGrantPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGrantPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGrantPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGrantPermissionsRequest, BatchGrantPermissionsResponse>()
                            .withOperationName("BatchGrantPermissions")
                            .withMarshaller(new BatchGrantPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGrantPermissionsRequest));
            CompletableFuture<BatchGrantPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Batch operation to revoke permissions from the principal.
     * </p>
     *
     * @param batchRevokePermissionsRequest
     * @return A Java Future containing the result of the BatchRevokePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.BatchRevokePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/BatchRevokePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchRevokePermissionsResponse> batchRevokePermissions(
            BatchRevokePermissionsRequest batchRevokePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchRevokePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchRevokePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchRevokePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchRevokePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchRevokePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchRevokePermissionsRequest, BatchRevokePermissionsResponse>()
                            .withOperationName("BatchRevokePermissions")
                            .withMarshaller(new BatchRevokePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchRevokePermissionsRequest));
            CompletableFuture<BatchRevokePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attempts to cancel the specified transaction. Returns an exception if the transaction was previously committed.
     * </p>
     *
     * @param cancelTransactionRequest
     * @return A Java Future containing the result of the CancelTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>TransactionCommittedException Contains details about an error where the specified transaction has
     *         already been committed and cannot be used for <code>UpdateTableObjects</code>.</li>
     *         <li>TransactionCommitInProgressException Contains details about an error related to a transaction commit
     *         that was in progress.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.CancelTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/CancelTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelTransactionResponse> cancelTransaction(CancelTransactionRequest cancelTransactionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelTransaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelTransactionRequest, CancelTransactionResponse>()
                            .withOperationName("CancelTransaction")
                            .withMarshaller(new CancelTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelTransactionRequest));
            CompletableFuture<CancelTransactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attempts to commit the specified transaction. Returns an exception if the transaction was previously aborted.
     * This API action is idempotent if called multiple times for the same transaction.
     * </p>
     *
     * @param commitTransactionRequest
     * @return A Java Future containing the result of the CommitTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>TransactionCanceledException Contains details about an error related to a transaction that was
     *         cancelled.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.CommitTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/CommitTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CommitTransactionResponse> commitTransaction(CommitTransactionRequest commitTransactionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, commitTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CommitTransaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CommitTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CommitTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CommitTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CommitTransactionRequest, CommitTransactionResponse>()
                            .withOperationName("CommitTransaction")
                            .withMarshaller(new CommitTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(commitTransactionRequest));
            CompletableFuture<CommitTransactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a data cell filter to allow one to grant access to certain columns on certain rows.
     * </p>
     *
     * @param createDataCellsFilterRequest
     * @return A Java Future containing the result of the CreateDataCellsFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException A resource to be created or added already exists.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>ResourceNumberLimitExceededException A resource numerical limit was exceeded.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.CreateDataCellsFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/CreateDataCellsFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataCellsFilterResponse> createDataCellsFilter(
            CreateDataCellsFilterRequest createDataCellsFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataCellsFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataCellsFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataCellsFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataCellsFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataCellsFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataCellsFilterRequest, CreateDataCellsFilterResponse>()
                            .withOperationName("CreateDataCellsFilter")
                            .withMarshaller(new CreateDataCellsFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDataCellsFilterRequest));
            CompletableFuture<CreateDataCellsFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an LF-tag with the specified name and values.
     * </p>
     *
     * @param createLfTagRequest
     * @return A Java Future containing the result of the CreateLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>ResourceNumberLimitExceededException A resource numerical limit was exceeded.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.CreateLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/CreateLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLfTagResponse> createLFTag(CreateLfTagRequest createLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLfTagRequest, CreateLfTagResponse>()
                            .withOperationName("CreateLFTag").withMarshaller(new CreateLfTagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLfTagRequest));
            CompletableFuture<CreateLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a data cell filter.
     * </p>
     *
     * @param deleteDataCellsFilterRequest
     * @return A Java Future containing the result of the DeleteDataCellsFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DeleteDataCellsFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeleteDataCellsFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataCellsFilterResponse> deleteDataCellsFilter(
            DeleteDataCellsFilterRequest deleteDataCellsFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataCellsFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataCellsFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataCellsFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataCellsFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataCellsFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataCellsFilterRequest, DeleteDataCellsFilterResponse>()
                            .withOperationName("DeleteDataCellsFilter")
                            .withMarshaller(new DeleteDataCellsFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDataCellsFilterRequest));
            CompletableFuture<DeleteDataCellsFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified LF-tag given a key name. If the input parameter tag key was not found, then the operation
     * will throw an exception. When you delete an LF-tag, the <code>LFTagPolicy</code> attached to the LF-tag becomes
     * invalid. If the deleted LF-tag was still assigned to any resource, the tag policy attach to the deleted LF-tag
     * will no longer be applied to the resource.
     * </p>
     *
     * @param deleteLfTagRequest
     * @return A Java Future containing the result of the DeleteLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DeleteLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeleteLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLfTagResponse> deleteLFTag(DeleteLfTagRequest deleteLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLfTagRequest, DeleteLfTagResponse>()
                            .withOperationName("DeleteLFTag").withMarshaller(new DeleteLfTagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLfTagRequest));
            CompletableFuture<DeleteLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For a specific governed table, provides a list of Amazon S3 objects that will be written during the current
     * transaction and that can be automatically deleted if the transaction is canceled. Without this call, no Amazon S3
     * objects are automatically deleted when a transaction cancels.
     * </p>
     * <p>
     * The Glue ETL library function <code>write_dynamic_frame.from_catalog()</code> includes an option to automatically
     * call <code>DeleteObjectsOnCancel</code> before writes. For more information, see <a href=
     * "https://docs.aws.amazon.com/lake-formation/latest/dg/transactions-data-operations.html#rolling-back-writes"
     * >Rolling Back Amazon S3 Writes</a>.
     * </p>
     *
     * @param deleteObjectsOnCancelRequest
     * @return A Java Future containing the result of the DeleteObjectsOnCancel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>TransactionCommittedException Contains details about an error where the specified transaction has
     *         already been committed and cannot be used for <code>UpdateTableObjects</code>.</li>
     *         <li>TransactionCanceledException Contains details about an error related to a transaction that was
     *         cancelled.</li>
     *         <li>ResourceNotReadyException Contains details about an error related to a resource which is not ready
     *         for a transaction.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DeleteObjectsOnCancel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeleteObjectsOnCancel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteObjectsOnCancelResponse> deleteObjectsOnCancel(
            DeleteObjectsOnCancelRequest deleteObjectsOnCancelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteObjectsOnCancelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteObjectsOnCancel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteObjectsOnCancelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteObjectsOnCancelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteObjectsOnCancelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteObjectsOnCancelRequest, DeleteObjectsOnCancelResponse>()
                            .withOperationName("DeleteObjectsOnCancel")
                            .withMarshaller(new DeleteObjectsOnCancelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteObjectsOnCancelRequest));
            CompletableFuture<DeleteObjectsOnCancelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters the resource as managed by the Data Catalog.
     * </p>
     * <p>
     * When you deregister a path, Lake Formation removes the path from the inline policy attached to your
     * service-linked role.
     * </p>
     *
     * @param deregisterResourceRequest
     * @return A Java Future containing the result of the DeregisterResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DeregisterResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DeregisterResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterResourceResponse> deregisterResource(DeregisterResourceRequest deregisterResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterResourceRequest, DeregisterResourceResponse>()
                            .withOperationName("DeregisterResource")
                            .withMarshaller(new DeregisterResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterResourceRequest));
            CompletableFuture<DeregisterResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current data access role for the given resource registered in Lake Formation.
     * </p>
     *
     * @param describeResourceRequest
     * @return A Java Future containing the result of the DescribeResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DescribeResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DescribeResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeResourceResponse> describeResource(DescribeResourceRequest describeResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeResourceRequest, DescribeResourceResponse>()
                            .withOperationName("DescribeResource")
                            .withMarshaller(new DescribeResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeResourceRequest));
            CompletableFuture<DescribeResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details of a single transaction.
     * </p>
     *
     * @param describeTransactionRequest
     * @return A Java Future containing the result of the DescribeTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.DescribeTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/DescribeTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTransactionResponse> describeTransaction(
            DescribeTransactionRequest describeTransactionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTransaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTransactionRequest, DescribeTransactionResponse>()
                            .withOperationName("DescribeTransaction")
                            .withMarshaller(new DescribeTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTransactionRequest));
            CompletableFuture<DescribeTransactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Indicates to the service that the specified transaction is still active and should not be treated as idle and
     * aborted.
     * </p>
     * <p>
     * Write transactions that remain idle for a long period are automatically aborted unless explicitly extended.
     * </p>
     *
     * @param extendTransactionRequest
     * @return A Java Future containing the result of the ExtendTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>TransactionCommittedException Contains details about an error where the specified transaction has
     *         already been committed and cannot be used for <code>UpdateTableObjects</code>.</li>
     *         <li>TransactionCanceledException Contains details about an error related to a transaction that was
     *         cancelled.</li>
     *         <li>TransactionCommitInProgressException Contains details about an error related to a transaction commit
     *         that was in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ExtendTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ExtendTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExtendTransactionResponse> extendTransaction(ExtendTransactionRequest extendTransactionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, extendTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExtendTransaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExtendTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExtendTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExtendTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExtendTransactionRequest, ExtendTransactionResponse>()
                            .withOperationName("ExtendTransaction")
                            .withMarshaller(new ExtendTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(extendTransactionRequest));
            CompletableFuture<ExtendTransactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a data cells filter.
     * </p>
     *
     * @param getDataCellsFilterRequest
     * @return A Java Future containing the result of the GetDataCellsFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetDataCellsFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetDataCellsFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataCellsFilterResponse> getDataCellsFilter(GetDataCellsFilterRequest getDataCellsFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataCellsFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataCellsFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataCellsFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataCellsFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataCellsFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataCellsFilterRequest, GetDataCellsFilterResponse>()
                            .withOperationName("GetDataCellsFilter")
                            .withMarshaller(new GetDataCellsFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDataCellsFilterRequest));
            CompletableFuture<GetDataCellsFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of the data lake administrators of a Lake Formation-managed data lake.
     * </p>
     *
     * @param getDataLakeSettingsRequest
     * @return A Java Future containing the result of the GetDataLakeSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetDataLakeSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetDataLakeSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataLakeSettingsResponse> getDataLakeSettings(
            GetDataLakeSettingsRequest getDataLakeSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataLakeSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataLakeSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataLakeSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataLakeSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeSettingsRequest, GetDataLakeSettingsResponse>()
                            .withOperationName("GetDataLakeSettings")
                            .withMarshaller(new GetDataLakeSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDataLakeSettingsRequest));
            CompletableFuture<GetDataLakeSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the Lake Formation permissions for a specified table or database resource located at a path in Amazon S3.
     * <code>GetEffectivePermissionsForPath</code> will not return databases and tables if the catalog is encrypted.
     * </p>
     *
     * @param getEffectivePermissionsForPathRequest
     * @return A Java Future containing the result of the GetEffectivePermissionsForPath operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetEffectivePermissionsForPath
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetEffectivePermissionsForPath"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEffectivePermissionsForPathResponse> getEffectivePermissionsForPath(
            GetEffectivePermissionsForPathRequest getEffectivePermissionsForPathRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEffectivePermissionsForPathRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEffectivePermissionsForPath");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEffectivePermissionsForPathResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEffectivePermissionsForPathResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEffectivePermissionsForPathResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEffectivePermissionsForPathRequest, GetEffectivePermissionsForPathResponse>()
                            .withOperationName("GetEffectivePermissionsForPath")
                            .withMarshaller(new GetEffectivePermissionsForPathRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEffectivePermissionsForPathRequest));
            CompletableFuture<GetEffectivePermissionsForPathResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an LF-tag definition.
     * </p>
     *
     * @param getLfTagRequest
     * @return A Java Future containing the result of the GetLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLfTagResponse> getLFTag(GetLfTagRequest getLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLfTagRequest, GetLfTagResponse>().withOperationName("GetLFTag")
                            .withMarshaller(new GetLfTagRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLfTagRequest));
            CompletableFuture<GetLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the state of a query previously submitted. Clients are expected to poll <code>GetQueryState</code> to
     * monitor the current state of the planning before retrieving the work units. A query state is only visible to the
     * principal that made the initial call to <code>StartQueryPlanning</code>.
     * </p>
     *
     * @param getQueryStateRequest
     * @return A Java Future containing the result of the GetQueryState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetQueryState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetQueryState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryStateResponse> getQueryState(GetQueryStateRequest getQueryStateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryStateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetQueryStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "query-";
            String resolvedHostExpression = "query-";

            CompletableFuture<GetQueryStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryStateRequest, GetQueryStateResponse>()
                            .withOperationName("GetQueryState")
                            .withMarshaller(new GetQueryStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getQueryStateRequest));
            CompletableFuture<GetQueryStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves statistics on the planning and execution of a query.
     * </p>
     *
     * @param getQueryStatisticsRequest
     * @return A Java Future containing the result of the GetQueryStatistics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>StatisticsNotReadyYetException Contains details about an error related to statistics not being ready.
     *         </li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ExpiredException Contains details about an error where the query request expired.</li>
     *         <li>ThrottledException Contains details about an error where the query request was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetQueryStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetQueryStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryStatisticsResponse> getQueryStatistics(GetQueryStatisticsRequest getQueryStatisticsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatistics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQueryStatisticsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "query-";
            String resolvedHostExpression = "query-";

            CompletableFuture<GetQueryStatisticsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatisticsRequest, GetQueryStatisticsResponse>()
                            .withOperationName("GetQueryStatistics")
                            .withMarshaller(new GetQueryStatisticsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getQueryStatisticsRequest));
            CompletableFuture<GetQueryStatisticsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the LF-tags applied to a resource.
     * </p>
     *
     * @param getResourceLfTagsRequest
     * @return A Java Future containing the result of the GetResourceLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetResourceLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetResourceLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceLfTagsResponse> getResourceLFTags(GetResourceLfTagsRequest getResourceLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceLfTagsRequest, GetResourceLfTagsResponse>()
                            .withOperationName("GetResourceLFTags")
                            .withMarshaller(new GetResourceLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourceLfTagsRequest));
            CompletableFuture<GetResourceLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the set of Amazon S3 objects that make up the specified governed table. A transaction ID or timestamp can
     * be specified for time-travel queries.
     * </p>
     *
     * @param getTableObjectsRequest
     * @return A Java Future containing the result of the GetTableObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>TransactionCommittedException Contains details about an error where the specified transaction has
     *         already been committed and cannot be used for <code>UpdateTableObjects</code>.</li>
     *         <li>TransactionCanceledException Contains details about an error related to a transaction that was
     *         cancelled.</li>
     *         <li>ResourceNotReadyException Contains details about an error related to a resource which is not ready
     *         for a transaction.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetTableObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetTableObjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableObjectsResponse> getTableObjects(GetTableObjectsRequest getTableObjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableObjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableObjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableObjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableObjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableObjectsRequest, GetTableObjectsResponse>()
                            .withOperationName("GetTableObjects")
                            .withMarshaller(new GetTableObjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTableObjectsRequest));
            CompletableFuture<GetTableObjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API is identical to <code>GetTemporaryTableCredentials</code> except that this is used when the target Data
     * Catalog resource is of type Partition. Lake Formation restricts the permission of the vended credentials with the
     * same scope down policy which restricts access to a single Amazon S3 prefix.
     * </p>
     *
     * @param getTemporaryGluePartitionCredentialsRequest
     * @return A Java Future containing the result of the GetTemporaryGluePartitionCredentials operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>PermissionTypeMismatchException The engine does not support filtering data based on the enforced
     *         permissions. For example, if you call the <code>GetTemporaryGlueTableCredentials</code> operation with
     *         <code>SupportedPermissionType</code> equal to <code>ColumnPermission</code>, but cell-level permissions
     *         exist on the table, this exception is thrown.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetTemporaryGluePartitionCredentials
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetTemporaryGluePartitionCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTemporaryGluePartitionCredentialsResponse> getTemporaryGluePartitionCredentials(
            GetTemporaryGluePartitionCredentialsRequest getTemporaryGluePartitionCredentialsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTemporaryGluePartitionCredentialsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTemporaryGluePartitionCredentials");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTemporaryGluePartitionCredentialsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetTemporaryGluePartitionCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTemporaryGluePartitionCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTemporaryGluePartitionCredentialsRequest, GetTemporaryGluePartitionCredentialsResponse>()
                            .withOperationName("GetTemporaryGluePartitionCredentials")
                            .withMarshaller(new GetTemporaryGluePartitionCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTemporaryGluePartitionCredentialsRequest));
            CompletableFuture<GetTemporaryGluePartitionCredentialsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows a caller in a secure environment to assume a role with permission to access Amazon S3. In order to vend
     * such credentials, Lake Formation assumes the role associated with a registered location, for example an Amazon S3
     * bucket, with a scope down policy which restricts the access to a single prefix.
     * </p>
     *
     * @param getTemporaryGlueTableCredentialsRequest
     * @return A Java Future containing the result of the GetTemporaryGlueTableCredentials operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>PermissionTypeMismatchException The engine does not support filtering data based on the enforced
     *         permissions. For example, if you call the <code>GetTemporaryGlueTableCredentials</code> operation with
     *         <code>SupportedPermissionType</code> equal to <code>ColumnPermission</code>, but cell-level permissions
     *         exist on the table, this exception is thrown.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetTemporaryGlueTableCredentials
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetTemporaryGlueTableCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTemporaryGlueTableCredentialsResponse> getTemporaryGlueTableCredentials(
            GetTemporaryGlueTableCredentialsRequest getTemporaryGlueTableCredentialsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTemporaryGlueTableCredentialsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTemporaryGlueTableCredentials");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTemporaryGlueTableCredentialsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetTemporaryGlueTableCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTemporaryGlueTableCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTemporaryGlueTableCredentialsRequest, GetTemporaryGlueTableCredentialsResponse>()
                            .withOperationName("GetTemporaryGlueTableCredentials")
                            .withMarshaller(new GetTemporaryGlueTableCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTemporaryGlueTableCredentialsRequest));
            CompletableFuture<GetTemporaryGlueTableCredentialsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the work units resulting from the query. Work units can be executed in any order and in parallel.
     * </p>
     *
     * @param getWorkUnitResultsRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        Rows returned from the <code>GetWorkUnitResults</code> operation as a stream of Apache Arrow v1.0
     *        messages.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ExpiredException Contains details about an error where the query request expired.</li>
     *         <li>ThrottledException Contains details about an error where the query request was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetWorkUnitResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetWorkUnitResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getWorkUnitResults(GetWorkUnitResultsRequest getWorkUnitResultsRequest,
            AsyncResponseTransformer<GetWorkUnitResultsResponse, ReturnT> asyncResponseTransformer) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkUnitResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkUnitResults");
            Pair<AsyncResponseTransformer<GetWorkUnitResultsResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetWorkUnitResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkUnitResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "data-";
            String resolvedHostExpression = "data-";

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetWorkUnitResultsRequest, GetWorkUnitResultsResponse>()
                            .withOperationName("GetWorkUnitResults")
                            .withMarshaller(new GetWorkUnitResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getWorkUnitResultsRequest), asyncResponseTransformer);
            AsyncResponseTransformer<GetWorkUnitResultsResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetWorkUnitResultsResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the work units generated by the <code>StartQueryPlanning</code> operation.
     * </p>
     *
     * @param getWorkUnitsRequest
     * @return A Java Future containing the result of the GetWorkUnits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WorkUnitsNotReadyYetException Contains details about an error related to work units not being ready.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ExpiredException Contains details about an error where the query request expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GetWorkUnits
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GetWorkUnits" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkUnitsResponse> getWorkUnits(GetWorkUnitsRequest getWorkUnitsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkUnitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkUnits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkUnitsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWorkUnitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "query-";
            String resolvedHostExpression = "query-";

            CompletableFuture<GetWorkUnitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkUnitsRequest, GetWorkUnitsResponse>()
                            .withOperationName("GetWorkUnits").withMarshaller(new GetWorkUnitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getWorkUnitsRequest));
            CompletableFuture<GetWorkUnitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permissions to the principal to access metadata in the Data Catalog and data organized in underlying data
     * storage such as Amazon S3.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs.aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     *
     * @param grantPermissionsRequest
     * @return A Java Future containing the result of the GrantPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.GrantPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/GrantPermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GrantPermissionsResponse> grantPermissions(GrantPermissionsRequest grantPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, grantPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GrantPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GrantPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GrantPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GrantPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GrantPermissionsRequest, GrantPermissionsResponse>()
                            .withOperationName("GrantPermissions")
                            .withMarshaller(new GrantPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(grantPermissionsRequest));
            CompletableFuture<GrantPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the data cell filters on a table.
     * </p>
     *
     * @param listDataCellsFilterRequest
     * @return A Java Future containing the result of the ListDataCellsFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListDataCellsFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListDataCellsFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataCellsFilterResponse> listDataCellsFilter(
            ListDataCellsFilterRequest listDataCellsFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataCellsFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataCellsFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataCellsFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataCellsFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataCellsFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataCellsFilterRequest, ListDataCellsFilterResponse>()
                            .withOperationName("ListDataCellsFilter")
                            .withMarshaller(new ListDataCellsFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDataCellsFilterRequest));
            CompletableFuture<ListDataCellsFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists LF-tags that the requester has permission to view.
     * </p>
     *
     * @param listLfTagsRequest
     * @return A Java Future containing the result of the ListLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListLFTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLfTagsResponse> listLFTags(ListLfTagsRequest listLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLfTagsRequest, ListLfTagsResponse>().withOperationName("ListLFTags")
                            .withMarshaller(new ListLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLfTagsRequest));
            CompletableFuture<ListLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the principal permissions on the resource, filtered by the permissions of the caller. For
     * example, if you are granted an ALTER permission, you are able to see only the principal permissions for ALTER.
     * </p>
     * <p>
     * This operation returns only those permissions that have been explicitly granted.
     * </p>
     * <p>
     * For information about permissions, see <a
     * href="https://docs-aws.amazon.com/lake-formation/latest/dg/security-data-access.html">Security and Access Control
     * to Metadata and Data</a>.
     * </p>
     *
     * @param listPermissionsRequest
     * @return A Java Future containing the result of the ListPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionsResponse> listPermissions(ListPermissionsRequest listPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                            .withOperationName("ListPermissions")
                            .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPermissionsRequest));
            CompletableFuture<ListPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resources registered to be managed by the Data Catalog.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                            .withOperationName("ListResources")
                            .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourcesRequest));
            CompletableFuture<ListResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the configuration of all storage optimizers associated with a specified table.
     * </p>
     *
     * @param listTableStorageOptimizersRequest
     * @return A Java Future containing the result of the ListTableStorageOptimizers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListTableStorageOptimizers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListTableStorageOptimizers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTableStorageOptimizersResponse> listTableStorageOptimizers(
            ListTableStorageOptimizersRequest listTableStorageOptimizersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTableStorageOptimizersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTableStorageOptimizers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTableStorageOptimizersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTableStorageOptimizersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTableStorageOptimizersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTableStorageOptimizersRequest, ListTableStorageOptimizersResponse>()
                            .withOperationName("ListTableStorageOptimizers")
                            .withMarshaller(new ListTableStorageOptimizersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTableStorageOptimizersRequest));
            CompletableFuture<ListTableStorageOptimizersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about transactions and their status. To prevent the response from growing indefinitely, only
     * uncommitted transactions and those available for time-travel queries are returned.
     * </p>
     * <p>
     * This operation can help you identify uncommitted transactions or to get information about transactions.
     * </p>
     *
     * @param listTransactionsRequest
     * @return A Java Future containing the result of the ListTransactions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.ListTransactions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/ListTransactions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTransactionsResponse> listTransactions(ListTransactionsRequest listTransactionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransactionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransactions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTransactionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTransactionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTransactionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTransactionsRequest, ListTransactionsResponse>()
                            .withOperationName("ListTransactions")
                            .withMarshaller(new ListTransactionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTransactionsRequest));
            CompletableFuture<ListTransactionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the list of data lake administrators who have admin privileges on all resources managed by Lake Formation.
     * For more information on admin privileges, see <a
     * href="https://docs.aws.amazon.com/lake-formation/latest/dg/lake-formation-permissions.html">Granting Lake
     * Formation Permissions</a>.
     * </p>
     * <p>
     * This API replaces the current list of data lake admins with the new list being passed. To add an admin, fetch the
     * current list and add the new admin to that list and pass that list in this API.
     * </p>
     *
     * @param putDataLakeSettingsRequest
     * @return A Java Future containing the result of the PutDataLakeSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.PutDataLakeSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/PutDataLakeSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDataLakeSettingsResponse> putDataLakeSettings(
            PutDataLakeSettingsRequest putDataLakeSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDataLakeSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDataLakeSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDataLakeSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDataLakeSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDataLakeSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDataLakeSettingsRequest, PutDataLakeSettingsResponse>()
                            .withOperationName("PutDataLakeSettings")
                            .withMarshaller(new PutDataLakeSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putDataLakeSettingsRequest));
            CompletableFuture<PutDataLakeSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers the resource as managed by the Data Catalog.
     * </p>
     * <p>
     * To add or update data, Lake Formation needs read/write access to the chosen Amazon S3 path. Choose a role that
     * you know has permission to do this, or choose the AWSServiceRoleForLakeFormationDataAccess service-linked role.
     * When you register the first Amazon S3 path, the service-linked role and a new inline policy are created on your
     * behalf. Lake Formation adds the first path to the inline policy and attaches it to the service-linked role. When
     * you register subsequent paths, Lake Formation adds the path to the existing policy.
     * </p>
     * <p>
     * The following request registers a new location and gives Lake Formation permission to use the service-linked role
     * to access that location.
     * </p>
     * <p>
     * <code>ResourceArn = arn:aws:s3:::my-bucket UseServiceLinkedRole = true</code>
     * </p>
     * <p>
     * If <code>UseServiceLinkedRole</code> is not set to true, you must provide or set the <code>RoleArn</code>:
     * </p>
     * <p>
     * <code>arn:aws:iam::12345:role/my-data-access-role</code>
     * </p>
     *
     * @param registerResourceRequest
     * @return A Java Future containing the result of the RegisterResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AlreadyExistsException A resource to be created or added already exists.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>ResourceNumberLimitExceededException A resource numerical limit was exceeded.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.RegisterResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RegisterResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterResourceResponse> registerResource(RegisterResourceRequest registerResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterResourceRequest, RegisterResourceResponse>()
                            .withOperationName("RegisterResource")
                            .withMarshaller(new RegisterResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerResourceRequest));
            CompletableFuture<RegisterResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an LF-tag from the resource. Only database, table, or tableWithColumns resource are allowed. To tag
     * columns, use the column inclusion list in <code>tableWithColumns</code> to specify column input.
     * </p>
     *
     * @param removeLfTagsFromResourceRequest
     * @return A Java Future containing the result of the RemoveLFTagsFromResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.RemoveLFTagsFromResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RemoveLFTagsFromResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveLfTagsFromResourceResponse> removeLFTagsFromResource(
            RemoveLfTagsFromResourceRequest removeLfTagsFromResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeLfTagsFromResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveLFTagsFromResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveLfTagsFromResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveLfTagsFromResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveLfTagsFromResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveLfTagsFromResourceRequest, RemoveLfTagsFromResourceResponse>()
                            .withOperationName("RemoveLFTagsFromResource")
                            .withMarshaller(new RemoveLfTagsFromResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeLfTagsFromResourceRequest));
            CompletableFuture<RemoveLfTagsFromResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes permissions to the principal to access metadata in the Data Catalog and data organized in underlying data
     * storage such as Amazon S3.
     * </p>
     *
     * @param revokePermissionsRequest
     * @return A Java Future containing the result of the RevokePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.RevokePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/RevokePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokePermissionsResponse> revokePermissions(RevokePermissionsRequest revokePermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokePermissionsRequest, RevokePermissionsResponse>()
                            .withOperationName("RevokePermissions")
                            .withMarshaller(new RevokePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(revokePermissionsRequest));
            CompletableFuture<RevokePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation allows a search on <code>DATABASE</code> resources by <code>TagCondition</code>. This operation is
     * used by admins who want to grant user permissions on certain <code>TagConditions</code>. Before making a grant,
     * the admin can use <code>SearchDatabasesByTags</code> to find all resources where the given
     * <code>TagConditions</code> are valid to verify whether the returned resources can be shared.
     * </p>
     *
     * @param searchDatabasesByLfTagsRequest
     * @return A Java Future containing the result of the SearchDatabasesByLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.SearchDatabasesByLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/SearchDatabasesByLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchDatabasesByLfTagsResponse> searchDatabasesByLFTags(
            SearchDatabasesByLfTagsRequest searchDatabasesByLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchDatabasesByLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchDatabasesByLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchDatabasesByLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchDatabasesByLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchDatabasesByLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchDatabasesByLfTagsRequest, SearchDatabasesByLfTagsResponse>()
                            .withOperationName("SearchDatabasesByLFTags")
                            .withMarshaller(new SearchDatabasesByLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchDatabasesByLfTagsRequest));
            CompletableFuture<SearchDatabasesByLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation allows a search on <code>TABLE</code> resources by <code>LFTag</code>s. This will be used by
     * admins who want to grant user permissions on certain LF-tags. Before making a grant, the admin can use
     * <code>SearchTablesByLFTags</code> to find all resources where the given <code>LFTag</code>s are valid to verify
     * whether the returned resources can be shared.
     * </p>
     *
     * @param searchTablesByLfTagsRequest
     * @return A Java Future containing the result of the SearchTablesByLFTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>GlueEncryptionException An encryption operation failed.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.SearchTablesByLFTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/SearchTablesByLFTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchTablesByLfTagsResponse> searchTablesByLFTags(
            SearchTablesByLfTagsRequest searchTablesByLfTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchTablesByLfTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchTablesByLFTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchTablesByLfTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchTablesByLfTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchTablesByLfTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchTablesByLfTagsRequest, SearchTablesByLfTagsResponse>()
                            .withOperationName("SearchTablesByLFTags")
                            .withMarshaller(new SearchTablesByLfTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchTablesByLfTagsRequest));
            CompletableFuture<SearchTablesByLfTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits a request to process a query statement.
     * </p>
     * <p>
     * This operation generates work units that can be retrieved with the <code>GetWorkUnits</code> operation as soon as
     * the query state is WORKUNITS_AVAILABLE or FINISHED.
     * </p>
     *
     * @param startQueryPlanningRequest
     * @return A Java Future containing the result of the StartQueryPlanning operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>ThrottledException Contains details about an error where the query request was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.StartQueryPlanning
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/StartQueryPlanning"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryPlanningResponse> startQueryPlanning(StartQueryPlanningRequest startQueryPlanningRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startQueryPlanningRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryPlanning");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryPlanningResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartQueryPlanningResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "query-";
            String resolvedHostExpression = "query-";

            CompletableFuture<StartQueryPlanningResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryPlanningRequest, StartQueryPlanningResponse>()
                            .withOperationName("StartQueryPlanning")
                            .withMarshaller(new StartQueryPlanningRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(startQueryPlanningRequest));
            CompletableFuture<StartQueryPlanningResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new transaction and returns its transaction ID. Transaction IDs are opaque objects that you can use to
     * identify a transaction.
     * </p>
     *
     * @param startTransactionRequest
     * @return A Java Future containing the result of the StartTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.StartTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/StartTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTransactionResponse> startTransaction(StartTransactionRequest startTransactionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTransaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTransactionRequest, StartTransactionResponse>()
                            .withOperationName("StartTransaction")
                            .withMarshaller(new StartTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startTransactionRequest));
            CompletableFuture<StartTransactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a data cell filter.
     * </p>
     *
     * @param updateDataCellsFilterRequest
     * @return A Java Future containing the result of the UpdateDataCellsFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateDataCellsFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateDataCellsFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataCellsFilterResponse> updateDataCellsFilter(
            UpdateDataCellsFilterRequest updateDataCellsFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataCellsFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataCellsFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataCellsFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataCellsFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataCellsFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataCellsFilterRequest, UpdateDataCellsFilterResponse>()
                            .withOperationName("UpdateDataCellsFilter")
                            .withMarshaller(new UpdateDataCellsFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDataCellsFilterRequest));
            CompletableFuture<UpdateDataCellsFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the list of possible values for the specified LF-tag key. If the LF-tag does not exist, the operation
     * throws an EntityNotFoundException. The values in the delete key values will be deleted from list of possible
     * values. If any value in the delete key values is attached to a resource, then API errors out with a 400 Exception
     * - "Update not allowed". Untag the attribute before deleting the LF-tag key's value.
     * </p>
     *
     * @param updateLfTagRequest
     * @return A Java Future containing the result of the UpdateLFTag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateLFTag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateLFTag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLfTagResponse> updateLFTag(UpdateLfTagRequest updateLfTagRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLfTagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLFTag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLfTagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLfTagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLfTagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLfTagRequest, UpdateLfTagResponse>()
                            .withOperationName("UpdateLFTag").withMarshaller(new UpdateLfTagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLfTagRequest));
            CompletableFuture<UpdateLfTagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the data access role used for vending access to the given (registered) resource in Lake Formation.
     * </p>
     *
     * @param updateResourceRequest
     * @return A Java Future containing the result of the UpdateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceResponse> updateResource(UpdateResourceRequest updateResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceRequest, UpdateResourceResponse>()
                            .withOperationName("UpdateResource")
                            .withMarshaller(new UpdateResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResourceRequest));
            CompletableFuture<UpdateResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the manifest of Amazon S3 objects that make up the specified governed table.
     * </p>
     *
     * @param updateTableObjectsRequest
     * @return A Java Future containing the result of the UpdateTableObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>TransactionCommittedException Contains details about an error where the specified transaction has
     *         already been committed and cannot be used for <code>UpdateTableObjects</code>.</li>
     *         <li>TransactionCanceledException Contains details about an error related to a transaction that was
     *         cancelled.</li>
     *         <li>TransactionCommitInProgressException Contains details about an error related to a transaction commit
     *         that was in progress.</li>
     *         <li>ResourceNotReadyException Contains details about an error related to a resource which is not ready
     *         for a transaction.</li>
     *         <li>ConcurrentModificationException Two processes are trying to modify a resource simultaneously.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateTableObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateTableObjects"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableObjectsResponse> updateTableObjects(UpdateTableObjectsRequest updateTableObjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTableObjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableObjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTableObjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTableObjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableObjectsRequest, UpdateTableObjectsResponse>()
                            .withOperationName("UpdateTableObjects")
                            .withMarshaller(new UpdateTableObjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTableObjectsRequest));
            CompletableFuture<UpdateTableObjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of the storage optimizers for a table.
     * </p>
     *
     * @param updateTableStorageOptimizerRequest
     * @return A Java Future containing the result of the UpdateTableStorageOptimizer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityNotFoundException A specified entity does not exist.</li>
     *         <li>InvalidInputException The input provided was not valid.</li>
     *         <li>AccessDeniedException Access to a resource was denied.</li>
     *         <li>InternalServiceException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LakeFormationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LakeFormationAsyncClient.UpdateTableStorageOptimizer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/UpdateTableStorageOptimizer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableStorageOptimizerResponse> updateTableStorageOptimizer(
            UpdateTableStorageOptimizerRequest updateTableStorageOptimizerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableStorageOptimizerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LakeFormation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTableStorageOptimizer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableStorageOptimizerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTableStorageOptimizerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTableStorageOptimizerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableStorageOptimizerRequest, UpdateTableStorageOptimizerResponse>()
                            .withOperationName("UpdateTableStorageOptimizer")
                            .withMarshaller(new UpdateTableStorageOptimizerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTableStorageOptimizerRequest));
            CompletableFuture<UpdateTableStorageOptimizerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LakeFormationServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LakeFormationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionCommitInProgressException")
                                .exceptionBuilderSupplier(TransactionCommitInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotReadyException")
                                .exceptionBuilderSupplier(ResourceNotReadyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNumberLimitExceededException")
                                .exceptionBuilderSupplier(ResourceNumberLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledException")
                                .exceptionBuilderSupplier(ThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredException")
                                .exceptionBuilderSupplier(ExpiredException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityNotFoundException")
                                .exceptionBuilderSupplier(EntityNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StatisticsNotReadyYetException")
                                .exceptionBuilderSupplier(StatisticsNotReadyYetException::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionCanceledException")
                                .exceptionBuilderSupplier(TransactionCanceledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlueEncryptionException")
                                .exceptionBuilderSupplier(GlueEncryptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionTypeMismatchException")
                                .exceptionBuilderSupplier(PermissionTypeMismatchException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WorkUnitsNotReadyYetException")
                                .exceptionBuilderSupplier(WorkUnitsNotReadyYetException::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionCommittedException")
                                .exceptionBuilderSupplier(TransactionCommittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationTimeoutException")
                                .exceptionBuilderSupplier(OperationTimeoutException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
