/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object to delete from the governed table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeleteObjectInput implements SdkPojo, Serializable,
        ToCopyableBuilder<DeleteObjectInput.Builder, DeleteObjectInput> {
    private static final SdkField<String> URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Uri")
            .getter(getter(DeleteObjectInput::uri)).setter(setter(Builder::uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Uri").build()).build();

    private static final SdkField<String> E_TAG_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ETag")
            .getter(getter(DeleteObjectInput::eTag)).setter(setter(Builder::eTag))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ETag").build()).build();

    private static final SdkField<List<String>> PARTITION_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PartitionValues")
            .getter(getter(DeleteObjectInput::partitionValues))
            .setter(setter(Builder::partitionValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URI_FIELD, E_TAG_FIELD,
            PARTITION_VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String uri;

    private final String eTag;

    private final List<String> partitionValues;

    private DeleteObjectInput(BuilderImpl builder) {
        this.uri = builder.uri;
        this.eTag = builder.eTag;
        this.partitionValues = builder.partitionValues;
    }

    /**
     * <p>
     * The Amazon S3 location of the object to delete.
     * </p>
     * 
     * @return The Amazon S3 location of the object to delete.
     */
    public final String uri() {
        return uri;
    }

    /**
     * <p>
     * The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to identify
     * changes to the underlying data.
     * </p>
     * 
     * @return The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to
     *         identify changes to the underlying data.
     */
    public final String eTag() {
        return eTag;
    }

    /**
     * For responses, this returns true if the service returned a value for the PartitionValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPartitionValues() {
        return partitionValues != null && !(partitionValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of partition values for the object. A value must be specified for each partition key associated with the
     * governed table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitionValues} method.
     * </p>
     * 
     * @return A list of partition values for the object. A value must be specified for each partition key associated
     *         with the governed table.
     */
    public final List<String> partitionValues() {
        return partitionValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(uri());
        hashCode = 31 * hashCode + Objects.hashCode(eTag());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionValues() ? partitionValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeleteObjectInput)) {
            return false;
        }
        DeleteObjectInput other = (DeleteObjectInput) obj;
        return Objects.equals(uri(), other.uri()) && Objects.equals(eTag(), other.eTag())
                && hasPartitionValues() == other.hasPartitionValues()
                && Objects.equals(partitionValues(), other.partitionValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeleteObjectInput").add("Uri", uri()).add("ETag", eTag())
                .add("PartitionValues", hasPartitionValues() ? partitionValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Uri":
            return Optional.ofNullable(clazz.cast(uri()));
        case "ETag":
            return Optional.ofNullable(clazz.cast(eTag()));
        case "PartitionValues":
            return Optional.ofNullable(clazz.cast(partitionValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Uri", URI_FIELD);
        map.put("ETag", E_TAG_FIELD);
        map.put("PartitionValues", PARTITION_VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeleteObjectInput, T> g) {
        return obj -> g.apply((DeleteObjectInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeleteObjectInput> {
        /**
         * <p>
         * The Amazon S3 location of the object to delete.
         * </p>
         * 
         * @param uri
         *        The Amazon S3 location of the object to delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uri(String uri);

        /**
         * <p>
         * The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to
         * identify changes to the underlying data.
         * </p>
         * 
         * @param eTag
         *        The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to
         *        identify changes to the underlying data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eTag(String eTag);

        /**
         * <p>
         * A list of partition values for the object. A value must be specified for each partition key associated with
         * the governed table.
         * </p>
         * 
         * @param partitionValues
         *        A list of partition values for the object. A value must be specified for each partition key associated
         *        with the governed table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionValues(Collection<String> partitionValues);

        /**
         * <p>
         * A list of partition values for the object. A value must be specified for each partition key associated with
         * the governed table.
         * </p>
         * 
         * @param partitionValues
         *        A list of partition values for the object. A value must be specified for each partition key associated
         *        with the governed table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionValues(String... partitionValues);
    }

    static final class BuilderImpl implements Builder {
        private String uri;

        private String eTag;

        private List<String> partitionValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DeleteObjectInput model) {
            uri(model.uri);
            eTag(model.eTag);
            partitionValues(model.partitionValues);
        }

        public final String getUri() {
            return uri;
        }

        public final void setUri(String uri) {
            this.uri = uri;
        }

        @Override
        public final Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public final String getETag() {
            return eTag;
        }

        public final void setETag(String eTag) {
            this.eTag = eTag;
        }

        @Override
        public final Builder eTag(String eTag) {
            this.eTag = eTag;
            return this;
        }

        public final Collection<String> getPartitionValues() {
            if (partitionValues instanceof SdkAutoConstructList) {
                return null;
            }
            return partitionValues;
        }

        public final void setPartitionValues(Collection<String> partitionValues) {
            this.partitionValues = PartitionValuesListCopier.copy(partitionValues);
        }

        @Override
        public final Builder partitionValues(Collection<String> partitionValues) {
            this.partitionValues = PartitionValuesListCopier.copy(partitionValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionValues(String... partitionValues) {
            partitionValues(Arrays.asList(partitionValues));
            return this;
        }

        @Override
        public DeleteObjectInput build() {
            return new DeleteObjectInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
