/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A new object to add to the governed table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AddObjectInput implements SdkPojo, Serializable, ToCopyableBuilder<AddObjectInput.Builder, AddObjectInput> {
    private static final SdkField<String> URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Uri")
            .getter(getter(AddObjectInput::uri)).setter(setter(Builder::uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Uri").build()).build();

    private static final SdkField<String> E_TAG_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ETag")
            .getter(getter(AddObjectInput::eTag)).setter(setter(Builder::eTag))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ETag").build()).build();

    private static final SdkField<Long> SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Size")
            .getter(getter(AddObjectInput::size)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size").build()).build();

    private static final SdkField<List<String>> PARTITION_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PartitionValues")
            .getter(getter(AddObjectInput::partitionValues))
            .setter(setter(Builder::partitionValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URI_FIELD, E_TAG_FIELD,
            SIZE_FIELD, PARTITION_VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String uri;

    private final String eTag;

    private final Long size;

    private final List<String> partitionValues;

    private AddObjectInput(BuilderImpl builder) {
        this.uri = builder.uri;
        this.eTag = builder.eTag;
        this.size = builder.size;
        this.partitionValues = builder.partitionValues;
    }

    /**
     * <p>
     * The Amazon S3 location of the object.
     * </p>
     * 
     * @return The Amazon S3 location of the object.
     */
    public final String uri() {
        return uri;
    }

    /**
     * <p>
     * The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to identify
     * changes to the underlying data.
     * </p>
     * 
     * @return The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to
     *         identify changes to the underlying data.
     */
    public final String eTag() {
        return eTag;
    }

    /**
     * <p>
     * The size of the Amazon S3 object in bytes.
     * </p>
     * 
     * @return The size of the Amazon S3 object in bytes.
     */
    public final Long size() {
        return size;
    }

    /**
     * For responses, this returns true if the service returned a value for the PartitionValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPartitionValues() {
        return partitionValues != null && !(partitionValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of partition values for the object. A value must be specified for each partition key associated with the
     * table.
     * </p>
     * <p>
     * The supported data types are integer, long, date(yyyy-MM-dd), timestamp(yyyy-MM-dd HH:mm:ssXXX or yyyy-MM-dd
     * HH:mm:ss"), string and decimal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitionValues} method.
     * </p>
     * 
     * @return A list of partition values for the object. A value must be specified for each partition key associated
     *         with the table.</p>
     *         <p>
     *         The supported data types are integer, long, date(yyyy-MM-dd), timestamp(yyyy-MM-dd HH:mm:ssXXX or
     *         yyyy-MM-dd HH:mm:ss"), string and decimal.
     */
    public final List<String> partitionValues() {
        return partitionValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(uri());
        hashCode = 31 * hashCode + Objects.hashCode(eTag());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionValues() ? partitionValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AddObjectInput)) {
            return false;
        }
        AddObjectInput other = (AddObjectInput) obj;
        return Objects.equals(uri(), other.uri()) && Objects.equals(eTag(), other.eTag()) && Objects.equals(size(), other.size())
                && hasPartitionValues() == other.hasPartitionValues()
                && Objects.equals(partitionValues(), other.partitionValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AddObjectInput").add("Uri", uri()).add("ETag", eTag()).add("Size", size())
                .add("PartitionValues", hasPartitionValues() ? partitionValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Uri":
            return Optional.ofNullable(clazz.cast(uri()));
        case "ETag":
            return Optional.ofNullable(clazz.cast(eTag()));
        case "Size":
            return Optional.ofNullable(clazz.cast(size()));
        case "PartitionValues":
            return Optional.ofNullable(clazz.cast(partitionValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Uri", URI_FIELD);
        map.put("ETag", E_TAG_FIELD);
        map.put("Size", SIZE_FIELD);
        map.put("PartitionValues", PARTITION_VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AddObjectInput, T> g) {
        return obj -> g.apply((AddObjectInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AddObjectInput> {
        /**
         * <p>
         * The Amazon S3 location of the object.
         * </p>
         * 
         * @param uri
         *        The Amazon S3 location of the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uri(String uri);

        /**
         * <p>
         * The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to
         * identify changes to the underlying data.
         * </p>
         * 
         * @param eTag
         *        The Amazon S3 ETag of the object. Returned by <code>GetTableObjects</code> for validation and used to
         *        identify changes to the underlying data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eTag(String eTag);

        /**
         * <p>
         * The size of the Amazon S3 object in bytes.
         * </p>
         * 
         * @param size
         *        The size of the Amazon S3 object in bytes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Long size);

        /**
         * <p>
         * A list of partition values for the object. A value must be specified for each partition key associated with
         * the table.
         * </p>
         * <p>
         * The supported data types are integer, long, date(yyyy-MM-dd), timestamp(yyyy-MM-dd HH:mm:ssXXX or yyyy-MM-dd
         * HH:mm:ss"), string and decimal.
         * </p>
         * 
         * @param partitionValues
         *        A list of partition values for the object. A value must be specified for each partition key associated
         *        with the table.</p>
         *        <p>
         *        The supported data types are integer, long, date(yyyy-MM-dd), timestamp(yyyy-MM-dd HH:mm:ssXXX or
         *        yyyy-MM-dd HH:mm:ss"), string and decimal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionValues(Collection<String> partitionValues);

        /**
         * <p>
         * A list of partition values for the object. A value must be specified for each partition key associated with
         * the table.
         * </p>
         * <p>
         * The supported data types are integer, long, date(yyyy-MM-dd), timestamp(yyyy-MM-dd HH:mm:ssXXX or yyyy-MM-dd
         * HH:mm:ss"), string and decimal.
         * </p>
         * 
         * @param partitionValues
         *        A list of partition values for the object. A value must be specified for each partition key associated
         *        with the table.</p>
         *        <p>
         *        The supported data types are integer, long, date(yyyy-MM-dd), timestamp(yyyy-MM-dd HH:mm:ssXXX or
         *        yyyy-MM-dd HH:mm:ss"), string and decimal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionValues(String... partitionValues);
    }

    static final class BuilderImpl implements Builder {
        private String uri;

        private String eTag;

        private Long size;

        private List<String> partitionValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AddObjectInput model) {
            uri(model.uri);
            eTag(model.eTag);
            size(model.size);
            partitionValues(model.partitionValues);
        }

        public final String getUri() {
            return uri;
        }

        public final void setUri(String uri) {
            this.uri = uri;
        }

        @Override
        public final Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public final String getETag() {
            return eTag;
        }

        public final void setETag(String eTag) {
            this.eTag = eTag;
        }

        @Override
        public final Builder eTag(String eTag) {
            this.eTag = eTag;
            return this;
        }

        public final Long getSize() {
            return size;
        }

        public final void setSize(Long size) {
            this.size = size;
        }

        @Override
        public final Builder size(Long size) {
            this.size = size;
            return this;
        }

        public final Collection<String> getPartitionValues() {
            if (partitionValues instanceof SdkAutoConstructList) {
                return null;
            }
            return partitionValues;
        }

        public final void setPartitionValues(Collection<String> partitionValues) {
            this.partitionValues = PartitionValuesListCopier.copy(partitionValues);
        }

        @Override
        public final Builder partitionValues(Collection<String> partitionValues) {
            this.partitionValues = PartitionValuesListCopier.copy(partitionValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionValues(String... partitionValues) {
            partitionValues(Arrays.asList(partitionValues));
            return this;
        }

        @Override
        public AddObjectInput build() {
            return new AddObjectInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
