/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.lambda.LambdaAsyncClient;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationResponse;
import software.amazon.awssdk.services.lambda.model.GetFunctionRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface LambdaAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getFunctionConfigurationRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetFunctionConfigurationRequest#builder()}
     *
     * @param getFunctionConfigurationRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionActive(
            Consumer<GetFunctionConfigurationRequest.Builder> getFunctionConfigurationRequest) {
        return waitUntilFunctionActive(GetFunctionConfigurationRequest.builder().applyMutation(getFunctionConfigurationRequest)
                .build());
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getFunctionConfigurationRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getFunctionConfigurationRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionActive(
            Consumer<GetFunctionConfigurationRequest.Builder> getFunctionConfigurationRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFunctionActive(GetFunctionConfigurationRequest.builder().applyMutation(getFunctionConfigurationRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunction} API until the desired condition {@code FunctionExists} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getFunctionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionExists(GetFunctionRequest getFunctionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunction} API until the desired condition {@code FunctionExists} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetFunctionRequest#builder()}
     *
     * @param getFunctionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionExists(
            Consumer<GetFunctionRequest.Builder> getFunctionRequest) {
        return waitUntilFunctionExists(GetFunctionRequest.builder().applyMutation(getFunctionRequest).build());
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunction} API until the desired condition {@code FunctionExists} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getFunctionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionExists(GetFunctionRequest getFunctionRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunction} API until the desired condition {@code FunctionExists} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getFunctionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionExists(
            Consumer<GetFunctionRequest.Builder> getFunctionRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFunctionExists(GetFunctionRequest.builder().applyMutation(getFunctionRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionUpdated}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getFunctionConfigurationRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionUpdated(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionUpdated}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetFunctionConfigurationRequest#builder()}
     *
     * @param getFunctionConfigurationRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionUpdated(
            Consumer<GetFunctionConfigurationRequest.Builder> getFunctionConfigurationRequest) {
        return waitUntilFunctionUpdated(GetFunctionConfigurationRequest.builder().applyMutation(getFunctionConfigurationRequest)
                .build());
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionUpdated}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getFunctionConfigurationRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionUpdated(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link LambdaAsyncClient#getFunctionConfiguration} API until the desired condition {@code FunctionUpdated}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getFunctionConfigurationRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionUpdated(
            Consumer<GetFunctionConfigurationRequest.Builder> getFunctionConfigurationRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFunctionUpdated(GetFunctionConfigurationRequest.builder().applyMutation(getFunctionConfigurationRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link LambdaAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultLambdaAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link LambdaAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link LambdaAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link LambdaAsyncWaiter}
     */
    static LambdaAsyncWaiter create() {
        return DefaultLambdaAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link LambdaAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(LambdaAsyncClient client);

        /**
         * Builds an instance of {@link LambdaAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link LambdaAsyncWaiter}
         */
        LambdaAsyncWaiter build();
    }
}
