/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A mapping between an Amazon Web Services resource and a Lambda function. For details, see
 * <a>CreateEventSourceMapping</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventSourceMappingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EventSourceMappingConfiguration.Builder, EventSourceMappingConfiguration> {
    private static final SdkField<String> UUID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("UUID")
            .getter(getter(EventSourceMappingConfiguration::uuid)).setter(setter(Builder::uuid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UUID").build()).build();

    private static final SdkField<String> STARTING_POSITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartingPosition").getter(getter(EventSourceMappingConfiguration::startingPositionAsString))
            .setter(setter(Builder::startingPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartingPosition").build()).build();

    private static final SdkField<Instant> STARTING_POSITION_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).memberName("StartingPositionTimestamp")
            .getter(getter(EventSourceMappingConfiguration::startingPositionTimestamp))
            .setter(setter(Builder::startingPositionTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartingPositionTimestamp").build())
            .build();

    private static final SdkField<Integer> BATCH_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BatchSize").getter(getter(EventSourceMappingConfiguration::batchSize))
            .setter(setter(Builder::batchSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BatchSize").build()).build();

    private static final SdkField<Integer> MAXIMUM_BATCHING_WINDOW_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumBatchingWindowInSeconds")
            .getter(getter(EventSourceMappingConfiguration::maximumBatchingWindowInSeconds))
            .setter(setter(Builder::maximumBatchingWindowInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumBatchingWindowInSeconds")
                    .build()).build();

    private static final SdkField<Integer> PARALLELIZATION_FACTOR_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ParallelizationFactor").getter(getter(EventSourceMappingConfiguration::parallelizationFactor))
            .setter(setter(Builder::parallelizationFactor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParallelizationFactor").build())
            .build();

    private static final SdkField<String> EVENT_SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventSourceArn").getter(getter(EventSourceMappingConfiguration::eventSourceArn))
            .setter(setter(Builder::eventSourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventSourceArn").build()).build();

    private static final SdkField<FilterCriteria> FILTER_CRITERIA_FIELD = SdkField
            .<FilterCriteria> builder(MarshallingType.SDK_POJO).memberName("FilterCriteria")
            .getter(getter(EventSourceMappingConfiguration::filterCriteria)).setter(setter(Builder::filterCriteria))
            .constructor(FilterCriteria::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FilterCriteria").build()).build();

    private static final SdkField<String> FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FunctionArn").getter(getter(EventSourceMappingConfiguration::functionArn))
            .setter(setter(Builder::functionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionArn").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModified").getter(getter(EventSourceMappingConfiguration::lastModified))
            .setter(setter(Builder::lastModified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModified").build()).build();

    private static final SdkField<String> LAST_PROCESSING_RESULT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastProcessingResult").getter(getter(EventSourceMappingConfiguration::lastProcessingResult))
            .setter(setter(Builder::lastProcessingResult))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastProcessingResult").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(EventSourceMappingConfiguration::state)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATE_TRANSITION_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateTransitionReason").getter(getter(EventSourceMappingConfiguration::stateTransitionReason))
            .setter(setter(Builder::stateTransitionReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateTransitionReason").build())
            .build();

    private static final SdkField<DestinationConfig> DESTINATION_CONFIG_FIELD = SdkField
            .<DestinationConfig> builder(MarshallingType.SDK_POJO).memberName("DestinationConfig")
            .getter(getter(EventSourceMappingConfiguration::destinationConfig)).setter(setter(Builder::destinationConfig))
            .constructor(DestinationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationConfig").build()).build();

    private static final SdkField<List<String>> TOPICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Topics")
            .getter(getter(EventSourceMappingConfiguration::topics))
            .setter(setter(Builder::topics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Topics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> QUEUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Queues")
            .getter(getter(EventSourceMappingConfiguration::queues))
            .setter(setter(Builder::queues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Queues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<SourceAccessConfiguration>> SOURCE_ACCESS_CONFIGURATIONS_FIELD = SdkField
            .<List<SourceAccessConfiguration>> builder(MarshallingType.LIST)
            .memberName("SourceAccessConfigurations")
            .getter(getter(EventSourceMappingConfiguration::sourceAccessConfigurations))
            .setter(setter(Builder::sourceAccessConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceAccessConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SourceAccessConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(SourceAccessConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SelfManagedEventSource> SELF_MANAGED_EVENT_SOURCE_FIELD = SdkField
            .<SelfManagedEventSource> builder(MarshallingType.SDK_POJO).memberName("SelfManagedEventSource")
            .getter(getter(EventSourceMappingConfiguration::selfManagedEventSource))
            .setter(setter(Builder::selfManagedEventSource)).constructor(SelfManagedEventSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelfManagedEventSource").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_RECORD_AGE_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaximumRecordAgeInSeconds")
            .getter(getter(EventSourceMappingConfiguration::maximumRecordAgeInSeconds))
            .setter(setter(Builder::maximumRecordAgeInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumRecordAgeInSeconds").build())
            .build();

    private static final SdkField<Boolean> BISECT_BATCH_ON_FUNCTION_ERROR_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("BisectBatchOnFunctionError")
            .getter(getter(EventSourceMappingConfiguration::bisectBatchOnFunctionError))
            .setter(setter(Builder::bisectBatchOnFunctionError))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BisectBatchOnFunctionError").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_RETRY_ATTEMPTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumRetryAttempts").getter(getter(EventSourceMappingConfiguration::maximumRetryAttempts))
            .setter(setter(Builder::maximumRetryAttempts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumRetryAttempts").build())
            .build();

    private static final SdkField<Integer> TUMBLING_WINDOW_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TumblingWindowInSeconds").getter(getter(EventSourceMappingConfiguration::tumblingWindowInSeconds))
            .setter(setter(Builder::tumblingWindowInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TumblingWindowInSeconds").build())
            .build();

    private static final SdkField<List<String>> FUNCTION_RESPONSE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FunctionResponseTypes")
            .getter(getter(EventSourceMappingConfiguration::functionResponseTypesAsStrings))
            .setter(setter(Builder::functionResponseTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionResponseTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AmazonManagedKafkaEventSourceConfig> AMAZON_MANAGED_KAFKA_EVENT_SOURCE_CONFIG_FIELD = SdkField
            .<AmazonManagedKafkaEventSourceConfig> builder(MarshallingType.SDK_POJO)
            .memberName("AmazonManagedKafkaEventSourceConfig")
            .getter(getter(EventSourceMappingConfiguration::amazonManagedKafkaEventSourceConfig))
            .setter(setter(Builder::amazonManagedKafkaEventSourceConfig))
            .constructor(AmazonManagedKafkaEventSourceConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("AmazonManagedKafkaEventSourceConfig").build()).build();

    private static final SdkField<SelfManagedKafkaEventSourceConfig> SELF_MANAGED_KAFKA_EVENT_SOURCE_CONFIG_FIELD = SdkField
            .<SelfManagedKafkaEventSourceConfig> builder(MarshallingType.SDK_POJO)
            .memberName("SelfManagedKafkaEventSourceConfig")
            .getter(getter(EventSourceMappingConfiguration::selfManagedKafkaEventSourceConfig))
            .setter(setter(Builder::selfManagedKafkaEventSourceConfig))
            .constructor(SelfManagedKafkaEventSourceConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelfManagedKafkaEventSourceConfig")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(UUID_FIELD,
            STARTING_POSITION_FIELD, STARTING_POSITION_TIMESTAMP_FIELD, BATCH_SIZE_FIELD,
            MAXIMUM_BATCHING_WINDOW_IN_SECONDS_FIELD, PARALLELIZATION_FACTOR_FIELD, EVENT_SOURCE_ARN_FIELD,
            FILTER_CRITERIA_FIELD, FUNCTION_ARN_FIELD, LAST_MODIFIED_FIELD, LAST_PROCESSING_RESULT_FIELD, STATE_FIELD,
            STATE_TRANSITION_REASON_FIELD, DESTINATION_CONFIG_FIELD, TOPICS_FIELD, QUEUES_FIELD,
            SOURCE_ACCESS_CONFIGURATIONS_FIELD, SELF_MANAGED_EVENT_SOURCE_FIELD, MAXIMUM_RECORD_AGE_IN_SECONDS_FIELD,
            BISECT_BATCH_ON_FUNCTION_ERROR_FIELD, MAXIMUM_RETRY_ATTEMPTS_FIELD, TUMBLING_WINDOW_IN_SECONDS_FIELD,
            FUNCTION_RESPONSE_TYPES_FIELD, AMAZON_MANAGED_KAFKA_EVENT_SOURCE_CONFIG_FIELD,
            SELF_MANAGED_KAFKA_EVENT_SOURCE_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String uuid;

    private final String startingPosition;

    private final Instant startingPositionTimestamp;

    private final Integer batchSize;

    private final Integer maximumBatchingWindowInSeconds;

    private final Integer parallelizationFactor;

    private final String eventSourceArn;

    private final FilterCriteria filterCriteria;

    private final String functionArn;

    private final Instant lastModified;

    private final String lastProcessingResult;

    private final String state;

    private final String stateTransitionReason;

    private final DestinationConfig destinationConfig;

    private final List<String> topics;

    private final List<String> queues;

    private final List<SourceAccessConfiguration> sourceAccessConfigurations;

    private final SelfManagedEventSource selfManagedEventSource;

    private final Integer maximumRecordAgeInSeconds;

    private final Boolean bisectBatchOnFunctionError;

    private final Integer maximumRetryAttempts;

    private final Integer tumblingWindowInSeconds;

    private final List<String> functionResponseTypes;

    private final AmazonManagedKafkaEventSourceConfig amazonManagedKafkaEventSourceConfig;

    private final SelfManagedKafkaEventSourceConfig selfManagedKafkaEventSourceConfig;

    private EventSourceMappingConfiguration(BuilderImpl builder) {
        this.uuid = builder.uuid;
        this.startingPosition = builder.startingPosition;
        this.startingPositionTimestamp = builder.startingPositionTimestamp;
        this.batchSize = builder.batchSize;
        this.maximumBatchingWindowInSeconds = builder.maximumBatchingWindowInSeconds;
        this.parallelizationFactor = builder.parallelizationFactor;
        this.eventSourceArn = builder.eventSourceArn;
        this.filterCriteria = builder.filterCriteria;
        this.functionArn = builder.functionArn;
        this.lastModified = builder.lastModified;
        this.lastProcessingResult = builder.lastProcessingResult;
        this.state = builder.state;
        this.stateTransitionReason = builder.stateTransitionReason;
        this.destinationConfig = builder.destinationConfig;
        this.topics = builder.topics;
        this.queues = builder.queues;
        this.sourceAccessConfigurations = builder.sourceAccessConfigurations;
        this.selfManagedEventSource = builder.selfManagedEventSource;
        this.maximumRecordAgeInSeconds = builder.maximumRecordAgeInSeconds;
        this.bisectBatchOnFunctionError = builder.bisectBatchOnFunctionError;
        this.maximumRetryAttempts = builder.maximumRetryAttempts;
        this.tumblingWindowInSeconds = builder.tumblingWindowInSeconds;
        this.functionResponseTypes = builder.functionResponseTypes;
        this.amazonManagedKafkaEventSourceConfig = builder.amazonManagedKafkaEventSourceConfig;
        this.selfManagedKafkaEventSourceConfig = builder.selfManagedKafkaEventSourceConfig;
    }

    /**
     * <p>
     * The identifier of the event source mapping.
     * </p>
     * 
     * @return The identifier of the event source mapping.
     */
    public final String uuid() {
        return uuid;
    }

    /**
     * <p>
     * The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB, and Amazon
     * MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #startingPosition}
     * will return {@link EventSourcePosition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #startingPositionAsString}.
     * </p>
     * 
     * @return The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB, and
     *         Amazon MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
     * @see EventSourcePosition
     */
    public final EventSourcePosition startingPosition() {
        return EventSourcePosition.fromValue(startingPosition);
    }

    /**
     * <p>
     * The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB, and Amazon
     * MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #startingPosition}
     * will return {@link EventSourcePosition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #startingPositionAsString}.
     * </p>
     * 
     * @return The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB, and
     *         Amazon MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
     * @see EventSourcePosition
     */
    public final String startingPositionAsString() {
        return startingPosition;
    }

    /**
     * <p>
     * With <code>StartingPosition</code> set to <code>AT_TIMESTAMP</code>, the time from which to start reading.
     * </p>
     * 
     * @return With <code>StartingPosition</code> set to <code>AT_TIMESTAMP</code>, the time from which to start
     *         reading.
     */
    public final Instant startingPositionTimestamp() {
        return startingPositionTimestamp;
    }

    /**
     * <p>
     * The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your
     * function. Lambda passes all of the records in the batch to the function in a single call, up to the payload limit
     * for synchronous invocation (6 MB).
     * </p>
     * <p>
     * Default value: Varies by service. For Amazon SQS, the default is 10. For all other services, the default is 100.
     * </p>
     * <p>
     * Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
     * <code>MaximumBatchingWindowInSeconds</code> to at least 1.
     * </p>
     * 
     * @return The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your
     *         function. Lambda passes all of the records in the batch to the function in a single call, up to the
     *         payload limit for synchronous invocation (6 MB).</p>
     *         <p>
     *         Default value: Varies by service. For Amazon SQS, the default is 10. For all other services, the default
     *         is 100.
     *         </p>
     *         <p>
     *         Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
     *         <code>MaximumBatchingWindowInSeconds</code> to at least 1.
     */
    public final Integer batchSize() {
        return batchSize;
    }

    /**
     * <p>
     * (Streams and Amazon SQS standard queues) The maximum amount of time, in seconds, that Lambda spends gathering
     * records before invoking the function.
     * </p>
     * <p>
     * Default: 0
     * </p>
     * <p>
     * Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
     * <code>MaximumBatchingWindowInSeconds</code> to at least 1.
     * </p>
     * 
     * @return (Streams and Amazon SQS standard queues) The maximum amount of time, in seconds, that Lambda spends
     *         gathering records before invoking the function.</p>
     *         <p>
     *         Default: 0
     *         </p>
     *         <p>
     *         Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
     *         <code>MaximumBatchingWindowInSeconds</code> to at least 1.
     */
    public final Integer maximumBatchingWindowInSeconds() {
        return maximumBatchingWindowInSeconds;
    }

    /**
     * <p>
     * (Streams only) The number of batches to process concurrently from each shard. The default value is 1.
     * </p>
     * 
     * @return (Streams only) The number of batches to process concurrently from each shard. The default value is 1.
     */
    public final Integer parallelizationFactor() {
        return parallelizationFactor;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the event source.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the event source.
     */
    public final String eventSourceArn() {
        return eventSourceArn;
    }

    /**
     * <p>
     * (Streams and Amazon SQS) An object that defines the filter criteria that determine whether Lambda should process
     * an event. For more information, see <a
     * href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html">Lambda event filtering</a>.
     * </p>
     * 
     * @return (Streams and Amazon SQS) An object that defines the filter criteria that determine whether Lambda should
     *         process an event. For more information, see <a
     *         href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html">Lambda event
     *         filtering</a>.
     */
    public final FilterCriteria filterCriteria() {
        return filterCriteria;
    }

    /**
     * <p>
     * The ARN of the Lambda function.
     * </p>
     * 
     * @return The ARN of the Lambda function.
     */
    public final String functionArn() {
        return functionArn;
    }

    /**
     * <p>
     * The date that the event source mapping was last updated or that its state changed.
     * </p>
     * 
     * @return The date that the event source mapping was last updated or that its state changed.
     */
    public final Instant lastModified() {
        return lastModified;
    }

    /**
     * <p>
     * The result of the last Lambda invocation of your function.
     * </p>
     * 
     * @return The result of the last Lambda invocation of your function.
     */
    public final String lastProcessingResult() {
        return lastProcessingResult;
    }

    /**
     * <p>
     * The state of the event source mapping. It can be one of the following: <code>Creating</code>,
     * <code>Enabling</code>, <code>Enabled</code>, <code>Disabling</code>, <code>Disabled</code>, <code>Updating</code>
     * , or <code>Deleting</code>.
     * </p>
     * 
     * @return The state of the event source mapping. It can be one of the following: <code>Creating</code>,
     *         <code>Enabling</code>, <code>Enabled</code>, <code>Disabling</code>, <code>Disabled</code>,
     *         <code>Updating</code>, or <code>Deleting</code>.
     */
    public final String state() {
        return state;
    }

    /**
     * <p>
     * Indicates whether a user or Lambda made the last change to the event source mapping.
     * </p>
     * 
     * @return Indicates whether a user or Lambda made the last change to the event source mapping.
     */
    public final String stateTransitionReason() {
        return stateTransitionReason;
    }

    /**
     * <p>
     * (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
     * </p>
     * 
     * @return (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
     */
    public final DestinationConfig destinationConfig() {
        return destinationConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the Topics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTopics() {
        return topics != null && !(topics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the Kafka topic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTopics} method.
     * </p>
     * 
     * @return The name of the Kafka topic.
     */
    public final List<String> topics() {
        return topics;
    }

    /**
     * For responses, this returns true if the service returned a value for the Queues property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasQueues() {
        return queues != null && !(queues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQueues} method.
     * </p>
     * 
     * @return (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
     */
    public final List<String> queues() {
        return queues;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceAccessConfigurations property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSourceAccessConfigurations() {
        return sourceAccessConfigurations != null && !(sourceAccessConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceAccessConfigurations} method.
     * </p>
     * 
     * @return An array of the authentication protocol, VPC components, or virtual host to secure and define your event
     *         source.
     */
    public final List<SourceAccessConfiguration> sourceAccessConfigurations() {
        return sourceAccessConfigurations;
    }

    /**
     * <p>
     * The self-managed Apache Kafka cluster for your event source.
     * </p>
     * 
     * @return The self-managed Apache Kafka cluster for your event source.
     */
    public final SelfManagedEventSource selfManagedEventSource() {
        return selfManagedEventSource;
    }

    /**
     * <p>
     * (Streams only) Discard records older than the specified age. The default value is -1, which sets the maximum age
     * to infinite. When the value is set to infinite, Lambda never discards old records.
     * </p>
     * 
     * @return (Streams only) Discard records older than the specified age. The default value is -1, which sets the
     *         maximum age to infinite. When the value is set to infinite, Lambda never discards old records.
     */
    public final Integer maximumRecordAgeInSeconds() {
        return maximumRecordAgeInSeconds;
    }

    /**
     * <p>
     * (Streams only) If the function returns an error, split the batch in two and retry. The default value is false.
     * </p>
     * 
     * @return (Streams only) If the function returns an error, split the batch in two and retry. The default value is
     *         false.
     */
    public final Boolean bisectBatchOnFunctionError() {
        return bisectBatchOnFunctionError;
    }

    /**
     * <p>
     * (Streams only) Discard records after the specified number of retries. The default value is -1, which sets the
     * maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until
     * the record expires in the event source.
     * </p>
     * 
     * @return (Streams only) Discard records after the specified number of retries. The default value is -1, which sets
     *         the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed
     *         records until the record expires in the event source.
     */
    public final Integer maximumRetryAttempts() {
        return maximumRetryAttempts;
    }

    /**
     * <p>
     * (Streams only) The duration in seconds of a processing window. The range is 1–900 seconds.
     * </p>
     * 
     * @return (Streams only) The duration in seconds of a processing window. The range is 1–900 seconds.
     */
    public final Integer tumblingWindowInSeconds() {
        return tumblingWindowInSeconds;
    }

    /**
     * <p>
     * (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFunctionResponseTypes} method.
     * </p>
     * 
     * @return (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
     */
    public final List<FunctionResponseType> functionResponseTypes() {
        return FunctionResponseTypeListCopier.copyStringToEnum(functionResponseTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FunctionResponseTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasFunctionResponseTypes() {
        return functionResponseTypes != null && !(functionResponseTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFunctionResponseTypes} method.
     * </p>
     * 
     * @return (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
     */
    public final List<String> functionResponseTypesAsStrings() {
        return functionResponseTypes;
    }

    /**
     * <p>
     * Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
     * </p>
     * 
     * @return Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event
     *         source.
     */
    public final AmazonManagedKafkaEventSourceConfig amazonManagedKafkaEventSourceConfig() {
        return amazonManagedKafkaEventSourceConfig;
    }

    /**
     * <p>
     * Specific configuration settings for a self-managed Apache Kafka event source.
     * </p>
     * 
     * @return Specific configuration settings for a self-managed Apache Kafka event source.
     */
    public final SelfManagedKafkaEventSourceConfig selfManagedKafkaEventSourceConfig() {
        return selfManagedKafkaEventSourceConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(uuid());
        hashCode = 31 * hashCode + Objects.hashCode(startingPositionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startingPositionTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(batchSize());
        hashCode = 31 * hashCode + Objects.hashCode(maximumBatchingWindowInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(parallelizationFactor());
        hashCode = 31 * hashCode + Objects.hashCode(eventSourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(filterCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(functionArn());
        hashCode = 31 * hashCode + Objects.hashCode(lastModified());
        hashCode = 31 * hashCode + Objects.hashCode(lastProcessingResult());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(stateTransitionReason());
        hashCode = 31 * hashCode + Objects.hashCode(destinationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasTopics() ? topics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasQueues() ? queues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceAccessConfigurations() ? sourceAccessConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(selfManagedEventSource());
        hashCode = 31 * hashCode + Objects.hashCode(maximumRecordAgeInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(bisectBatchOnFunctionError());
        hashCode = 31 * hashCode + Objects.hashCode(maximumRetryAttempts());
        hashCode = 31 * hashCode + Objects.hashCode(tumblingWindowInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(hasFunctionResponseTypes() ? functionResponseTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(amazonManagedKafkaEventSourceConfig());
        hashCode = 31 * hashCode + Objects.hashCode(selfManagedKafkaEventSourceConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventSourceMappingConfiguration)) {
            return false;
        }
        EventSourceMappingConfiguration other = (EventSourceMappingConfiguration) obj;
        return Objects.equals(uuid(), other.uuid())
                && Objects.equals(startingPositionAsString(), other.startingPositionAsString())
                && Objects.equals(startingPositionTimestamp(), other.startingPositionTimestamp())
                && Objects.equals(batchSize(), other.batchSize())
                && Objects.equals(maximumBatchingWindowInSeconds(), other.maximumBatchingWindowInSeconds())
                && Objects.equals(parallelizationFactor(), other.parallelizationFactor())
                && Objects.equals(eventSourceArn(), other.eventSourceArn())
                && Objects.equals(filterCriteria(), other.filterCriteria()) && Objects.equals(functionArn(), other.functionArn())
                && Objects.equals(lastModified(), other.lastModified())
                && Objects.equals(lastProcessingResult(), other.lastProcessingResult()) && Objects.equals(state(), other.state())
                && Objects.equals(stateTransitionReason(), other.stateTransitionReason())
                && Objects.equals(destinationConfig(), other.destinationConfig()) && hasTopics() == other.hasTopics()
                && Objects.equals(topics(), other.topics()) && hasQueues() == other.hasQueues()
                && Objects.equals(queues(), other.queues())
                && hasSourceAccessConfigurations() == other.hasSourceAccessConfigurations()
                && Objects.equals(sourceAccessConfigurations(), other.sourceAccessConfigurations())
                && Objects.equals(selfManagedEventSource(), other.selfManagedEventSource())
                && Objects.equals(maximumRecordAgeInSeconds(), other.maximumRecordAgeInSeconds())
                && Objects.equals(bisectBatchOnFunctionError(), other.bisectBatchOnFunctionError())
                && Objects.equals(maximumRetryAttempts(), other.maximumRetryAttempts())
                && Objects.equals(tumblingWindowInSeconds(), other.tumblingWindowInSeconds())
                && hasFunctionResponseTypes() == other.hasFunctionResponseTypes()
                && Objects.equals(functionResponseTypesAsStrings(), other.functionResponseTypesAsStrings())
                && Objects.equals(amazonManagedKafkaEventSourceConfig(), other.amazonManagedKafkaEventSourceConfig())
                && Objects.equals(selfManagedKafkaEventSourceConfig(), other.selfManagedKafkaEventSourceConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventSourceMappingConfiguration").add("UUID", uuid())
                .add("StartingPosition", startingPositionAsString())
                .add("StartingPositionTimestamp", startingPositionTimestamp()).add("BatchSize", batchSize())
                .add("MaximumBatchingWindowInSeconds", maximumBatchingWindowInSeconds())
                .add("ParallelizationFactor", parallelizationFactor()).add("EventSourceArn", eventSourceArn())
                .add("FilterCriteria", filterCriteria()).add("FunctionArn", functionArn()).add("LastModified", lastModified())
                .add("LastProcessingResult", lastProcessingResult()).add("State", state())
                .add("StateTransitionReason", stateTransitionReason()).add("DestinationConfig", destinationConfig())
                .add("Topics", hasTopics() ? topics() : null).add("Queues", hasQueues() ? queues() : null)
                .add("SourceAccessConfigurations", hasSourceAccessConfigurations() ? sourceAccessConfigurations() : null)
                .add("SelfManagedEventSource", selfManagedEventSource())
                .add("MaximumRecordAgeInSeconds", maximumRecordAgeInSeconds())
                .add("BisectBatchOnFunctionError", bisectBatchOnFunctionError())
                .add("MaximumRetryAttempts", maximumRetryAttempts()).add("TumblingWindowInSeconds", tumblingWindowInSeconds())
                .add("FunctionResponseTypes", hasFunctionResponseTypes() ? functionResponseTypesAsStrings() : null)
                .add("AmazonManagedKafkaEventSourceConfig", amazonManagedKafkaEventSourceConfig())
                .add("SelfManagedKafkaEventSourceConfig", selfManagedKafkaEventSourceConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UUID":
            return Optional.ofNullable(clazz.cast(uuid()));
        case "StartingPosition":
            return Optional.ofNullable(clazz.cast(startingPositionAsString()));
        case "StartingPositionTimestamp":
            return Optional.ofNullable(clazz.cast(startingPositionTimestamp()));
        case "BatchSize":
            return Optional.ofNullable(clazz.cast(batchSize()));
        case "MaximumBatchingWindowInSeconds":
            return Optional.ofNullable(clazz.cast(maximumBatchingWindowInSeconds()));
        case "ParallelizationFactor":
            return Optional.ofNullable(clazz.cast(parallelizationFactor()));
        case "EventSourceArn":
            return Optional.ofNullable(clazz.cast(eventSourceArn()));
        case "FilterCriteria":
            return Optional.ofNullable(clazz.cast(filterCriteria()));
        case "FunctionArn":
            return Optional.ofNullable(clazz.cast(functionArn()));
        case "LastModified":
            return Optional.ofNullable(clazz.cast(lastModified()));
        case "LastProcessingResult":
            return Optional.ofNullable(clazz.cast(lastProcessingResult()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "StateTransitionReason":
            return Optional.ofNullable(clazz.cast(stateTransitionReason()));
        case "DestinationConfig":
            return Optional.ofNullable(clazz.cast(destinationConfig()));
        case "Topics":
            return Optional.ofNullable(clazz.cast(topics()));
        case "Queues":
            return Optional.ofNullable(clazz.cast(queues()));
        case "SourceAccessConfigurations":
            return Optional.ofNullable(clazz.cast(sourceAccessConfigurations()));
        case "SelfManagedEventSource":
            return Optional.ofNullable(clazz.cast(selfManagedEventSource()));
        case "MaximumRecordAgeInSeconds":
            return Optional.ofNullable(clazz.cast(maximumRecordAgeInSeconds()));
        case "BisectBatchOnFunctionError":
            return Optional.ofNullable(clazz.cast(bisectBatchOnFunctionError()));
        case "MaximumRetryAttempts":
            return Optional.ofNullable(clazz.cast(maximumRetryAttempts()));
        case "TumblingWindowInSeconds":
            return Optional.ofNullable(clazz.cast(tumblingWindowInSeconds()));
        case "FunctionResponseTypes":
            return Optional.ofNullable(clazz.cast(functionResponseTypesAsStrings()));
        case "AmazonManagedKafkaEventSourceConfig":
            return Optional.ofNullable(clazz.cast(amazonManagedKafkaEventSourceConfig()));
        case "SelfManagedKafkaEventSourceConfig":
            return Optional.ofNullable(clazz.cast(selfManagedKafkaEventSourceConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventSourceMappingConfiguration, T> g) {
        return obj -> g.apply((EventSourceMappingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventSourceMappingConfiguration> {
        /**
         * <p>
         * The identifier of the event source mapping.
         * </p>
         * 
         * @param uuid
         *        The identifier of the event source mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uuid(String uuid);

        /**
         * <p>
         * The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB, and
         * Amazon MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
         * </p>
         * 
         * @param startingPosition
         *        The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB,
         *        and Amazon MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
         * @see EventSourcePosition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventSourcePosition
         */
        Builder startingPosition(String startingPosition);

        /**
         * <p>
         * The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB, and
         * Amazon MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
         * </p>
         * 
         * @param startingPosition
         *        The position in a stream from which to start reading. Required for Amazon Kinesis, Amazon DynamoDB,
         *        and Amazon MSK stream sources. <code>AT_TIMESTAMP</code> is supported only for Amazon Kinesis streams.
         * @see EventSourcePosition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventSourcePosition
         */
        Builder startingPosition(EventSourcePosition startingPosition);

        /**
         * <p>
         * With <code>StartingPosition</code> set to <code>AT_TIMESTAMP</code>, the time from which to start reading.
         * </p>
         * 
         * @param startingPositionTimestamp
         *        With <code>StartingPosition</code> set to <code>AT_TIMESTAMP</code>, the time from which to start
         *        reading.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startingPositionTimestamp(Instant startingPositionTimestamp);

        /**
         * <p>
         * The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your
         * function. Lambda passes all of the records in the batch to the function in a single call, up to the payload
         * limit for synchronous invocation (6 MB).
         * </p>
         * <p>
         * Default value: Varies by service. For Amazon SQS, the default is 10. For all other services, the default is
         * 100.
         * </p>
         * <p>
         * Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
         * <code>MaximumBatchingWindowInSeconds</code> to at least 1.
         * </p>
         * 
         * @param batchSize
         *        The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to
         *        your function. Lambda passes all of the records in the batch to the function in a single call, up to
         *        the payload limit for synchronous invocation (6 MB).</p>
         *        <p>
         *        Default value: Varies by service. For Amazon SQS, the default is 10. For all other services, the
         *        default is 100.
         *        </p>
         *        <p>
         *        Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
         *        <code>MaximumBatchingWindowInSeconds</code> to at least 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder batchSize(Integer batchSize);

        /**
         * <p>
         * (Streams and Amazon SQS standard queues) The maximum amount of time, in seconds, that Lambda spends gathering
         * records before invoking the function.
         * </p>
         * <p>
         * Default: 0
         * </p>
         * <p>
         * Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
         * <code>MaximumBatchingWindowInSeconds</code> to at least 1.
         * </p>
         * 
         * @param maximumBatchingWindowInSeconds
         *        (Streams and Amazon SQS standard queues) The maximum amount of time, in seconds, that Lambda spends
         *        gathering records before invoking the function.</p>
         *        <p>
         *        Default: 0
         *        </p>
         *        <p>
         *        Related setting: When you set <code>BatchSize</code> to a value greater than 10, you must set
         *        <code>MaximumBatchingWindowInSeconds</code> to at least 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumBatchingWindowInSeconds(Integer maximumBatchingWindowInSeconds);

        /**
         * <p>
         * (Streams only) The number of batches to process concurrently from each shard. The default value is 1.
         * </p>
         * 
         * @param parallelizationFactor
         *        (Streams only) The number of batches to process concurrently from each shard. The default value is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelizationFactor(Integer parallelizationFactor);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the event source.
         * </p>
         * 
         * @param eventSourceArn
         *        The Amazon Resource Name (ARN) of the event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSourceArn(String eventSourceArn);

        /**
         * <p>
         * (Streams and Amazon SQS) An object that defines the filter criteria that determine whether Lambda should
         * process an event. For more information, see <a
         * href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html">Lambda event
         * filtering</a>.
         * </p>
         * 
         * @param filterCriteria
         *        (Streams and Amazon SQS) An object that defines the filter criteria that determine whether Lambda
         *        should process an event. For more information, see <a
         *        href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html">Lambda event
         *        filtering</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterCriteria(FilterCriteria filterCriteria);

        /**
         * <p>
         * (Streams and Amazon SQS) An object that defines the filter criteria that determine whether Lambda should
         * process an event. For more information, see <a
         * href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html">Lambda event
         * filtering</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link FilterCriteria.Builder} avoiding the need
         * to create one manually via {@link FilterCriteria#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FilterCriteria.Builder#build()} is called immediately and its
         * result is passed to {@link #filterCriteria(FilterCriteria)}.
         * 
         * @param filterCriteria
         *        a consumer that will call methods on {@link FilterCriteria.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filterCriteria(FilterCriteria)
         */
        default Builder filterCriteria(Consumer<FilterCriteria.Builder> filterCriteria) {
            return filterCriteria(FilterCriteria.builder().applyMutation(filterCriteria).build());
        }

        /**
         * <p>
         * The ARN of the Lambda function.
         * </p>
         * 
         * @param functionArn
         *        The ARN of the Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionArn(String functionArn);

        /**
         * <p>
         * The date that the event source mapping was last updated or that its state changed.
         * </p>
         * 
         * @param lastModified
         *        The date that the event source mapping was last updated or that its state changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModified(Instant lastModified);

        /**
         * <p>
         * The result of the last Lambda invocation of your function.
         * </p>
         * 
         * @param lastProcessingResult
         *        The result of the last Lambda invocation of your function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastProcessingResult(String lastProcessingResult);

        /**
         * <p>
         * The state of the event source mapping. It can be one of the following: <code>Creating</code>,
         * <code>Enabling</code>, <code>Enabled</code>, <code>Disabling</code>, <code>Disabled</code>,
         * <code>Updating</code>, or <code>Deleting</code>.
         * </p>
         * 
         * @param state
         *        The state of the event source mapping. It can be one of the following: <code>Creating</code>,
         *        <code>Enabling</code>, <code>Enabled</code>, <code>Disabling</code>, <code>Disabled</code>,
         *        <code>Updating</code>, or <code>Deleting</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * Indicates whether a user or Lambda made the last change to the event source mapping.
         * </p>
         * 
         * @param stateTransitionReason
         *        Indicates whether a user or Lambda made the last change to the event source mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateTransitionReason(String stateTransitionReason);

        /**
         * <p>
         * (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
         * </p>
         * 
         * @param destinationConfig
         *        (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationConfig(DestinationConfig destinationConfig);

        /**
         * <p>
         * (Streams only) An Amazon SQS queue or Amazon SNS topic destination for discarded records.
         * </p>
         * This is a convenience method that creates an instance of the {@link DestinationConfig.Builder} avoiding the
         * need to create one manually via {@link DestinationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DestinationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #destinationConfig(DestinationConfig)}.
         * 
         * @param destinationConfig
         *        a consumer that will call methods on {@link DestinationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationConfig(DestinationConfig)
         */
        default Builder destinationConfig(Consumer<DestinationConfig.Builder> destinationConfig) {
            return destinationConfig(DestinationConfig.builder().applyMutation(destinationConfig).build());
        }

        /**
         * <p>
         * The name of the Kafka topic.
         * </p>
         * 
         * @param topics
         *        The name of the Kafka topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topics(Collection<String> topics);

        /**
         * <p>
         * The name of the Kafka topic.
         * </p>
         * 
         * @param topics
         *        The name of the Kafka topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topics(String... topics);

        /**
         * <p>
         * (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
         * </p>
         * 
         * @param queues
         *        (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queues(Collection<String> queues);

        /**
         * <p>
         * (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
         * </p>
         * 
         * @param queues
         *        (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queues(String... queues);

        /**
         * <p>
         * An array of the authentication protocol, VPC components, or virtual host to secure and define your event
         * source.
         * </p>
         * 
         * @param sourceAccessConfigurations
         *        An array of the authentication protocol, VPC components, or virtual host to secure and define your
         *        event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceAccessConfigurations(Collection<SourceAccessConfiguration> sourceAccessConfigurations);

        /**
         * <p>
         * An array of the authentication protocol, VPC components, or virtual host to secure and define your event
         * source.
         * </p>
         * 
         * @param sourceAccessConfigurations
         *        An array of the authentication protocol, VPC components, or virtual host to secure and define your
         *        event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceAccessConfigurations(SourceAccessConfiguration... sourceAccessConfigurations);

        /**
         * <p>
         * An array of the authentication protocol, VPC components, or virtual host to secure and define your event
         * source.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lambda.model.SourceAccessConfiguration.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.lambda.model.SourceAccessConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lambda.model.SourceAccessConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceAccessConfigurations(List<SourceAccessConfiguration>)}.
         * 
         * @param sourceAccessConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lambda.model.SourceAccessConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceAccessConfigurations(java.util.Collection<SourceAccessConfiguration>)
         */
        Builder sourceAccessConfigurations(Consumer<SourceAccessConfiguration.Builder>... sourceAccessConfigurations);

        /**
         * <p>
         * The self-managed Apache Kafka cluster for your event source.
         * </p>
         * 
         * @param selfManagedEventSource
         *        The self-managed Apache Kafka cluster for your event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfManagedEventSource(SelfManagedEventSource selfManagedEventSource);

        /**
         * <p>
         * The self-managed Apache Kafka cluster for your event source.
         * </p>
         * This is a convenience method that creates an instance of the {@link SelfManagedEventSource.Builder} avoiding
         * the need to create one manually via {@link SelfManagedEventSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SelfManagedEventSource.Builder#build()} is called immediately and
         * its result is passed to {@link #selfManagedEventSource(SelfManagedEventSource)}.
         * 
         * @param selfManagedEventSource
         *        a consumer that will call methods on {@link SelfManagedEventSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #selfManagedEventSource(SelfManagedEventSource)
         */
        default Builder selfManagedEventSource(Consumer<SelfManagedEventSource.Builder> selfManagedEventSource) {
            return selfManagedEventSource(SelfManagedEventSource.builder().applyMutation(selfManagedEventSource).build());
        }

        /**
         * <p>
         * (Streams only) Discard records older than the specified age. The default value is -1, which sets the maximum
         * age to infinite. When the value is set to infinite, Lambda never discards old records.
         * </p>
         * 
         * @param maximumRecordAgeInSeconds
         *        (Streams only) Discard records older than the specified age. The default value is -1, which sets the
         *        maximum age to infinite. When the value is set to infinite, Lambda never discards old records.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumRecordAgeInSeconds(Integer maximumRecordAgeInSeconds);

        /**
         * <p>
         * (Streams only) If the function returns an error, split the batch in two and retry. The default value is
         * false.
         * </p>
         * 
         * @param bisectBatchOnFunctionError
         *        (Streams only) If the function returns an error, split the batch in two and retry. The default value
         *        is false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bisectBatchOnFunctionError(Boolean bisectBatchOnFunctionError);

        /**
         * <p>
         * (Streams only) Discard records after the specified number of retries. The default value is -1, which sets the
         * maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records
         * until the record expires in the event source.
         * </p>
         * 
         * @param maximumRetryAttempts
         *        (Streams only) Discard records after the specified number of retries. The default value is -1, which
         *        sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries
         *        failed records until the record expires in the event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumRetryAttempts(Integer maximumRetryAttempts);

        /**
         * <p>
         * (Streams only) The duration in seconds of a processing window. The range is 1–900 seconds.
         * </p>
         * 
         * @param tumblingWindowInSeconds
         *        (Streams only) The duration in seconds of a processing window. The range is 1–900 seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tumblingWindowInSeconds(Integer tumblingWindowInSeconds);

        /**
         * <p>
         * (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * </p>
         * 
         * @param functionResponseTypes
         *        (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionResponseTypesWithStrings(Collection<String> functionResponseTypes);

        /**
         * <p>
         * (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * </p>
         * 
         * @param functionResponseTypes
         *        (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionResponseTypesWithStrings(String... functionResponseTypes);

        /**
         * <p>
         * (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * </p>
         * 
         * @param functionResponseTypes
         *        (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionResponseTypes(Collection<FunctionResponseType> functionResponseTypes);

        /**
         * <p>
         * (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * </p>
         * 
         * @param functionResponseTypes
         *        (Streams and Amazon SQS) A list of current response type enums applied to the event source mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionResponseTypes(FunctionResponseType... functionResponseTypes);

        /**
         * <p>
         * Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
         * </p>
         * 
         * @param amazonManagedKafkaEventSourceConfig
         *        Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event
         *        source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder amazonManagedKafkaEventSourceConfig(AmazonManagedKafkaEventSourceConfig amazonManagedKafkaEventSourceConfig);

        /**
         * <p>
         * Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link AmazonManagedKafkaEventSourceConfig.Builder} avoiding the need to create one manually via
         * {@link AmazonManagedKafkaEventSourceConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AmazonManagedKafkaEventSourceConfig.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #amazonManagedKafkaEventSourceConfig(AmazonManagedKafkaEventSourceConfig)}.
         * 
         * @param amazonManagedKafkaEventSourceConfig
         *        a consumer that will call methods on {@link AmazonManagedKafkaEventSourceConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #amazonManagedKafkaEventSourceConfig(AmazonManagedKafkaEventSourceConfig)
         */
        default Builder amazonManagedKafkaEventSourceConfig(
                Consumer<AmazonManagedKafkaEventSourceConfig.Builder> amazonManagedKafkaEventSourceConfig) {
            return amazonManagedKafkaEventSourceConfig(AmazonManagedKafkaEventSourceConfig.builder()
                    .applyMutation(amazonManagedKafkaEventSourceConfig).build());
        }

        /**
         * <p>
         * Specific configuration settings for a self-managed Apache Kafka event source.
         * </p>
         * 
         * @param selfManagedKafkaEventSourceConfig
         *        Specific configuration settings for a self-managed Apache Kafka event source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfManagedKafkaEventSourceConfig(SelfManagedKafkaEventSourceConfig selfManagedKafkaEventSourceConfig);

        /**
         * <p>
         * Specific configuration settings for a self-managed Apache Kafka event source.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link SelfManagedKafkaEventSourceConfig.Builder} avoiding the need to create one manually via
         * {@link SelfManagedKafkaEventSourceConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SelfManagedKafkaEventSourceConfig.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #selfManagedKafkaEventSourceConfig(SelfManagedKafkaEventSourceConfig)}.
         * 
         * @param selfManagedKafkaEventSourceConfig
         *        a consumer that will call methods on {@link SelfManagedKafkaEventSourceConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #selfManagedKafkaEventSourceConfig(SelfManagedKafkaEventSourceConfig)
         */
        default Builder selfManagedKafkaEventSourceConfig(
                Consumer<SelfManagedKafkaEventSourceConfig.Builder> selfManagedKafkaEventSourceConfig) {
            return selfManagedKafkaEventSourceConfig(SelfManagedKafkaEventSourceConfig.builder()
                    .applyMutation(selfManagedKafkaEventSourceConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String uuid;

        private String startingPosition;

        private Instant startingPositionTimestamp;

        private Integer batchSize;

        private Integer maximumBatchingWindowInSeconds;

        private Integer parallelizationFactor;

        private String eventSourceArn;

        private FilterCriteria filterCriteria;

        private String functionArn;

        private Instant lastModified;

        private String lastProcessingResult;

        private String state;

        private String stateTransitionReason;

        private DestinationConfig destinationConfig;

        private List<String> topics = DefaultSdkAutoConstructList.getInstance();

        private List<String> queues = DefaultSdkAutoConstructList.getInstance();

        private List<SourceAccessConfiguration> sourceAccessConfigurations = DefaultSdkAutoConstructList.getInstance();

        private SelfManagedEventSource selfManagedEventSource;

        private Integer maximumRecordAgeInSeconds;

        private Boolean bisectBatchOnFunctionError;

        private Integer maximumRetryAttempts;

        private Integer tumblingWindowInSeconds;

        private List<String> functionResponseTypes = DefaultSdkAutoConstructList.getInstance();

        private AmazonManagedKafkaEventSourceConfig amazonManagedKafkaEventSourceConfig;

        private SelfManagedKafkaEventSourceConfig selfManagedKafkaEventSourceConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(EventSourceMappingConfiguration model) {
            uuid(model.uuid);
            startingPosition(model.startingPosition);
            startingPositionTimestamp(model.startingPositionTimestamp);
            batchSize(model.batchSize);
            maximumBatchingWindowInSeconds(model.maximumBatchingWindowInSeconds);
            parallelizationFactor(model.parallelizationFactor);
            eventSourceArn(model.eventSourceArn);
            filterCriteria(model.filterCriteria);
            functionArn(model.functionArn);
            lastModified(model.lastModified);
            lastProcessingResult(model.lastProcessingResult);
            state(model.state);
            stateTransitionReason(model.stateTransitionReason);
            destinationConfig(model.destinationConfig);
            topics(model.topics);
            queues(model.queues);
            sourceAccessConfigurations(model.sourceAccessConfigurations);
            selfManagedEventSource(model.selfManagedEventSource);
            maximumRecordAgeInSeconds(model.maximumRecordAgeInSeconds);
            bisectBatchOnFunctionError(model.bisectBatchOnFunctionError);
            maximumRetryAttempts(model.maximumRetryAttempts);
            tumblingWindowInSeconds(model.tumblingWindowInSeconds);
            functionResponseTypesWithStrings(model.functionResponseTypes);
            amazonManagedKafkaEventSourceConfig(model.amazonManagedKafkaEventSourceConfig);
            selfManagedKafkaEventSourceConfig(model.selfManagedKafkaEventSourceConfig);
        }

        public final String getUuid() {
            return uuid;
        }

        public final void setUuid(String uuid) {
            this.uuid = uuid;
        }

        @Override
        public final Builder uuid(String uuid) {
            this.uuid = uuid;
            return this;
        }

        public final String getStartingPosition() {
            return startingPosition;
        }

        public final void setStartingPosition(String startingPosition) {
            this.startingPosition = startingPosition;
        }

        @Override
        public final Builder startingPosition(String startingPosition) {
            this.startingPosition = startingPosition;
            return this;
        }

        @Override
        public final Builder startingPosition(EventSourcePosition startingPosition) {
            this.startingPosition(startingPosition == null ? null : startingPosition.toString());
            return this;
        }

        public final Instant getStartingPositionTimestamp() {
            return startingPositionTimestamp;
        }

        public final void setStartingPositionTimestamp(Instant startingPositionTimestamp) {
            this.startingPositionTimestamp = startingPositionTimestamp;
        }

        @Override
        public final Builder startingPositionTimestamp(Instant startingPositionTimestamp) {
            this.startingPositionTimestamp = startingPositionTimestamp;
            return this;
        }

        public final Integer getBatchSize() {
            return batchSize;
        }

        public final void setBatchSize(Integer batchSize) {
            this.batchSize = batchSize;
        }

        @Override
        public final Builder batchSize(Integer batchSize) {
            this.batchSize = batchSize;
            return this;
        }

        public final Integer getMaximumBatchingWindowInSeconds() {
            return maximumBatchingWindowInSeconds;
        }

        public final void setMaximumBatchingWindowInSeconds(Integer maximumBatchingWindowInSeconds) {
            this.maximumBatchingWindowInSeconds = maximumBatchingWindowInSeconds;
        }

        @Override
        public final Builder maximumBatchingWindowInSeconds(Integer maximumBatchingWindowInSeconds) {
            this.maximumBatchingWindowInSeconds = maximumBatchingWindowInSeconds;
            return this;
        }

        public final Integer getParallelizationFactor() {
            return parallelizationFactor;
        }

        public final void setParallelizationFactor(Integer parallelizationFactor) {
            this.parallelizationFactor = parallelizationFactor;
        }

        @Override
        public final Builder parallelizationFactor(Integer parallelizationFactor) {
            this.parallelizationFactor = parallelizationFactor;
            return this;
        }

        public final String getEventSourceArn() {
            return eventSourceArn;
        }

        public final void setEventSourceArn(String eventSourceArn) {
            this.eventSourceArn = eventSourceArn;
        }

        @Override
        public final Builder eventSourceArn(String eventSourceArn) {
            this.eventSourceArn = eventSourceArn;
            return this;
        }

        public final FilterCriteria.Builder getFilterCriteria() {
            return filterCriteria != null ? filterCriteria.toBuilder() : null;
        }

        public final void setFilterCriteria(FilterCriteria.BuilderImpl filterCriteria) {
            this.filterCriteria = filterCriteria != null ? filterCriteria.build() : null;
        }

        @Override
        public final Builder filterCriteria(FilterCriteria filterCriteria) {
            this.filterCriteria = filterCriteria;
            return this;
        }

        public final String getFunctionArn() {
            return functionArn;
        }

        public final void setFunctionArn(String functionArn) {
            this.functionArn = functionArn;
        }

        @Override
        public final Builder functionArn(String functionArn) {
            this.functionArn = functionArn;
            return this;
        }

        public final Instant getLastModified() {
            return lastModified;
        }

        public final void setLastModified(Instant lastModified) {
            this.lastModified = lastModified;
        }

        @Override
        public final Builder lastModified(Instant lastModified) {
            this.lastModified = lastModified;
            return this;
        }

        public final String getLastProcessingResult() {
            return lastProcessingResult;
        }

        public final void setLastProcessingResult(String lastProcessingResult) {
            this.lastProcessingResult = lastProcessingResult;
        }

        @Override
        public final Builder lastProcessingResult(String lastProcessingResult) {
            this.lastProcessingResult = lastProcessingResult;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final String getStateTransitionReason() {
            return stateTransitionReason;
        }

        public final void setStateTransitionReason(String stateTransitionReason) {
            this.stateTransitionReason = stateTransitionReason;
        }

        @Override
        public final Builder stateTransitionReason(String stateTransitionReason) {
            this.stateTransitionReason = stateTransitionReason;
            return this;
        }

        public final DestinationConfig.Builder getDestinationConfig() {
            return destinationConfig != null ? destinationConfig.toBuilder() : null;
        }

        public final void setDestinationConfig(DestinationConfig.BuilderImpl destinationConfig) {
            this.destinationConfig = destinationConfig != null ? destinationConfig.build() : null;
        }

        @Override
        public final Builder destinationConfig(DestinationConfig destinationConfig) {
            this.destinationConfig = destinationConfig;
            return this;
        }

        public final Collection<String> getTopics() {
            if (topics instanceof SdkAutoConstructList) {
                return null;
            }
            return topics;
        }

        public final void setTopics(Collection<String> topics) {
            this.topics = TopicsCopier.copy(topics);
        }

        @Override
        public final Builder topics(Collection<String> topics) {
            this.topics = TopicsCopier.copy(topics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topics(String... topics) {
            topics(Arrays.asList(topics));
            return this;
        }

        public final Collection<String> getQueues() {
            if (queues instanceof SdkAutoConstructList) {
                return null;
            }
            return queues;
        }

        public final void setQueues(Collection<String> queues) {
            this.queues = QueuesCopier.copy(queues);
        }

        @Override
        public final Builder queues(Collection<String> queues) {
            this.queues = QueuesCopier.copy(queues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder queues(String... queues) {
            queues(Arrays.asList(queues));
            return this;
        }

        public final List<SourceAccessConfiguration.Builder> getSourceAccessConfigurations() {
            List<SourceAccessConfiguration.Builder> result = SourceAccessConfigurationsCopier
                    .copyToBuilder(this.sourceAccessConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceAccessConfigurations(
                Collection<SourceAccessConfiguration.BuilderImpl> sourceAccessConfigurations) {
            this.sourceAccessConfigurations = SourceAccessConfigurationsCopier.copyFromBuilder(sourceAccessConfigurations);
        }

        @Override
        public final Builder sourceAccessConfigurations(Collection<SourceAccessConfiguration> sourceAccessConfigurations) {
            this.sourceAccessConfigurations = SourceAccessConfigurationsCopier.copy(sourceAccessConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceAccessConfigurations(SourceAccessConfiguration... sourceAccessConfigurations) {
            sourceAccessConfigurations(Arrays.asList(sourceAccessConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceAccessConfigurations(Consumer<SourceAccessConfiguration.Builder>... sourceAccessConfigurations) {
            sourceAccessConfigurations(Stream.of(sourceAccessConfigurations)
                    .map(c -> SourceAccessConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final SelfManagedEventSource.Builder getSelfManagedEventSource() {
            return selfManagedEventSource != null ? selfManagedEventSource.toBuilder() : null;
        }

        public final void setSelfManagedEventSource(SelfManagedEventSource.BuilderImpl selfManagedEventSource) {
            this.selfManagedEventSource = selfManagedEventSource != null ? selfManagedEventSource.build() : null;
        }

        @Override
        public final Builder selfManagedEventSource(SelfManagedEventSource selfManagedEventSource) {
            this.selfManagedEventSource = selfManagedEventSource;
            return this;
        }

        public final Integer getMaximumRecordAgeInSeconds() {
            return maximumRecordAgeInSeconds;
        }

        public final void setMaximumRecordAgeInSeconds(Integer maximumRecordAgeInSeconds) {
            this.maximumRecordAgeInSeconds = maximumRecordAgeInSeconds;
        }

        @Override
        public final Builder maximumRecordAgeInSeconds(Integer maximumRecordAgeInSeconds) {
            this.maximumRecordAgeInSeconds = maximumRecordAgeInSeconds;
            return this;
        }

        public final Boolean getBisectBatchOnFunctionError() {
            return bisectBatchOnFunctionError;
        }

        public final void setBisectBatchOnFunctionError(Boolean bisectBatchOnFunctionError) {
            this.bisectBatchOnFunctionError = bisectBatchOnFunctionError;
        }

        @Override
        public final Builder bisectBatchOnFunctionError(Boolean bisectBatchOnFunctionError) {
            this.bisectBatchOnFunctionError = bisectBatchOnFunctionError;
            return this;
        }

        public final Integer getMaximumRetryAttempts() {
            return maximumRetryAttempts;
        }

        public final void setMaximumRetryAttempts(Integer maximumRetryAttempts) {
            this.maximumRetryAttempts = maximumRetryAttempts;
        }

        @Override
        public final Builder maximumRetryAttempts(Integer maximumRetryAttempts) {
            this.maximumRetryAttempts = maximumRetryAttempts;
            return this;
        }

        public final Integer getTumblingWindowInSeconds() {
            return tumblingWindowInSeconds;
        }

        public final void setTumblingWindowInSeconds(Integer tumblingWindowInSeconds) {
            this.tumblingWindowInSeconds = tumblingWindowInSeconds;
        }

        @Override
        public final Builder tumblingWindowInSeconds(Integer tumblingWindowInSeconds) {
            this.tumblingWindowInSeconds = tumblingWindowInSeconds;
            return this;
        }

        public final Collection<String> getFunctionResponseTypes() {
            if (functionResponseTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return functionResponseTypes;
        }

        public final void setFunctionResponseTypes(Collection<String> functionResponseTypes) {
            this.functionResponseTypes = FunctionResponseTypeListCopier.copy(functionResponseTypes);
        }

        @Override
        public final Builder functionResponseTypesWithStrings(Collection<String> functionResponseTypes) {
            this.functionResponseTypes = FunctionResponseTypeListCopier.copy(functionResponseTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder functionResponseTypesWithStrings(String... functionResponseTypes) {
            functionResponseTypesWithStrings(Arrays.asList(functionResponseTypes));
            return this;
        }

        @Override
        public final Builder functionResponseTypes(Collection<FunctionResponseType> functionResponseTypes) {
            this.functionResponseTypes = FunctionResponseTypeListCopier.copyEnumToString(functionResponseTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder functionResponseTypes(FunctionResponseType... functionResponseTypes) {
            functionResponseTypes(Arrays.asList(functionResponseTypes));
            return this;
        }

        public final AmazonManagedKafkaEventSourceConfig.Builder getAmazonManagedKafkaEventSourceConfig() {
            return amazonManagedKafkaEventSourceConfig != null ? amazonManagedKafkaEventSourceConfig.toBuilder() : null;
        }

        public final void setAmazonManagedKafkaEventSourceConfig(
                AmazonManagedKafkaEventSourceConfig.BuilderImpl amazonManagedKafkaEventSourceConfig) {
            this.amazonManagedKafkaEventSourceConfig = amazonManagedKafkaEventSourceConfig != null ? amazonManagedKafkaEventSourceConfig
                    .build() : null;
        }

        @Override
        public final Builder amazonManagedKafkaEventSourceConfig(
                AmazonManagedKafkaEventSourceConfig amazonManagedKafkaEventSourceConfig) {
            this.amazonManagedKafkaEventSourceConfig = amazonManagedKafkaEventSourceConfig;
            return this;
        }

        public final SelfManagedKafkaEventSourceConfig.Builder getSelfManagedKafkaEventSourceConfig() {
            return selfManagedKafkaEventSourceConfig != null ? selfManagedKafkaEventSourceConfig.toBuilder() : null;
        }

        public final void setSelfManagedKafkaEventSourceConfig(
                SelfManagedKafkaEventSourceConfig.BuilderImpl selfManagedKafkaEventSourceConfig) {
            this.selfManagedKafkaEventSourceConfig = selfManagedKafkaEventSourceConfig != null ? selfManagedKafkaEventSourceConfig
                    .build() : null;
        }

        @Override
        public final Builder selfManagedKafkaEventSourceConfig(SelfManagedKafkaEventSourceConfig selfManagedKafkaEventSourceConfig) {
            this.selfManagedKafkaEventSourceConfig = selfManagedKafkaEventSourceConfig;
            return this;
        }

        @Override
        public EventSourceMappingConfiguration build() {
            return new EventSourceMappingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
