/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.services.lambda.LambdaClient;
import software.amazon.awssdk.services.lambda.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationResponse;
import software.amazon.awssdk.services.lambda.model.GetFunctionRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionResponse;
import software.amazon.awssdk.services.lambda.model.LambdaRequest;
import software.amazon.awssdk.services.lambda.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultLambdaWaiter implements LambdaWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final LambdaClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetFunctionResponse> functionExistsWaiter;

    private final Waiter<GetFunctionConfigurationResponse> functionActiveWaiter;

    private final Waiter<GetFunctionConfigurationResponse> functionUpdatedWaiter;

    private final Waiter<GetFunctionResponse> functionActiveV2Waiter;

    private final Waiter<GetFunctionResponse> functionUpdatedV2Waiter;

    private final Waiter<GetFunctionConfigurationResponse> publishedVersionActiveWaiter;

    private DefaultLambdaWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = LambdaClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.functionExistsWaiter = Waiter.builder(GetFunctionResponse.class).acceptors(functionExistsWaiterAcceptors())
                .overrideConfiguration(functionExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.functionActiveWaiter = Waiter.builder(GetFunctionConfigurationResponse.class)
                .acceptors(functionActiveWaiterAcceptors())
                .overrideConfiguration(functionActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.functionUpdatedWaiter = Waiter.builder(GetFunctionConfigurationResponse.class)
                .acceptors(functionUpdatedWaiterAcceptors())
                .overrideConfiguration(functionUpdatedWaiterConfig(builder.overrideConfiguration)).build();
        this.functionActiveV2Waiter = Waiter.builder(GetFunctionResponse.class).acceptors(functionActiveV2WaiterAcceptors())
                .overrideConfiguration(functionActiveV2WaiterConfig(builder.overrideConfiguration)).build();
        this.functionUpdatedV2Waiter = Waiter.builder(GetFunctionResponse.class).acceptors(functionUpdatedV2WaiterAcceptors())
                .overrideConfiguration(functionUpdatedV2WaiterConfig(builder.overrideConfiguration)).build();
        this.publishedVersionActiveWaiter = Waiter.builder(GetFunctionConfigurationResponse.class)
                .acceptors(publishedVersionActiveWaiterAcceptors())
                .overrideConfiguration(publishedVersionActiveWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetFunctionConfigurationResponse> waitUntilFunctionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        return functionActiveWaiter.run(() -> client
                .getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)));
    }

    @Override
    public WaiterResponse<GetFunctionConfigurationResponse> waitUntilFunctionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        return functionActiveWaiter.run(
                () -> client.getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)),
                functionActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetFunctionResponse> waitUntilFunctionActiveV2(GetFunctionRequest getFunctionRequest) {
        return functionActiveV2Waiter.run(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)));
    }

    @Override
    public WaiterResponse<GetFunctionResponse> waitUntilFunctionActiveV2(GetFunctionRequest getFunctionRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return functionActiveV2Waiter.run(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)),
                functionActiveV2WaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetFunctionResponse> waitUntilFunctionExists(GetFunctionRequest getFunctionRequest) {
        return functionExistsWaiter.run(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)));
    }

    @Override
    public WaiterResponse<GetFunctionResponse> waitUntilFunctionExists(GetFunctionRequest getFunctionRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return functionExistsWaiter.run(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)),
                functionExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetFunctionConfigurationResponse> waitUntilFunctionUpdated(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        return functionUpdatedWaiter.run(() -> client
                .getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)));
    }

    @Override
    public WaiterResponse<GetFunctionConfigurationResponse> waitUntilFunctionUpdated(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        return functionUpdatedWaiter.run(
                () -> client.getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)),
                functionUpdatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetFunctionResponse> waitUntilFunctionUpdatedV2(GetFunctionRequest getFunctionRequest) {
        return functionUpdatedV2Waiter.run(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)));
    }

    @Override
    public WaiterResponse<GetFunctionResponse> waitUntilFunctionUpdatedV2(GetFunctionRequest getFunctionRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return functionUpdatedV2Waiter.run(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)),
                functionUpdatedV2WaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetFunctionConfigurationResponse> waitUntilPublishedVersionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        return publishedVersionActiveWaiter.run(() -> client
                .getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)));
    }

    @Override
    public WaiterResponse<GetFunctionConfigurationResponse> waitUntilPublishedVersionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        return publishedVersionActiveWaiter.run(
                () -> client.getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)),
                publishedVersionActiveWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetFunctionResponse>> functionExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> functionActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Active");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Failed");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> functionUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("LastUpdateStatus").value(), "Successful");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("LastUpdateStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("LastUpdateStatus").value(), "InProgress");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionResponse>> functionActiveV2WaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("State").value(), "Active");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("State").value(), "Failed");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionResponse>> functionUpdatedV2WaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("LastUpdateStatus").value(), "Successful");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("LastUpdateStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("LastUpdateStatus").value(), "InProgress");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> publishedVersionActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Active");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Failed");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration functionExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionActiveV2WaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(300);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionUpdatedV2WaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(300);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration publishedVersionActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(312);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static LambdaWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends LambdaRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements LambdaWaiter.Builder {
        private LambdaClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public LambdaWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public LambdaWaiter.Builder client(LambdaClient client) {
            this.client = client;
            return this;
        }

        public LambdaWaiter build() {
            return new DefaultLambdaWaiter(this);
        }
    }
}
