/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.launchwizard;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.launchwizard.internal.LaunchWizardServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.launchwizard.model.CreateDeploymentRequest;
import software.amazon.awssdk.services.launchwizard.model.CreateDeploymentResponse;
import software.amazon.awssdk.services.launchwizard.model.DeleteDeploymentRequest;
import software.amazon.awssdk.services.launchwizard.model.DeleteDeploymentResponse;
import software.amazon.awssdk.services.launchwizard.model.GetDeploymentRequest;
import software.amazon.awssdk.services.launchwizard.model.GetDeploymentResponse;
import software.amazon.awssdk.services.launchwizard.model.GetWorkloadDeploymentPatternRequest;
import software.amazon.awssdk.services.launchwizard.model.GetWorkloadDeploymentPatternResponse;
import software.amazon.awssdk.services.launchwizard.model.GetWorkloadRequest;
import software.amazon.awssdk.services.launchwizard.model.GetWorkloadResponse;
import software.amazon.awssdk.services.launchwizard.model.InternalServerException;
import software.amazon.awssdk.services.launchwizard.model.LaunchWizardException;
import software.amazon.awssdk.services.launchwizard.model.ListDeploymentEventsRequest;
import software.amazon.awssdk.services.launchwizard.model.ListDeploymentEventsResponse;
import software.amazon.awssdk.services.launchwizard.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.launchwizard.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.launchwizard.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.launchwizard.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.launchwizard.model.ListWorkloadDeploymentPatternsRequest;
import software.amazon.awssdk.services.launchwizard.model.ListWorkloadDeploymentPatternsResponse;
import software.amazon.awssdk.services.launchwizard.model.ListWorkloadsRequest;
import software.amazon.awssdk.services.launchwizard.model.ListWorkloadsResponse;
import software.amazon.awssdk.services.launchwizard.model.ResourceLimitException;
import software.amazon.awssdk.services.launchwizard.model.ResourceNotFoundException;
import software.amazon.awssdk.services.launchwizard.model.TagResourceRequest;
import software.amazon.awssdk.services.launchwizard.model.TagResourceResponse;
import software.amazon.awssdk.services.launchwizard.model.UntagResourceRequest;
import software.amazon.awssdk.services.launchwizard.model.UntagResourceResponse;
import software.amazon.awssdk.services.launchwizard.model.ValidationException;
import software.amazon.awssdk.services.launchwizard.transform.CreateDeploymentRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.DeleteDeploymentRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.GetWorkloadDeploymentPatternRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.GetWorkloadRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.ListDeploymentEventsRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.ListWorkloadDeploymentPatternsRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.ListWorkloadsRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.launchwizard.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LaunchWizardClient}.
 *
 * @see LaunchWizardClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLaunchWizardClient implements LaunchWizardClient {
    private static final Logger log = Logger.loggerFor(DefaultLaunchWizardClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLaunchWizardClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a deployment for the given workload. Deployments created by this operation are not available in the
     * Launch Wizard console to use the <code>Clone deployment</code> action on.
     * </p>
     *
     * @param createDeploymentRequest
     * @return Result of the CreateDeployment operation returned by the service.
     * @throws ResourceLimitException
     *         You have exceeded an Launch Wizard resource limit. For example, you might have too many deployments in
     *         progress.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.CreateDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/CreateDeployment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDeploymentResponse createDeployment(CreateDeploymentRequest createDeploymentRequest)
            throws ResourceLimitException, InternalServerException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeployment");

            return clientHandler.execute(new ClientExecutionParams<CreateDeploymentRequest, CreateDeploymentResponse>()
                    .withOperationName("CreateDeployment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a deployment.
     * </p>
     *
     * @param deleteDeploymentRequest
     * @return Result of the DeleteDeployment operation returned by the service.
     * @throws ResourceLimitException
     *         You have exceeded an Launch Wizard resource limit. For example, you might have too many deployments in
     *         progress.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.DeleteDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/DeleteDeployment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDeploymentResponse deleteDeployment(DeleteDeploymentRequest deleteDeploymentRequest)
            throws ResourceLimitException, InternalServerException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeployment");

            return clientHandler.execute(new ClientExecutionParams<DeleteDeploymentRequest, DeleteDeploymentResponse>()
                    .withOperationName("DeleteDeployment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the deployment.
     * </p>
     *
     * @param getDeploymentRequest
     * @return Result of the GetDeployment operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.GetDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/GetDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDeploymentResponse getDeployment(GetDeploymentRequest getDeploymentRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeploymentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeployment");

            return clientHandler.execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                    .withOperationName("GetDeployment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a workload.
     * </p>
     *
     * @param getWorkloadRequest
     * @return Result of the GetWorkload operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.GetWorkload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/GetWorkload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetWorkloadResponse getWorkload(GetWorkloadRequest getWorkloadRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetWorkloadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkloadRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkload");

            return clientHandler.execute(new ClientExecutionParams<GetWorkloadRequest, GetWorkloadResponse>()
                    .withOperationName("GetWorkload").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getWorkloadRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetWorkloadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for a given workload and deployment pattern, including the available specifications. You can use
     * the <a
     * href="https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListWorkloads.html">ListWorkloads</a>
     * operation to discover the available workload names and the <a
     * href="https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListWorkloadDeploymentPatterns.html"
     * >ListWorkloadDeploymentPatterns</a> operation to discover the available deployment pattern names of a given
     * workload.
     * </p>
     *
     * @param getWorkloadDeploymentPatternRequest
     * @return Result of the GetWorkloadDeploymentPattern operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.GetWorkloadDeploymentPattern
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/GetWorkloadDeploymentPattern"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetWorkloadDeploymentPatternResponse getWorkloadDeploymentPattern(
            GetWorkloadDeploymentPatternRequest getWorkloadDeploymentPatternRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWorkloadDeploymentPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetWorkloadDeploymentPatternResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkloadDeploymentPatternRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkloadDeploymentPatternRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkloadDeploymentPattern");

            return clientHandler
                    .execute(new ClientExecutionParams<GetWorkloadDeploymentPatternRequest, GetWorkloadDeploymentPatternResponse>()
                            .withOperationName("GetWorkloadDeploymentPattern").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getWorkloadDeploymentPatternRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetWorkloadDeploymentPatternRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the events of a deployment.
     * </p>
     *
     * @param listDeploymentEventsRequest
     * @return Result of the ListDeploymentEvents operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.ListDeploymentEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/ListDeploymentEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeploymentEventsResponse listDeploymentEvents(ListDeploymentEventsRequest listDeploymentEventsRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeploymentEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDeploymentEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeploymentEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeploymentEvents");

            return clientHandler.execute(new ClientExecutionParams<ListDeploymentEventsRequest, ListDeploymentEventsResponse>()
                    .withOperationName("ListDeploymentEvents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDeploymentEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeploymentEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the deployments that have been created.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return Result of the ListDeployments operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.ListDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/ListDeployments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDeploymentsResponse listDeployments(ListDeploymentsRequest listDeploymentsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDeploymentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeploymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeployments");

            return clientHandler.execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                    .withOperationName("ListDeployments").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDeploymentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags associated with a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the workload deployment patterns for a given workload name. You can use the <a
     * href="https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListWorkloads.html">ListWorkloads</a>
     * operation to discover the available workload names.
     * </p>
     *
     * @param listWorkloadDeploymentPatternsRequest
     * @return Result of the ListWorkloadDeploymentPatterns operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.ListWorkloadDeploymentPatterns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/ListWorkloadDeploymentPatterns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkloadDeploymentPatternsResponse listWorkloadDeploymentPatterns(
            ListWorkloadDeploymentPatternsRequest listWorkloadDeploymentPatternsRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkloadDeploymentPatternsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListWorkloadDeploymentPatternsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkloadDeploymentPatternsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listWorkloadDeploymentPatternsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkloadDeploymentPatterns");

            return clientHandler
                    .execute(new ClientExecutionParams<ListWorkloadDeploymentPatternsRequest, ListWorkloadDeploymentPatternsResponse>()
                            .withOperationName("ListWorkloadDeploymentPatterns").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listWorkloadDeploymentPatternsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListWorkloadDeploymentPatternsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the available workload names. You can use the <a
     * href="https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListWorkloadDeploymentPatterns.html"
     * >ListWorkloadDeploymentPatterns</a> operation to discover the available deployment patterns for a given workload.
     * </p>
     *
     * @param listWorkloadsRequest
     * @return Result of the ListWorkloads operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.ListWorkloads
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/ListWorkloads" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListWorkloadsResponse listWorkloads(ListWorkloadsRequest listWorkloadsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkloadsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkloadsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkloadsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkloadsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkloads");

            return clientHandler.execute(new ClientExecutionParams<ListWorkloadsRequest, ListWorkloadsResponse>()
                    .withOperationName("ListWorkloads").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listWorkloadsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkloadsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the given resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the given resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Retry your request, but if the problem persists, contact us with details
     *         by posting a question on <a href="https://repost.aws/">re:Post</a>.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified workload or deployment resource can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LaunchWizardException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LaunchWizardClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/launch-wizard-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, LaunchWizardException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Launch Wizard");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        LaunchWizardServiceClientConfigurationBuilder serviceConfigBuilder = new LaunchWizardServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LaunchWizardException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitException")
                                .exceptionBuilderSupplier(ResourceLimitException::builder).httpStatusCode(400).build());
    }

    @Override
    public final LaunchWizardServiceClientConfiguration serviceClientConfiguration() {
        return new LaunchWizardServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
