/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.lexmodelbuilding.transform.StatementMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A collection of messages that convey information to the user. At runtime, Amazon Lex selects the message to convey.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Statement implements StructuredPojo, ToCopyableBuilder<Statement.Builder, Statement> {
    private final List<Message> messages;

    private final String responseCard;

    private Statement(BuilderImpl builder) {
        this.messages = builder.messages;
        this.responseCard = builder.responseCard;
    }

    /**
     * <p>
     * A collection of message objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A collection of message objects.
     */
    public List<Message> messages() {
        return messages;
    }

    /**
     * <p>
     * At runtime, if the client is using the <a
     * href="http://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html">PostText</a> API, Amazon Lex includes
     * the response card in the response. It substitutes all of the session attributes and slot values for placeholders
     * in the response card.
     * </p>
     * 
     * @return At runtime, if the client is using the <a
     *         href="http://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html">PostText</a> API, Amazon Lex
     *         includes the response card in the response. It substitutes all of the session attributes and slot values
     *         for placeholders in the response card.
     */
    public String responseCard() {
        return responseCard;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(messages());
        hashCode = 31 * hashCode + Objects.hashCode(responseCard());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Statement)) {
            return false;
        }
        Statement other = (Statement) obj;
        return Objects.equals(messages(), other.messages()) && Objects.equals(responseCard(), other.responseCard());
    }

    @Override
    public String toString() {
        return ToString.builder("Statement").add("Messages", messages()).add("ResponseCard", responseCard()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "messages":
            return Optional.of(clazz.cast(messages()));
        case "responseCard":
            return Optional.of(clazz.cast(responseCard()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        StatementMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Statement> {
        /**
         * <p>
         * A collection of message objects.
         * </p>
         * 
         * @param messages
         *        A collection of message objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(Collection<Message> messages);

        /**
         * <p>
         * A collection of message objects.
         * </p>
         * 
         * @param messages
         *        A collection of message objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(Message... messages);

        /**
         * <p>
         * A collection of message objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Message>.Builder} avoiding the need to
         * create one manually via {@link List<Message>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Message>.Builder#build()} is called immediately and its
         * result is passed to {@link #messages(List<Message>)}.
         * 
         * @param messages
         *        a consumer that will call methods on {@link List<Message>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messages(List<Message>)
         */
        Builder messages(Consumer<Message.Builder>... messages);

        /**
         * <p>
         * At runtime, if the client is using the <a
         * href="http://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html">PostText</a> API, Amazon Lex
         * includes the response card in the response. It substitutes all of the session attributes and slot values for
         * placeholders in the response card.
         * </p>
         * 
         * @param responseCard
         *        At runtime, if the client is using the <a
         *        href="http://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html">PostText</a> API, Amazon Lex
         *        includes the response card in the response. It substitutes all of the session attributes and slot
         *        values for placeholders in the response card.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCard(String responseCard);
    }

    static final class BuilderImpl implements Builder {
        private List<Message> messages;

        private String responseCard;

        private BuilderImpl() {
        }

        private BuilderImpl(Statement model) {
            messages(model.messages);
            responseCard(model.responseCard);
        }

        public final Collection<Message.Builder> getMessages() {
            return messages != null ? messages.stream().map(Message::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder messages(Collection<Message> messages) {
            this.messages = MessageListCopier.copy(messages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(Message... messages) {
            messages(Arrays.asList(messages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(Consumer<Message.Builder>... messages) {
            messages(Stream.of(messages).map(c -> Message.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setMessages(Collection<Message.BuilderImpl> messages) {
            this.messages = MessageListCopier.copyFromBuilder(messages);
        }

        public final String getResponseCard() {
            return responseCard;
        }

        @Override
        public final Builder responseCard(String responseCard) {
            this.responseCard = responseCard;
            return this;
        }

        public final void setResponseCard(String responseCard) {
            this.responseCard = responseCard;
        }

        @Override
        public Statement build() {
            return new Statement(this);
        }
    }
}
