/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.lexmodelbuilding.transform.BuiltinIntentMetadataMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides metadata for a built-in intent.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BuiltinIntentMetadata implements StructuredPojo,
        ToCopyableBuilder<BuiltinIntentMetadata.Builder, BuiltinIntentMetadata> {
    private final String signature;

    private final List<String> supportedLocales;

    private BuiltinIntentMetadata(BuilderImpl builder) {
        this.signature = builder.signature;
        this.supportedLocales = builder.supportedLocales;
    }

    /**
     * <p>
     * A unique identifier for the built-in intent. To find the signature for an intent, see <a href=
     * "https://developer.amazon.com/public/solutions/alexa/alexa-skills-kit/docs/built-in-intent-ref/standard-intents"
     * >Standard Built-in Intents</a> in the <i>Alexa Skills Kit</i>.
     * </p>
     * 
     * @return A unique identifier for the built-in intent. To find the signature for an intent, see <a href=
     *         "https://developer.amazon.com/public/solutions/alexa/alexa-skills-kit/docs/built-in-intent-ref/standard-intents"
     *         >Standard Built-in Intents</a> in the <i>Alexa Skills Kit</i>.
     */
    public String signature() {
        return signature;
    }

    /**
     * <p>
     * A list of identifiers for the locales that the intent supports.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of identifiers for the locales that the intent supports.
     */
    public List<Locale> supportedLocales() {
        return TypeConverter.convert(supportedLocales, Locale::fromValue);
    }

    /**
     * <p>
     * A list of identifiers for the locales that the intent supports.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of identifiers for the locales that the intent supports.
     */
    public List<String> supportedLocalesAsStrings() {
        return supportedLocales;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(signature());
        hashCode = 31 * hashCode + Objects.hashCode(supportedLocalesAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BuiltinIntentMetadata)) {
            return false;
        }
        BuiltinIntentMetadata other = (BuiltinIntentMetadata) obj;
        return Objects.equals(signature(), other.signature())
                && Objects.equals(supportedLocalesAsStrings(), other.supportedLocalesAsStrings());
    }

    @Override
    public String toString() {
        return ToString.builder("BuiltinIntentMetadata").add("Signature", signature())
                .add("SupportedLocales", supportedLocalesAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "signature":
            return Optional.ofNullable(clazz.cast(signature()));
        case "supportedLocales":
            return Optional.ofNullable(clazz.cast(supportedLocalesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        BuiltinIntentMetadataMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, BuiltinIntentMetadata> {
        /**
         * <p>
         * A unique identifier for the built-in intent. To find the signature for an intent, see <a href=
         * "https://developer.amazon.com/public/solutions/alexa/alexa-skills-kit/docs/built-in-intent-ref/standard-intents"
         * >Standard Built-in Intents</a> in the <i>Alexa Skills Kit</i>.
         * </p>
         * 
         * @param signature
         *        A unique identifier for the built-in intent. To find the signature for an intent, see <a href=
         *        "https://developer.amazon.com/public/solutions/alexa/alexa-skills-kit/docs/built-in-intent-ref/standard-intents"
         *        >Standard Built-in Intents</a> in the <i>Alexa Skills Kit</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signature(String signature);

        /**
         * <p>
         * A list of identifiers for the locales that the intent supports.
         * </p>
         * 
         * @param supportedLocales
         *        A list of identifiers for the locales that the intent supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLocalesWithStrings(Collection<String> supportedLocales);

        /**
         * <p>
         * A list of identifiers for the locales that the intent supports.
         * </p>
         * 
         * @param supportedLocales
         *        A list of identifiers for the locales that the intent supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLocalesWithStrings(String... supportedLocales);

        /**
         * <p>
         * A list of identifiers for the locales that the intent supports.
         * </p>
         * 
         * @param supportedLocales
         *        A list of identifiers for the locales that the intent supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLocales(Collection<Locale> supportedLocales);

        /**
         * <p>
         * A list of identifiers for the locales that the intent supports.
         * </p>
         * 
         * @param supportedLocales
         *        A list of identifiers for the locales that the intent supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLocales(Locale... supportedLocales);
    }

    static final class BuilderImpl implements Builder {
        private String signature;

        private List<String> supportedLocales = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BuiltinIntentMetadata model) {
            signature(model.signature);
            supportedLocalesWithStrings(model.supportedLocales);
        }

        public final String getSignature() {
            return signature;
        }

        @Override
        public final Builder signature(String signature) {
            this.signature = signature;
            return this;
        }

        public final void setSignature(String signature) {
            this.signature = signature;
        }

        public final Collection<String> getSupportedLocales() {
            return supportedLocales;
        }

        @Override
        public final Builder supportedLocalesWithStrings(Collection<String> supportedLocales) {
            this.supportedLocales = LocaleListCopier.copy(supportedLocales);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedLocalesWithStrings(String... supportedLocales) {
            supportedLocalesWithStrings(Arrays.asList(supportedLocales));
            return this;
        }

        @Override
        public final Builder supportedLocales(Collection<Locale> supportedLocales) {
            this.supportedLocales = LocaleListCopier.copyEnumToString(supportedLocales);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedLocales(Locale... supportedLocales) {
            supportedLocales(Arrays.asList(supportedLocales));
            return this;
        }

        public final void setSupportedLocales(Collection<String> supportedLocales) {
            this.supportedLocales = LocaleListCopier.copy(supportedLocales);
        }

        @Override
        public BuiltinIntentMetadata build() {
            return new BuiltinIntentMetadata(this);
        }
    }
}
