/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutBotRequest extends LexModelBuildingRequest implements
        ToCopyableBuilder<PutBotRequest.Builder, PutBotRequest> {
    private final String name;

    private final String description;

    private final List<Intent> intents;

    private final Prompt clarificationPrompt;

    private final Statement abortStatement;

    private final Integer idleSessionTTLInSeconds;

    private final String voiceId;

    private final String checksum;

    private final String processBehavior;

    private final String locale;

    private final Boolean childDirected;

    private PutBotRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.description = builder.description;
        this.intents = builder.intents;
        this.clarificationPrompt = builder.clarificationPrompt;
        this.abortStatement = builder.abortStatement;
        this.idleSessionTTLInSeconds = builder.idleSessionTTLInSeconds;
        this.voiceId = builder.voiceId;
        this.checksum = builder.checksum;
        this.processBehavior = builder.processBehavior;
        this.locale = builder.locale;
        this.childDirected = builder.childDirected;
    }

    /**
     * <p>
     * The name of the bot. The name is <i>not</i> case sensitive.
     * </p>
     * 
     * @return The name of the bot. The name is <i>not</i> case sensitive.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A description of the bot.
     * </p>
     * 
     * @return A description of the bot.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For example, a
     * pizza ordering bot might support an OrderPizza intent. For more information, see <a>how-it-works</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For
     *         example, a pizza ordering bot might support an OrderPizza intent. For more information, see
     *         <a>how-it-works</a>.
     */
    public List<Intent> intents() {
        return intents;
    }

    /**
     * <p>
     * When Amazon Lex doesn't understand the user's intent, it uses this message to get clarification. To specify how
     * many times Amazon Lex should repeate the clarification prompt, use the <code>maxAttempts</code> field. If Amazon
     * Lex still doesn't understand, it sends the message in the <code>abortStatement</code> field.
     * </p>
     * <p>
     * When you create a clarification prompt, make sure that it suggests the correct response from the user. for
     * example, for a bot that orders pizza and drinks, you might create this clarification prompt:
     * "What would you like to do? You can say 'Order a pizza' or 'Order a drink.'"
     * </p>
     * 
     * @return When Amazon Lex doesn't understand the user's intent, it uses this message to get clarification. To
     *         specify how many times Amazon Lex should repeate the clarification prompt, use the
     *         <code>maxAttempts</code> field. If Amazon Lex still doesn't understand, it sends the message in the
     *         <code>abortStatement</code> field. </p>
     *         <p>
     *         When you create a clarification prompt, make sure that it suggests the correct response from the user.
     *         for example, for a bot that orders pizza and drinks, you might create this clarification prompt:
     *         "What would you like to do? You can say 'Order a pizza' or 'Order a drink.'"
     */
    public Prompt clarificationPrompt() {
        return clarificationPrompt;
    }

    /**
     * <p>
     * When Amazon Lex can't understand the user's input in context, it tries to elicit the information a few times.
     * After that, Amazon Lex sends the message defined in <code>abortStatement</code> to the user, and then aborts the
     * conversation. To set the number of retries, use the <code>valueElicitationPrompt</code> field for the slot type.
     * </p>
     * <p>
     * For example, in a pizza ordering bot, Amazon Lex might ask a user "What type of crust would you like?" If the
     * user's response is not one of the expected responses (for example, "thin crust, "deep dish," etc.), Amazon Lex
     * tries to elicit a correct response a few more times.
     * </p>
     * <p>
     * For example, in a pizza ordering application, <code>OrderPizza</code> might be one of the intents. This intent
     * might require the <code>CrustType</code> slot. You specify the <code>valueElicitationPrompt</code> field when you
     * create the <code>CrustType</code> slot.
     * </p>
     * 
     * @return When Amazon Lex can't understand the user's input in context, it tries to elicit the information a few
     *         times. After that, Amazon Lex sends the message defined in <code>abortStatement</code> to the user, and
     *         then aborts the conversation. To set the number of retries, use the <code>valueElicitationPrompt</code>
     *         field for the slot type. </p>
     *         <p>
     *         For example, in a pizza ordering bot, Amazon Lex might ask a user "What type of crust would you like?" If
     *         the user's response is not one of the expected responses (for example, "thin crust, "deep dish," etc.),
     *         Amazon Lex tries to elicit a correct response a few more times.
     *         </p>
     *         <p>
     *         For example, in a pizza ordering application, <code>OrderPizza</code> might be one of the intents. This
     *         intent might require the <code>CrustType</code> slot. You specify the <code>valueElicitationPrompt</code>
     *         field when you create the <code>CrustType</code> slot.
     */
    public Statement abortStatement() {
        return abortStatement;
    }

    /**
     * <p>
     * The maximum time in seconds that Amazon Lex retains the data gathered in a conversation.
     * </p>
     * <p>
     * A user interaction session remains active for the amount of time specified. If no conversation occurs during this
     * time, the session expires and Amazon Lex deletes any data provided before the timeout.
     * </p>
     * <p>
     * For example, suppose that a user chooses the OrderPizza intent, but gets sidetracked halfway through placing an
     * order. If the user doesn't complete the order within the specified time, Amazon Lex discards the slot information
     * that it gathered, and the user must start over.
     * </p>
     * <p>
     * If you don't include the <code>idleSessionTTLInSeconds</code> element in a <code>PutBot</code> operation request,
     * Amazon Lex uses the default value. This is also true if the request replaces an existing bot.
     * </p>
     * <p>
     * The default is 300 seconds (5 minutes).
     * </p>
     * 
     * @return The maximum time in seconds that Amazon Lex retains the data gathered in a conversation.</p>
     *         <p>
     *         A user interaction session remains active for the amount of time specified. If no conversation occurs
     *         during this time, the session expires and Amazon Lex deletes any data provided before the timeout.
     *         </p>
     *         <p>
     *         For example, suppose that a user chooses the OrderPizza intent, but gets sidetracked halfway through
     *         placing an order. If the user doesn't complete the order within the specified time, Amazon Lex discards
     *         the slot information that it gathered, and the user must start over.
     *         </p>
     *         <p>
     *         If you don't include the <code>idleSessionTTLInSeconds</code> element in a <code>PutBot</code> operation
     *         request, Amazon Lex uses the default value. This is also true if the request replaces an existing bot.
     *         </p>
     *         <p>
     *         The default is 300 seconds (5 minutes).
     */
    public Integer idleSessionTTLInSeconds() {
        return idleSessionTTLInSeconds;
    }

    /**
     * <p>
     * The Amazon Polly voice ID that you want Amazon Lex to use for voice interactions with the user. The locale
     * configured for the voice must match the locale of the bot. For more information, see <a
     * href="http://docs.aws.amazon.com/polly/latest/dg/voicelist.html">Available Voices</a> in the <i>Amazon Polly
     * Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Polly voice ID that you want Amazon Lex to use for voice interactions with the user. The
     *         locale configured for the voice must match the locale of the bot. For more information, see <a
     *         href="http://docs.aws.amazon.com/polly/latest/dg/voicelist.html">Available Voices</a> in the <i>Amazon
     *         Polly Developer Guide</i>.
     */
    public String voiceId() {
        return voiceId;
    }

    /**
     * <p>
     * Identifies a specific revision of the <code>$LATEST</code> version.
     * </p>
     * <p>
     * When you create a new bot, leave the <code>checksum</code> field blank. If you specify a checksum you get a
     * <code>BadRequestException</code> exception.
     * </p>
     * <p>
     * When you want to update a bot, set the <code>checksum</code> field to the checksum of the most recent revision of
     * the <code>$LATEST</code> version. If you don't specify the <code> checksum</code> field, or if the checksum does
     * not match the <code>$LATEST</code> version, you get a <code>PreconditionFailedException</code> exception.
     * </p>
     * 
     * @return Identifies a specific revision of the <code>$LATEST</code> version.</p>
     *         <p>
     *         When you create a new bot, leave the <code>checksum</code> field blank. If you specify a checksum you get
     *         a <code>BadRequestException</code> exception.
     *         </p>
     *         <p>
     *         When you want to update a bot, set the <code>checksum</code> field to the checksum of the most recent
     *         revision of the <code>$LATEST</code> version. If you don't specify the <code> checksum</code> field, or
     *         if the checksum does not match the <code>$LATEST</code> version, you get a
     *         <code>PreconditionFailedException</code> exception.
     */
    public String checksum() {
        return checksum;
    }

    /**
     * <p>
     * If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot so that it
     * can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but doesn't build it.
     * </p>
     * <p>
     * If you don't specify this value, the default value is <code>Save</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #processBehavior}
     * will return {@link ProcessBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #processBehaviorAsString}.
     * </p>
     * 
     * @return If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot so
     *         that it can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but doesn't
     *         build it. </p>
     *         <p>
     *         If you don't specify this value, the default value is <code>Save</code>.
     * @see ProcessBehavior
     */
    public ProcessBehavior processBehavior() {
        return ProcessBehavior.fromValue(processBehavior);
    }

    /**
     * <p>
     * If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot so that it
     * can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but doesn't build it.
     * </p>
     * <p>
     * If you don't specify this value, the default value is <code>Save</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #processBehavior}
     * will return {@link ProcessBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #processBehaviorAsString}.
     * </p>
     * 
     * @return If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot so
     *         that it can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but doesn't
     *         build it. </p>
     *         <p>
     *         If you don't specify this value, the default value is <code>Save</code>.
     * @see ProcessBehavior
     */
    public String processBehaviorAsString() {
        return processBehavior;
    }

    /**
     * <p>
     * Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale of the
     * bot.
     * </p>
     * <p>
     * The default is <code>en-US</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #locale} will
     * return {@link Locale#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #localeAsString}.
     * </p>
     * 
     * @return Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale of
     *         the bot. </p>
     *         <p>
     *         The default is <code>en-US</code>.
     * @see Locale
     */
    public Locale locale() {
        return Locale.fromValue(locale);
    }

    /**
     * <p>
     * Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale of the
     * bot.
     * </p>
     * <p>
     * The default is <code>en-US</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #locale} will
     * return {@link Locale#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #localeAsString}.
     * </p>
     * 
     * @return Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale of
     *         the bot. </p>
     *         <p>
     *         The default is <code>en-US</code>.
     * @see Locale
     */
    public String localeAsString() {
        return locale;
    }

    /**
     * <p>
     * For each Amazon Lex bot created with the Amazon Lex Model Building Service, you must specify whether your use of
     * Amazon Lex is related to a website, program, or other application that is directed or targeted, in whole or in
     * part, to children under age 13 and subject to the Children's Online Privacy Protection Act (COPPA) by specifying
     * <code>true</code> or <code>false</code> in the <code>childDirected</code> field. By specifying <code>true</code>
     * in the <code>childDirected</code> field, you confirm that your use of Amazon Lex <b>is</b> related to a website,
     * program, or other application that is directed or targeted, in whole or in part, to children under age 13 and
     * subject to COPPA. By specifying <code>false</code> in the <code>childDirected</code> field, you confirm that your
     * use of Amazon Lex <b>is not</b> related to a website, program, or other application that is directed or targeted,
     * in whole or in part, to children under age 13 and subject to COPPA. You may not specify a default value for the
     * <code>childDirected</code> field that does not accurately reflect whether your use of Amazon Lex is related to a
     * website, program, or other application that is directed or targeted, in whole or in part, to children under age
     * 13 and subject to COPPA.
     * </p>
     * <p>
     * If your use of Amazon Lex relates to a website, program, or other application that is directed in whole or in
     * part, to children under age 13, you must obtain any required verifiable parental consent under COPPA. For
     * information regarding the use of Amazon Lex in connection with websites, programs, or other applications that are
     * directed or targeted, in whole or in part, to children under age 13, see the <a
     * href="https://aws.amazon.com/lex/faqs#data-security">Amazon Lex FAQ.</a>
     * </p>
     * 
     * @return For each Amazon Lex bot created with the Amazon Lex Model Building Service, you must specify whether your
     *         use of Amazon Lex is related to a website, program, or other application that is directed or targeted, in
     *         whole or in part, to children under age 13 and subject to the Children's Online Privacy Protection Act
     *         (COPPA) by specifying <code>true</code> or <code>false</code> in the <code>childDirected</code> field. By
     *         specifying <code>true</code> in the <code>childDirected</code> field, you confirm that your use of Amazon
     *         Lex <b>is</b> related to a website, program, or other application that is directed or targeted, in whole
     *         or in part, to children under age 13 and subject to COPPA. By specifying <code>false</code> in the
     *         <code>childDirected</code> field, you confirm that your use of Amazon Lex <b>is not</b> related to a
     *         website, program, or other application that is directed or targeted, in whole or in part, to children
     *         under age 13 and subject to COPPA. You may not specify a default value for the <code>childDirected</code>
     *         field that does not accurately reflect whether your use of Amazon Lex is related to a website, program,
     *         or other application that is directed or targeted, in whole or in part, to children under age 13 and
     *         subject to COPPA.</p>
     *         <p>
     *         If your use of Amazon Lex relates to a website, program, or other application that is directed in whole
     *         or in part, to children under age 13, you must obtain any required verifiable parental consent under
     *         COPPA. For information regarding the use of Amazon Lex in connection with websites, programs, or other
     *         applications that are directed or targeted, in whole or in part, to children under age 13, see the <a
     *         href="https://aws.amazon.com/lex/faqs#data-security">Amazon Lex FAQ.</a>
     */
    public Boolean childDirected() {
        return childDirected;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(intents());
        hashCode = 31 * hashCode + Objects.hashCode(clarificationPrompt());
        hashCode = 31 * hashCode + Objects.hashCode(abortStatement());
        hashCode = 31 * hashCode + Objects.hashCode(idleSessionTTLInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(voiceId());
        hashCode = 31 * hashCode + Objects.hashCode(checksum());
        hashCode = 31 * hashCode + Objects.hashCode(processBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(localeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(childDirected());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutBotRequest)) {
            return false;
        }
        PutBotRequest other = (PutBotRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(intents(), other.intents())
                && Objects.equals(clarificationPrompt(), other.clarificationPrompt())
                && Objects.equals(abortStatement(), other.abortStatement())
                && Objects.equals(idleSessionTTLInSeconds(), other.idleSessionTTLInSeconds())
                && Objects.equals(voiceId(), other.voiceId()) && Objects.equals(checksum(), other.checksum())
                && Objects.equals(processBehaviorAsString(), other.processBehaviorAsString())
                && Objects.equals(localeAsString(), other.localeAsString())
                && Objects.equals(childDirected(), other.childDirected());
    }

    @Override
    public String toString() {
        return ToString.builder("PutBotRequest").add("Name", name()).add("Description", description()).add("Intents", intents())
                .add("ClarificationPrompt", clarificationPrompt()).add("AbortStatement", abortStatement())
                .add("IdleSessionTTLInSeconds", idleSessionTTLInSeconds()).add("VoiceId", voiceId()).add("Checksum", checksum())
                .add("ProcessBehavior", processBehaviorAsString()).add("Locale", localeAsString())
                .add("ChildDirected", childDirected()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "intents":
            return Optional.ofNullable(clazz.cast(intents()));
        case "clarificationPrompt":
            return Optional.ofNullable(clazz.cast(clarificationPrompt()));
        case "abortStatement":
            return Optional.ofNullable(clazz.cast(abortStatement()));
        case "idleSessionTTLInSeconds":
            return Optional.ofNullable(clazz.cast(idleSessionTTLInSeconds()));
        case "voiceId":
            return Optional.ofNullable(clazz.cast(voiceId()));
        case "checksum":
            return Optional.ofNullable(clazz.cast(checksum()));
        case "processBehavior":
            return Optional.ofNullable(clazz.cast(processBehaviorAsString()));
        case "locale":
            return Optional.ofNullable(clazz.cast(localeAsString()));
        case "childDirected":
            return Optional.ofNullable(clazz.cast(childDirected()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends LexModelBuildingRequest.Builder, CopyableBuilder<Builder, PutBotRequest> {
        /**
         * <p>
         * The name of the bot. The name is <i>not</i> case sensitive.
         * </p>
         * 
         * @param name
         *        The name of the bot. The name is <i>not</i> case sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the bot.
         * </p>
         * 
         * @param description
         *        A description of the bot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For
         * example, a pizza ordering bot might support an OrderPizza intent. For more information, see
         * <a>how-it-works</a>.
         * </p>
         * 
         * @param intents
         *        An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For
         *        example, a pizza ordering bot might support an OrderPizza intent. For more information, see
         *        <a>how-it-works</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intents(Collection<Intent> intents);

        /**
         * <p>
         * An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For
         * example, a pizza ordering bot might support an OrderPizza intent. For more information, see
         * <a>how-it-works</a>.
         * </p>
         * 
         * @param intents
         *        An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For
         *        example, a pizza ordering bot might support an OrderPizza intent. For more information, see
         *        <a>how-it-works</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intents(Intent... intents);

        /**
         * <p>
         * An array of <code>Intent</code> objects. Each intent represents a command that a user can express. For
         * example, a pizza ordering bot might support an OrderPizza intent. For more information, see
         * <a>how-it-works</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Intent>.Builder} avoiding the need to
         * create one manually via {@link List<Intent>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Intent>.Builder#build()} is called immediately and its
         * result is passed to {@link #intents(List<Intent>)}.
         * 
         * @param intents
         *        a consumer that will call methods on {@link List<Intent>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #intents(List<Intent>)
         */
        Builder intents(Consumer<Intent.Builder>... intents);

        /**
         * <p>
         * When Amazon Lex doesn't understand the user's intent, it uses this message to get clarification. To specify
         * how many times Amazon Lex should repeate the clarification prompt, use the <code>maxAttempts</code> field. If
         * Amazon Lex still doesn't understand, it sends the message in the <code>abortStatement</code> field.
         * </p>
         * <p>
         * When you create a clarification prompt, make sure that it suggests the correct response from the user. for
         * example, for a bot that orders pizza and drinks, you might create this clarification prompt:
         * "What would you like to do? You can say 'Order a pizza' or 'Order a drink.'"
         * </p>
         * 
         * @param clarificationPrompt
         *        When Amazon Lex doesn't understand the user's intent, it uses this message to get clarification. To
         *        specify how many times Amazon Lex should repeate the clarification prompt, use the
         *        <code>maxAttempts</code> field. If Amazon Lex still doesn't understand, it sends the message in the
         *        <code>abortStatement</code> field. </p>
         *        <p>
         *        When you create a clarification prompt, make sure that it suggests the correct response from the user.
         *        for example, for a bot that orders pizza and drinks, you might create this clarification prompt:
         *        "What would you like to do? You can say 'Order a pizza' or 'Order a drink.'"
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clarificationPrompt(Prompt clarificationPrompt);

        /**
         * <p>
         * When Amazon Lex doesn't understand the user's intent, it uses this message to get clarification. To specify
         * how many times Amazon Lex should repeate the clarification prompt, use the <code>maxAttempts</code> field. If
         * Amazon Lex still doesn't understand, it sends the message in the <code>abortStatement</code> field.
         * </p>
         * <p>
         * When you create a clarification prompt, make sure that it suggests the correct response from the user. for
         * example, for a bot that orders pizza and drinks, you might create this clarification prompt:
         * "What would you like to do? You can say 'Order a pizza' or 'Order a drink.'"
         * </p>
         * This is a convenience that creates an instance of the {@link Prompt.Builder} avoiding the need to create one
         * manually via {@link Prompt#builder()}.
         *
         * When the {@link Consumer} completes, {@link Prompt.Builder#build()} is called immediately and its result is
         * passed to {@link #clarificationPrompt(Prompt)}.
         * 
         * @param clarificationPrompt
         *        a consumer that will call methods on {@link Prompt.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clarificationPrompt(Prompt)
         */
        default Builder clarificationPrompt(Consumer<Prompt.Builder> clarificationPrompt) {
            return clarificationPrompt(Prompt.builder().applyMutation(clarificationPrompt).build());
        }

        /**
         * <p>
         * When Amazon Lex can't understand the user's input in context, it tries to elicit the information a few times.
         * After that, Amazon Lex sends the message defined in <code>abortStatement</code> to the user, and then aborts
         * the conversation. To set the number of retries, use the <code>valueElicitationPrompt</code> field for the
         * slot type.
         * </p>
         * <p>
         * For example, in a pizza ordering bot, Amazon Lex might ask a user "What type of crust would you like?" If the
         * user's response is not one of the expected responses (for example, "thin crust, "deep dish," etc.), Amazon
         * Lex tries to elicit a correct response a few more times.
         * </p>
         * <p>
         * For example, in a pizza ordering application, <code>OrderPizza</code> might be one of the intents. This
         * intent might require the <code>CrustType</code> slot. You specify the <code>valueElicitationPrompt</code>
         * field when you create the <code>CrustType</code> slot.
         * </p>
         * 
         * @param abortStatement
         *        When Amazon Lex can't understand the user's input in context, it tries to elicit the information a few
         *        times. After that, Amazon Lex sends the message defined in <code>abortStatement</code> to the user,
         *        and then aborts the conversation. To set the number of retries, use the
         *        <code>valueElicitationPrompt</code> field for the slot type. </p>
         *        <p>
         *        For example, in a pizza ordering bot, Amazon Lex might ask a user "What type of crust would you like?"
         *        If the user's response is not one of the expected responses (for example, "thin crust, "deep dish,"
         *        etc.), Amazon Lex tries to elicit a correct response a few more times.
         *        </p>
         *        <p>
         *        For example, in a pizza ordering application, <code>OrderPizza</code> might be one of the intents.
         *        This intent might require the <code>CrustType</code> slot. You specify the
         *        <code>valueElicitationPrompt</code> field when you create the <code>CrustType</code> slot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder abortStatement(Statement abortStatement);

        /**
         * <p>
         * When Amazon Lex can't understand the user's input in context, it tries to elicit the information a few times.
         * After that, Amazon Lex sends the message defined in <code>abortStatement</code> to the user, and then aborts
         * the conversation. To set the number of retries, use the <code>valueElicitationPrompt</code> field for the
         * slot type.
         * </p>
         * <p>
         * For example, in a pizza ordering bot, Amazon Lex might ask a user "What type of crust would you like?" If the
         * user's response is not one of the expected responses (for example, "thin crust, "deep dish," etc.), Amazon
         * Lex tries to elicit a correct response a few more times.
         * </p>
         * <p>
         * For example, in a pizza ordering application, <code>OrderPizza</code> might be one of the intents. This
         * intent might require the <code>CrustType</code> slot. You specify the <code>valueElicitationPrompt</code>
         * field when you create the <code>CrustType</code> slot.
         * </p>
         * This is a convenience that creates an instance of the {@link Statement.Builder} avoiding the need to create
         * one manually via {@link Statement#builder()}.
         *
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #abortStatement(Statement)}.
         * 
         * @param abortStatement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #abortStatement(Statement)
         */
        default Builder abortStatement(Consumer<Statement.Builder> abortStatement) {
            return abortStatement(Statement.builder().applyMutation(abortStatement).build());
        }

        /**
         * <p>
         * The maximum time in seconds that Amazon Lex retains the data gathered in a conversation.
         * </p>
         * <p>
         * A user interaction session remains active for the amount of time specified. If no conversation occurs during
         * this time, the session expires and Amazon Lex deletes any data provided before the timeout.
         * </p>
         * <p>
         * For example, suppose that a user chooses the OrderPizza intent, but gets sidetracked halfway through placing
         * an order. If the user doesn't complete the order within the specified time, Amazon Lex discards the slot
         * information that it gathered, and the user must start over.
         * </p>
         * <p>
         * If you don't include the <code>idleSessionTTLInSeconds</code> element in a <code>PutBot</code> operation
         * request, Amazon Lex uses the default value. This is also true if the request replaces an existing bot.
         * </p>
         * <p>
         * The default is 300 seconds (5 minutes).
         * </p>
         * 
         * @param idleSessionTTLInSeconds
         *        The maximum time in seconds that Amazon Lex retains the data gathered in a conversation.</p>
         *        <p>
         *        A user interaction session remains active for the amount of time specified. If no conversation occurs
         *        during this time, the session expires and Amazon Lex deletes any data provided before the timeout.
         *        </p>
         *        <p>
         *        For example, suppose that a user chooses the OrderPizza intent, but gets sidetracked halfway through
         *        placing an order. If the user doesn't complete the order within the specified time, Amazon Lex
         *        discards the slot information that it gathered, and the user must start over.
         *        </p>
         *        <p>
         *        If you don't include the <code>idleSessionTTLInSeconds</code> element in a <code>PutBot</code>
         *        operation request, Amazon Lex uses the default value. This is also true if the request replaces an
         *        existing bot.
         *        </p>
         *        <p>
         *        The default is 300 seconds (5 minutes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idleSessionTTLInSeconds(Integer idleSessionTTLInSeconds);

        /**
         * <p>
         * The Amazon Polly voice ID that you want Amazon Lex to use for voice interactions with the user. The locale
         * configured for the voice must match the locale of the bot. For more information, see <a
         * href="http://docs.aws.amazon.com/polly/latest/dg/voicelist.html">Available Voices</a> in the <i>Amazon Polly
         * Developer Guide</i>.
         * </p>
         * 
         * @param voiceId
         *        The Amazon Polly voice ID that you want Amazon Lex to use for voice interactions with the user. The
         *        locale configured for the voice must match the locale of the bot. For more information, see <a
         *        href="http://docs.aws.amazon.com/polly/latest/dg/voicelist.html">Available Voices</a> in the <i>Amazon
         *        Polly Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder voiceId(String voiceId);

        /**
         * <p>
         * Identifies a specific revision of the <code>$LATEST</code> version.
         * </p>
         * <p>
         * When you create a new bot, leave the <code>checksum</code> field blank. If you specify a checksum you get a
         * <code>BadRequestException</code> exception.
         * </p>
         * <p>
         * When you want to update a bot, set the <code>checksum</code> field to the checksum of the most recent
         * revision of the <code>$LATEST</code> version. If you don't specify the <code> checksum</code> field, or if
         * the checksum does not match the <code>$LATEST</code> version, you get a
         * <code>PreconditionFailedException</code> exception.
         * </p>
         * 
         * @param checksum
         *        Identifies a specific revision of the <code>$LATEST</code> version.</p>
         *        <p>
         *        When you create a new bot, leave the <code>checksum</code> field blank. If you specify a checksum you
         *        get a <code>BadRequestException</code> exception.
         *        </p>
         *        <p>
         *        When you want to update a bot, set the <code>checksum</code> field to the checksum of the most recent
         *        revision of the <code>$LATEST</code> version. If you don't specify the <code> checksum</code> field,
         *        or if the checksum does not match the <code>$LATEST</code> version, you get a
         *        <code>PreconditionFailedException</code> exception.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checksum(String checksum);

        /**
         * <p>
         * If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot so that
         * it can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but doesn't build it.
         * </p>
         * <p>
         * If you don't specify this value, the default value is <code>Save</code>.
         * </p>
         * 
         * @param processBehavior
         *        If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot
         *        so that it can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but
         *        doesn't build it. </p>
         *        <p>
         *        If you don't specify this value, the default value is <code>Save</code>.
         * @see ProcessBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProcessBehavior
         */
        Builder processBehavior(String processBehavior);

        /**
         * <p>
         * If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot so that
         * it can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but doesn't build it.
         * </p>
         * <p>
         * If you don't specify this value, the default value is <code>Save</code>.
         * </p>
         * 
         * @param processBehavior
         *        If you set the <code>processBehavior</code> element to <code>Build</code>, Amazon Lex builds the bot
         *        so that it can be run. If you set the element to <code>Save</code>Amazon Lex saves the bot, but
         *        doesn't build it. </p>
         *        <p>
         *        If you don't specify this value, the default value is <code>Save</code>.
         * @see ProcessBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProcessBehavior
         */
        Builder processBehavior(ProcessBehavior processBehavior);

        /**
         * <p>
         * Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale of the
         * bot.
         * </p>
         * <p>
         * The default is <code>en-US</code>.
         * </p>
         * 
         * @param locale
         *        Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale
         *        of the bot. </p>
         *        <p>
         *        The default is <code>en-US</code>.
         * @see Locale
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Locale
         */
        Builder locale(String locale);

        /**
         * <p>
         * Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale of the
         * bot.
         * </p>
         * <p>
         * The default is <code>en-US</code>.
         * </p>
         * 
         * @param locale
         *        Specifies the target locale for the bot. Any intent used in the bot must be compatible with the locale
         *        of the bot. </p>
         *        <p>
         *        The default is <code>en-US</code>.
         * @see Locale
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Locale
         */
        Builder locale(Locale locale);

        /**
         * <p>
         * For each Amazon Lex bot created with the Amazon Lex Model Building Service, you must specify whether your use
         * of Amazon Lex is related to a website, program, or other application that is directed or targeted, in whole
         * or in part, to children under age 13 and subject to the Children's Online Privacy Protection Act (COPPA) by
         * specifying <code>true</code> or <code>false</code> in the <code>childDirected</code> field. By specifying
         * <code>true</code> in the <code>childDirected</code> field, you confirm that your use of Amazon Lex <b>is</b>
         * related to a website, program, or other application that is directed or targeted, in whole or in part, to
         * children under age 13 and subject to COPPA. By specifying <code>false</code> in the
         * <code>childDirected</code> field, you confirm that your use of Amazon Lex <b>is not</b> related to a website,
         * program, or other application that is directed or targeted, in whole or in part, to children under age 13 and
         * subject to COPPA. You may not specify a default value for the <code>childDirected</code> field that does not
         * accurately reflect whether your use of Amazon Lex is related to a website, program, or other application that
         * is directed or targeted, in whole or in part, to children under age 13 and subject to COPPA.
         * </p>
         * <p>
         * If your use of Amazon Lex relates to a website, program, or other application that is directed in whole or in
         * part, to children under age 13, you must obtain any required verifiable parental consent under COPPA. For
         * information regarding the use of Amazon Lex in connection with websites, programs, or other applications that
         * are directed or targeted, in whole or in part, to children under age 13, see the <a
         * href="https://aws.amazon.com/lex/faqs#data-security">Amazon Lex FAQ.</a>
         * </p>
         * 
         * @param childDirected
         *        For each Amazon Lex bot created with the Amazon Lex Model Building Service, you must specify whether
         *        your use of Amazon Lex is related to a website, program, or other application that is directed or
         *        targeted, in whole or in part, to children under age 13 and subject to the Children's Online Privacy
         *        Protection Act (COPPA) by specifying <code>true</code> or <code>false</code> in the
         *        <code>childDirected</code> field. By specifying <code>true</code> in the <code>childDirected</code>
         *        field, you confirm that your use of Amazon Lex <b>is</b> related to a website, program, or other
         *        application that is directed or targeted, in whole or in part, to children under age 13 and subject to
         *        COPPA. By specifying <code>false</code> in the <code>childDirected</code> field, you confirm that your
         *        use of Amazon Lex <b>is not</b> related to a website, program, or other application that is directed
         *        or targeted, in whole or in part, to children under age 13 and subject to COPPA. You may not specify a
         *        default value for the <code>childDirected</code> field that does not accurately reflect whether your
         *        use of Amazon Lex is related to a website, program, or other application that is directed or targeted,
         *        in whole or in part, to children under age 13 and subject to COPPA.</p>
         *        <p>
         *        If your use of Amazon Lex relates to a website, program, or other application that is directed in
         *        whole or in part, to children under age 13, you must obtain any required verifiable parental consent
         *        under COPPA. For information regarding the use of Amazon Lex in connection with websites, programs, or
         *        other applications that are directed or targeted, in whole or in part, to children under age 13, see
         *        the <a href="https://aws.amazon.com/lex/faqs#data-security">Amazon Lex FAQ.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childDirected(Boolean childDirected);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LexModelBuildingRequest.BuilderImpl implements Builder {
        private String name;

        private String description;

        private List<Intent> intents = DefaultSdkAutoConstructList.getInstance();

        private Prompt clarificationPrompt;

        private Statement abortStatement;

        private Integer idleSessionTTLInSeconds;

        private String voiceId;

        private String checksum;

        private String processBehavior;

        private String locale;

        private Boolean childDirected;

        private BuilderImpl() {
        }

        private BuilderImpl(PutBotRequest model) {
            super(model);
            name(model.name);
            description(model.description);
            intents(model.intents);
            clarificationPrompt(model.clarificationPrompt);
            abortStatement(model.abortStatement);
            idleSessionTTLInSeconds(model.idleSessionTTLInSeconds);
            voiceId(model.voiceId);
            checksum(model.checksum);
            processBehavior(model.processBehavior);
            locale(model.locale);
            childDirected(model.childDirected);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Intent.Builder> getIntents() {
            return intents != null ? intents.stream().map(Intent::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder intents(Collection<Intent> intents) {
            this.intents = IntentListCopier.copy(intents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder intents(Intent... intents) {
            intents(Arrays.asList(intents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder intents(Consumer<Intent.Builder>... intents) {
            intents(Stream.of(intents).map(c -> Intent.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setIntents(Collection<Intent.BuilderImpl> intents) {
            this.intents = IntentListCopier.copyFromBuilder(intents);
        }

        public final Prompt.Builder getClarificationPrompt() {
            return clarificationPrompt != null ? clarificationPrompt.toBuilder() : null;
        }

        @Override
        public final Builder clarificationPrompt(Prompt clarificationPrompt) {
            this.clarificationPrompt = clarificationPrompt;
            return this;
        }

        public final void setClarificationPrompt(Prompt.BuilderImpl clarificationPrompt) {
            this.clarificationPrompt = clarificationPrompt != null ? clarificationPrompt.build() : null;
        }

        public final Statement.Builder getAbortStatement() {
            return abortStatement != null ? abortStatement.toBuilder() : null;
        }

        @Override
        public final Builder abortStatement(Statement abortStatement) {
            this.abortStatement = abortStatement;
            return this;
        }

        public final void setAbortStatement(Statement.BuilderImpl abortStatement) {
            this.abortStatement = abortStatement != null ? abortStatement.build() : null;
        }

        public final Integer getIdleSessionTTLInSeconds() {
            return idleSessionTTLInSeconds;
        }

        @Override
        public final Builder idleSessionTTLInSeconds(Integer idleSessionTTLInSeconds) {
            this.idleSessionTTLInSeconds = idleSessionTTLInSeconds;
            return this;
        }

        public final void setIdleSessionTTLInSeconds(Integer idleSessionTTLInSeconds) {
            this.idleSessionTTLInSeconds = idleSessionTTLInSeconds;
        }

        public final String getVoiceId() {
            return voiceId;
        }

        @Override
        public final Builder voiceId(String voiceId) {
            this.voiceId = voiceId;
            return this;
        }

        public final void setVoiceId(String voiceId) {
            this.voiceId = voiceId;
        }

        public final String getChecksum() {
            return checksum;
        }

        @Override
        public final Builder checksum(String checksum) {
            this.checksum = checksum;
            return this;
        }

        public final void setChecksum(String checksum) {
            this.checksum = checksum;
        }

        public final String getProcessBehavior() {
            return processBehavior;
        }

        @Override
        public final Builder processBehavior(String processBehavior) {
            this.processBehavior = processBehavior;
            return this;
        }

        @Override
        public final Builder processBehavior(ProcessBehavior processBehavior) {
            this.processBehavior(processBehavior.toString());
            return this;
        }

        public final void setProcessBehavior(String processBehavior) {
            this.processBehavior = processBehavior;
        }

        public final String getLocale() {
            return locale;
        }

        @Override
        public final Builder locale(String locale) {
            this.locale = locale;
            return this;
        }

        @Override
        public final Builder locale(Locale locale) {
            this.locale(locale.toString());
            return this;
        }

        public final void setLocale(String locale) {
            this.locale = locale;
        }

        public final Boolean getChildDirected() {
            return childDirected;
        }

        @Override
        public final Builder childDirected(Boolean childDirected) {
            this.childDirected = childDirected;
            return this;
        }

        public final void setChildDirected(Boolean childDirected) {
            this.childDirected = childDirected;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutBotRequest build() {
            return new PutBotRequest(this);
        }
    }
}
