/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.paginators;

import java.util.Iterator;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.NextPageFetcher;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.services.lexmodelbuilding.LexModelBuildingClient;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.lexmodelbuilding.LexModelBuildingClient#getSlotTypeVersionsIterable(software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsResponse} that can be used to
 * iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.lexmodelbuilding.paginators.GetSlotTypeVersionsPaginator responses = client.getSlotTypeVersionsIterable(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.lexmodelbuilding.paginators.GetSlotTypeVersionsPaginator responses = client
 *             .getSlotTypeVersionsIterable(request);
 *     for (software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.lexmodelbuilding.paginators.GetSlotTypeVersionsPaginator responses = client.getSlotTypeVersionsIterable(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getSlotTypeVersions(software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetSlotTypeVersionsPaginator implements SdkIterable<GetSlotTypeVersionsResponse> {
    private final LexModelBuildingClient client;

    private final GetSlotTypeVersionsRequest firstRequest;

    private final NextPageFetcher nextPageFetcher;

    public GetSlotTypeVersionsPaginator(final LexModelBuildingClient client, final GetSlotTypeVersionsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new GetSlotTypeVersionsResponseFetcher();
    }

    @Override
    public Iterator<GetSlotTypeVersionsResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    private class GetSlotTypeVersionsResponseFetcher implements NextPageFetcher<GetSlotTypeVersionsResponse> {
        @Override
        public boolean hasNextPage(GetSlotTypeVersionsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public GetSlotTypeVersionsResponse nextPage(GetSlotTypeVersionsResponse previousPage) {
            if (previousPage == null) {
                return client.getSlotTypeVersions(firstRequest);
            }
            return client.getSlotTypeVersions(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
