/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.paginators;

import java.util.concurrent.CompletableFuture;
import javax.annotation.Generated;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.pagination.async.SdkPublisher;
import software.amazon.awssdk.services.lexmodelbuilding.LexModelBuildingAsyncClient;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.lexmodelbuilding.LexModelBuildingAsyncClient#getIntentsPaginator(software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of {@link software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsResponse}
 * response pages as per demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the forEach helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.lexmodelbuilding.paginators.GetIntentsPublisher publisher = client.getIntentsPaginator(request);
 * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.lexmodelbuilding.paginators.GetIntentsPublisher publisher = client.getIntentsPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getIntents(software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetIntentsPublisher implements SdkPublisher<GetIntentsResponse> {
    private final LexModelBuildingAsyncClient client;

    private final GetIntentsRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public GetIntentsPublisher(final LexModelBuildingAsyncClient client, final GetIntentsRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private GetIntentsPublisher(final LexModelBuildingAsyncClient client, final GetIntentsRequest firstRequest,
            final boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new GetIntentsResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super GetIntentsResponse> subscriber) {
        subscriber.onSubscribe(new ResponsesSubscription(subscriber, nextPageFetcher));
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link GetIntentsPublisher} that can be used to retrieve the consecutive
     * pages that follows the input page.
     * </p>
     */
    public final GetIntentsPublisher resume(final GetIntentsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new GetIntentsPublisher(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new GetIntentsPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super GetIntentsResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class GetIntentsResponseFetcher implements AsyncPageFetcher<GetIntentsResponse> {
        @Override
        public boolean hasNextPage(final GetIntentsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public CompletableFuture<GetIntentsResponse> nextPage(final GetIntentsResponse previousPage) {
            if (previousPage == null) {
                return client.getIntents(firstRequest);
            }
            return client.getIntents(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
