/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Identifies the version of a specific slot.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Slot implements SdkPojo, Serializable, ToCopyableBuilder<Slot.Builder, Slot> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> SLOT_CONSTRAINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::slotConstraintAsString)).setter(setter(Builder::slotConstraint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotConstraint").build()).build();

    private static final SdkField<String> SLOT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::slotType)).setter(setter(Builder::slotType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotType").build()).build();

    private static final SdkField<String> SLOT_TYPE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::slotTypeVersion)).setter(setter(Builder::slotTypeVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotTypeVersion").build()).build();

    private static final SdkField<Prompt> VALUE_ELICITATION_PROMPT_FIELD = SdkField.<Prompt> builder(MarshallingType.SDK_POJO)
            .getter(getter(Slot::valueElicitationPrompt)).setter(setter(Builder::valueElicitationPrompt))
            .constructor(Prompt::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("valueElicitationPrompt").build())
            .build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Slot::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("priority").build()).build();

    private static final SdkField<List<String>> SAMPLE_UTTERANCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Slot::sampleUtterances))
            .setter(setter(Builder::sampleUtterances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sampleUtterances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESPONSE_CARD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::responseCard)).setter(setter(Builder::responseCard))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseCard").build()).build();

    private static final SdkField<String> OBFUSCATION_SETTING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Slot::obfuscationSettingAsString)).setter(setter(Builder::obfuscationSetting))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("obfuscationSetting").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            SLOT_CONSTRAINT_FIELD, SLOT_TYPE_FIELD, SLOT_TYPE_VERSION_FIELD, VALUE_ELICITATION_PROMPT_FIELD, PRIORITY_FIELD,
            SAMPLE_UTTERANCES_FIELD, RESPONSE_CARD_FIELD, OBFUSCATION_SETTING_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String slotConstraint;

    private final String slotType;

    private final String slotTypeVersion;

    private final Prompt valueElicitationPrompt;

    private final Integer priority;

    private final List<String> sampleUtterances;

    private final String responseCard;

    private final String obfuscationSetting;

    private Slot(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.slotConstraint = builder.slotConstraint;
        this.slotType = builder.slotType;
        this.slotTypeVersion = builder.slotTypeVersion;
        this.valueElicitationPrompt = builder.valueElicitationPrompt;
        this.priority = builder.priority;
        this.sampleUtterances = builder.sampleUtterances;
        this.responseCard = builder.responseCard;
        this.obfuscationSetting = builder.obfuscationSetting;
    }

    /**
     * <p>
     * The name of the slot.
     * </p>
     * 
     * @return The name of the slot.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A description of the slot.
     * </p>
     * 
     * @return A description of the slot.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * Specifies whether the slot is required or optional.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #slotConstraint}
     * will return {@link SlotConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #slotConstraintAsString}.
     * </p>
     * 
     * @return Specifies whether the slot is required or optional.
     * @see SlotConstraint
     */
    public SlotConstraint slotConstraint() {
        return SlotConstraint.fromValue(slotConstraint);
    }

    /**
     * <p>
     * Specifies whether the slot is required or optional.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #slotConstraint}
     * will return {@link SlotConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #slotConstraintAsString}.
     * </p>
     * 
     * @return Specifies whether the slot is required or optional.
     * @see SlotConstraint
     */
    public String slotConstraintAsString() {
        return slotConstraint;
    }

    /**
     * <p>
     * The type of the slot, either a custom slot type that you defined or one of the built-in slot types.
     * </p>
     * 
     * @return The type of the slot, either a custom slot type that you defined or one of the built-in slot types.
     */
    public String slotType() {
        return slotType;
    }

    /**
     * <p>
     * The version of the slot type.
     * </p>
     * 
     * @return The version of the slot type.
     */
    public String slotTypeVersion() {
        return slotTypeVersion;
    }

    /**
     * <p>
     * The prompt that Amazon Lex uses to elicit the slot value from the user.
     * </p>
     * 
     * @return The prompt that Amazon Lex uses to elicit the slot value from the user.
     */
    public Prompt valueElicitationPrompt() {
        return valueElicitationPrompt;
    }

    /**
     * <p>
     * Directs Lex the order in which to elicit this slot value from the user. For example, if the intent has two slots
     * with priorities 1 and 2, AWS Lex first elicits a value for the slot with priority 1.
     * </p>
     * <p>
     * If multiple slots share the same priority, the order in which Lex elicits values is arbitrary.
     * </p>
     * 
     * @return Directs Lex the order in which to elicit this slot value from the user. For example, if the intent has
     *         two slots with priorities 1 and 2, AWS Lex first elicits a value for the slot with priority 1.</p>
     *         <p>
     *         If multiple slots share the same priority, the order in which Lex elicits values is arbitrary.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * Returns true if the SampleUtterances property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSampleUtterances() {
        return sampleUtterances != null && !(sampleUtterances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If you know a specific pattern with which users might respond to an Amazon Lex request for a slot value, you can
     * provide those utterances to improve accuracy. This is optional. In most cases, Amazon Lex is capable of
     * understanding user utterances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSampleUtterances()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you know a specific pattern with which users might respond to an Amazon Lex request for a slot value,
     *         you can provide those utterances to improve accuracy. This is optional. In most cases, Amazon Lex is
     *         capable of understanding user utterances.
     */
    public List<String> sampleUtterances() {
        return sampleUtterances;
    }

    /**
     * <p>
     * A set of possible responses for the slot type used by text-based clients. A user chooses an option from the
     * response card, instead of using text to reply.
     * </p>
     * 
     * @return A set of possible responses for the slot type used by text-based clients. A user chooses an option from
     *         the response card, instead of using text to reply.
     */
    public String responseCard() {
        return responseCard;
    }

    /**
     * <p>
     * Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate a slot,
     * the value is replaced by the slot name in curly braces ({}). For example, if the slot name is "full_name",
     * obfuscated values are replaced with "{full_name}". For more information, see <a
     * href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #obfuscationSetting} will return {@link ObfuscationSetting#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #obfuscationSettingAsString}.
     * </p>
     * 
     * @return Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate a
     *         slot, the value is replaced by the slot name in curly braces ({}). For example, if the slot name is
     *         "full_name", obfuscated values are replaced with "{full_name}". For more information, see <a
     *         href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
     * @see ObfuscationSetting
     */
    public ObfuscationSetting obfuscationSetting() {
        return ObfuscationSetting.fromValue(obfuscationSetting);
    }

    /**
     * <p>
     * Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate a slot,
     * the value is replaced by the slot name in curly braces ({}). For example, if the slot name is "full_name",
     * obfuscated values are replaced with "{full_name}". For more information, see <a
     * href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #obfuscationSetting} will return {@link ObfuscationSetting#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #obfuscationSettingAsString}.
     * </p>
     * 
     * @return Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate a
     *         slot, the value is replaced by the slot name in curly braces ({}). For example, if the slot name is
     *         "full_name", obfuscated values are replaced with "{full_name}". For more information, see <a
     *         href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
     * @see ObfuscationSetting
     */
    public String obfuscationSettingAsString() {
        return obfuscationSetting;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(slotConstraintAsString());
        hashCode = 31 * hashCode + Objects.hashCode(slotType());
        hashCode = 31 * hashCode + Objects.hashCode(slotTypeVersion());
        hashCode = 31 * hashCode + Objects.hashCode(valueElicitationPrompt());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(sampleUtterances());
        hashCode = 31 * hashCode + Objects.hashCode(responseCard());
        hashCode = 31 * hashCode + Objects.hashCode(obfuscationSettingAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Slot)) {
            return false;
        }
        Slot other = (Slot) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(slotConstraintAsString(), other.slotConstraintAsString())
                && Objects.equals(slotType(), other.slotType()) && Objects.equals(slotTypeVersion(), other.slotTypeVersion())
                && Objects.equals(valueElicitationPrompt(), other.valueElicitationPrompt())
                && Objects.equals(priority(), other.priority()) && Objects.equals(sampleUtterances(), other.sampleUtterances())
                && Objects.equals(responseCard(), other.responseCard())
                && Objects.equals(obfuscationSettingAsString(), other.obfuscationSettingAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Slot").add("Name", name()).add("Description", description())
                .add("SlotConstraint", slotConstraintAsString()).add("SlotType", slotType())
                .add("SlotTypeVersion", slotTypeVersion()).add("ValueElicitationPrompt", valueElicitationPrompt())
                .add("Priority", priority()).add("SampleUtterances", sampleUtterances()).add("ResponseCard", responseCard())
                .add("ObfuscationSetting", obfuscationSettingAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "slotConstraint":
            return Optional.ofNullable(clazz.cast(slotConstraintAsString()));
        case "slotType":
            return Optional.ofNullable(clazz.cast(slotType()));
        case "slotTypeVersion":
            return Optional.ofNullable(clazz.cast(slotTypeVersion()));
        case "valueElicitationPrompt":
            return Optional.ofNullable(clazz.cast(valueElicitationPrompt()));
        case "priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "sampleUtterances":
            return Optional.ofNullable(clazz.cast(sampleUtterances()));
        case "responseCard":
            return Optional.ofNullable(clazz.cast(responseCard()));
        case "obfuscationSetting":
            return Optional.ofNullable(clazz.cast(obfuscationSettingAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Slot, T> g) {
        return obj -> g.apply((Slot) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Slot> {
        /**
         * <p>
         * The name of the slot.
         * </p>
         * 
         * @param name
         *        The name of the slot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the slot.
         * </p>
         * 
         * @param description
         *        A description of the slot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Specifies whether the slot is required or optional.
         * </p>
         * 
         * @param slotConstraint
         *        Specifies whether the slot is required or optional.
         * @see SlotConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SlotConstraint
         */
        Builder slotConstraint(String slotConstraint);

        /**
         * <p>
         * Specifies whether the slot is required or optional.
         * </p>
         * 
         * @param slotConstraint
         *        Specifies whether the slot is required or optional.
         * @see SlotConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SlotConstraint
         */
        Builder slotConstraint(SlotConstraint slotConstraint);

        /**
         * <p>
         * The type of the slot, either a custom slot type that you defined or one of the built-in slot types.
         * </p>
         * 
         * @param slotType
         *        The type of the slot, either a custom slot type that you defined or one of the built-in slot types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotType(String slotType);

        /**
         * <p>
         * The version of the slot type.
         * </p>
         * 
         * @param slotTypeVersion
         *        The version of the slot type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotTypeVersion(String slotTypeVersion);

        /**
         * <p>
         * The prompt that Amazon Lex uses to elicit the slot value from the user.
         * </p>
         * 
         * @param valueElicitationPrompt
         *        The prompt that Amazon Lex uses to elicit the slot value from the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder valueElicitationPrompt(Prompt valueElicitationPrompt);

        /**
         * <p>
         * The prompt that Amazon Lex uses to elicit the slot value from the user.
         * </p>
         * This is a convenience that creates an instance of the {@link Prompt.Builder} avoiding the need to create one
         * manually via {@link Prompt#builder()}.
         *
         * When the {@link Consumer} completes, {@link Prompt.Builder#build()} is called immediately and its result is
         * passed to {@link #valueElicitationPrompt(Prompt)}.
         * 
         * @param valueElicitationPrompt
         *        a consumer that will call methods on {@link Prompt.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #valueElicitationPrompt(Prompt)
         */
        default Builder valueElicitationPrompt(Consumer<Prompt.Builder> valueElicitationPrompt) {
            return valueElicitationPrompt(Prompt.builder().applyMutation(valueElicitationPrompt).build());
        }

        /**
         * <p>
         * Directs Lex the order in which to elicit this slot value from the user. For example, if the intent has two
         * slots with priorities 1 and 2, AWS Lex first elicits a value for the slot with priority 1.
         * </p>
         * <p>
         * If multiple slots share the same priority, the order in which Lex elicits values is arbitrary.
         * </p>
         * 
         * @param priority
         *        Directs Lex the order in which to elicit this slot value from the user. For example, if the intent has
         *        two slots with priorities 1 and 2, AWS Lex first elicits a value for the slot with priority 1.</p>
         *        <p>
         *        If multiple slots share the same priority, the order in which Lex elicits values is arbitrary.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * If you know a specific pattern with which users might respond to an Amazon Lex request for a slot value, you
         * can provide those utterances to improve accuracy. This is optional. In most cases, Amazon Lex is capable of
         * understanding user utterances.
         * </p>
         * 
         * @param sampleUtterances
         *        If you know a specific pattern with which users might respond to an Amazon Lex request for a slot
         *        value, you can provide those utterances to improve accuracy. This is optional. In most cases, Amazon
         *        Lex is capable of understanding user utterances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleUtterances(Collection<String> sampleUtterances);

        /**
         * <p>
         * If you know a specific pattern with which users might respond to an Amazon Lex request for a slot value, you
         * can provide those utterances to improve accuracy. This is optional. In most cases, Amazon Lex is capable of
         * understanding user utterances.
         * </p>
         * 
         * @param sampleUtterances
         *        If you know a specific pattern with which users might respond to an Amazon Lex request for a slot
         *        value, you can provide those utterances to improve accuracy. This is optional. In most cases, Amazon
         *        Lex is capable of understanding user utterances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleUtterances(String... sampleUtterances);

        /**
         * <p>
         * A set of possible responses for the slot type used by text-based clients. A user chooses an option from the
         * response card, instead of using text to reply.
         * </p>
         * 
         * @param responseCard
         *        A set of possible responses for the slot type used by text-based clients. A user chooses an option
         *        from the response card, instead of using text to reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCard(String responseCard);

        /**
         * <p>
         * Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate a
         * slot, the value is replaced by the slot name in curly braces ({}). For example, if the slot name is
         * "full_name", obfuscated values are replaced with "{full_name}". For more information, see <a
         * href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
         * </p>
         * 
         * @param obfuscationSetting
         *        Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate
         *        a slot, the value is replaced by the slot name in curly braces ({}). For example, if the slot name is
         *        "full_name", obfuscated values are replaced with "{full_name}". For more information, see <a
         *        href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
         * @see ObfuscationSetting
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObfuscationSetting
         */
        Builder obfuscationSetting(String obfuscationSetting);

        /**
         * <p>
         * Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate a
         * slot, the value is replaced by the slot name in curly braces ({}). For example, if the slot name is
         * "full_name", obfuscated values are replaced with "{full_name}". For more information, see <a
         * href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
         * </p>
         * 
         * @param obfuscationSetting
         *        Determines whether a slot is obfuscated in conversation logs and stored utterances. When you obfuscate
         *        a slot, the value is replaced by the slot name in curly braces ({}). For example, if the slot name is
         *        "full_name", obfuscated values are replaced with "{full_name}". For more information, see <a
         *        href="https://docs.aws.amazon.com/lex/latest/dg/how-obfuscate.html"> Slot Obfuscation </a>.
         * @see ObfuscationSetting
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObfuscationSetting
         */
        Builder obfuscationSetting(ObfuscationSetting obfuscationSetting);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String slotConstraint;

        private String slotType;

        private String slotTypeVersion;

        private Prompt valueElicitationPrompt;

        private Integer priority;

        private List<String> sampleUtterances = DefaultSdkAutoConstructList.getInstance();

        private String responseCard;

        private String obfuscationSetting;

        private BuilderImpl() {
        }

        private BuilderImpl(Slot model) {
            name(model.name);
            description(model.description);
            slotConstraint(model.slotConstraint);
            slotType(model.slotType);
            slotTypeVersion(model.slotTypeVersion);
            valueElicitationPrompt(model.valueElicitationPrompt);
            priority(model.priority);
            sampleUtterances(model.sampleUtterances);
            responseCard(model.responseCard);
            obfuscationSetting(model.obfuscationSetting);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getSlotConstraintAsString() {
            return slotConstraint;
        }

        @Override
        public final Builder slotConstraint(String slotConstraint) {
            this.slotConstraint = slotConstraint;
            return this;
        }

        @Override
        public final Builder slotConstraint(SlotConstraint slotConstraint) {
            this.slotConstraint(slotConstraint == null ? null : slotConstraint.toString());
            return this;
        }

        public final void setSlotConstraint(String slotConstraint) {
            this.slotConstraint = slotConstraint;
        }

        public final String getSlotType() {
            return slotType;
        }

        @Override
        public final Builder slotType(String slotType) {
            this.slotType = slotType;
            return this;
        }

        public final void setSlotType(String slotType) {
            this.slotType = slotType;
        }

        public final String getSlotTypeVersion() {
            return slotTypeVersion;
        }

        @Override
        public final Builder slotTypeVersion(String slotTypeVersion) {
            this.slotTypeVersion = slotTypeVersion;
            return this;
        }

        public final void setSlotTypeVersion(String slotTypeVersion) {
            this.slotTypeVersion = slotTypeVersion;
        }

        public final Prompt.Builder getValueElicitationPrompt() {
            return valueElicitationPrompt != null ? valueElicitationPrompt.toBuilder() : null;
        }

        @Override
        public final Builder valueElicitationPrompt(Prompt valueElicitationPrompt) {
            this.valueElicitationPrompt = valueElicitationPrompt;
            return this;
        }

        public final void setValueElicitationPrompt(Prompt.BuilderImpl valueElicitationPrompt) {
            this.valueElicitationPrompt = valueElicitationPrompt != null ? valueElicitationPrompt.build() : null;
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final Collection<String> getSampleUtterances() {
            return sampleUtterances;
        }

        @Override
        public final Builder sampleUtterances(Collection<String> sampleUtterances) {
            this.sampleUtterances = SlotUtteranceListCopier.copy(sampleUtterances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sampleUtterances(String... sampleUtterances) {
            sampleUtterances(Arrays.asList(sampleUtterances));
            return this;
        }

        public final void setSampleUtterances(Collection<String> sampleUtterances) {
            this.sampleUtterances = SlotUtteranceListCopier.copy(sampleUtterances);
        }

        public final String getResponseCard() {
            return responseCard;
        }

        @Override
        public final Builder responseCard(String responseCard) {
            this.responseCard = responseCard;
            return this;
        }

        public final void setResponseCard(String responseCard) {
            this.responseCard = responseCard;
        }

        public final String getObfuscationSettingAsString() {
            return obfuscationSetting;
        }

        @Override
        public final Builder obfuscationSetting(String obfuscationSetting) {
            this.obfuscationSetting = obfuscationSetting;
            return this;
        }

        @Override
        public final Builder obfuscationSetting(ObfuscationSetting obfuscationSetting) {
            this.obfuscationSetting(obfuscationSetting == null ? null : obfuscationSetting.toString());
            return this;
        }

        public final void setObfuscationSetting(String obfuscationSetting) {
            this.obfuscationSetting = obfuscationSetting;
        }

        @Override
        public Slot build() {
            return new Slot(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
