/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an association between an Amazon Lex bot and an external messaging platform.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BotChannelAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<BotChannelAssociation.Builder, BotChannelAssociation> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> BOT_ALIAS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::botAlias)).setter(setter(Builder::botAlias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botAlias").build()).build();

    private static final SdkField<String> BOT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::botName)).setter(setter(Builder::botName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botName").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(BotChannelAssociation::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdDate").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Map<String, String>> BOT_CONFIGURATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(BotChannelAssociation::botConfiguration))
            .setter(setter(Builder::botConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botConfiguration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BotChannelAssociation::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            BOT_ALIAS_FIELD, BOT_NAME_FIELD, CREATED_DATE_FIELD, TYPE_FIELD, BOT_CONFIGURATION_FIELD, STATUS_FIELD,
            FAILURE_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String botAlias;

    private final String botName;

    private final Instant createdDate;

    private final String type;

    private final Map<String, String> botConfiguration;

    private final String status;

    private final String failureReason;

    private BotChannelAssociation(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.botAlias = builder.botAlias;
        this.botName = builder.botName;
        this.createdDate = builder.createdDate;
        this.type = builder.type;
        this.botConfiguration = builder.botConfiguration;
        this.status = builder.status;
        this.failureReason = builder.failureReason;
    }

    /**
     * <p>
     * The name of the association between the bot and the channel.
     * </p>
     * 
     * @return The name of the association between the bot and the channel.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A text description of the association you are creating.
     * </p>
     * 
     * @return A text description of the association you are creating.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * An alias pointing to the specific version of the Amazon Lex bot to which this association is being made.
     * </p>
     * 
     * @return An alias pointing to the specific version of the Amazon Lex bot to which this association is being made.
     */
    public String botAlias() {
        return botAlias;
    }

    /**
     * <p>
     * The name of the Amazon Lex bot to which this association is being made.
     * </p>
     * <note>
     * <p>
     * Currently, Amazon Lex supports associations with Facebook and Slack, and Twilio.
     * </p>
     * </note>
     * 
     * @return The name of the Amazon Lex bot to which this association is being made. </p> <note>
     *         <p>
     *         Currently, Amazon Lex supports associations with Facebook and Slack, and Twilio.
     *         </p>
     */
    public String botName() {
        return botName;
    }

    /**
     * <p>
     * The date that the association between the Amazon Lex bot and the channel was created.
     * </p>
     * 
     * @return The date that the association between the Amazon Lex bot and the channel was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * Specifies the type of association by indicating the type of channel being established between the Amazon Lex bot
     * and the external messaging platform.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ChannelType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies the type of association by indicating the type of channel being established between the Amazon
     *         Lex bot and the external messaging platform.
     * @see ChannelType
     */
    public ChannelType type() {
        return ChannelType.fromValue(type);
    }

    /**
     * <p>
     * Specifies the type of association by indicating the type of channel being established between the Amazon Lex bot
     * and the external messaging platform.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ChannelType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies the type of association by indicating the type of channel being established between the Amazon
     *         Lex bot and the external messaging platform.
     * @see ChannelType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * Returns true if the BotConfiguration property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasBotConfiguration() {
        return botConfiguration != null && !(botConfiguration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Provides information necessary to communicate with the messaging platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasBotConfiguration()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Provides information necessary to communicate with the messaging platform.
     */
    public Map<String, String> botConfiguration() {
        return botConfiguration;
    }

    /**
     * <p>
     * The status of the bot channel.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATED</code> - The channel has been created and is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - Channel creation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - There was an error creating the channel. For information about the reason for the failure,
     * see the <code>failureReason</code> field.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ChannelStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the bot channel. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATED</code> - The channel has been created and is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - Channel creation is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - There was an error creating the channel. For information about the reason for the
     *         failure, see the <code>failureReason</code> field.
     *         </p>
     *         </li>
     * @see ChannelStatus
     */
    public ChannelStatus status() {
        return ChannelStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the bot channel.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATED</code> - The channel has been created and is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - Channel creation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - There was an error creating the channel. For information about the reason for the failure,
     * see the <code>failureReason</code> field.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ChannelStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the bot channel. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATED</code> - The channel has been created and is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - Channel creation is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - There was an error creating the channel. For information about the reason for the
     *         failure, see the <code>failureReason</code> field.
     *         </p>
     *         </li>
     * @see ChannelStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * If <code>status</code> is <code>FAILED</code>, Amazon Lex provides the reason that it failed to create the
     * association.
     * </p>
     * 
     * @return If <code>status</code> is <code>FAILED</code>, Amazon Lex provides the reason that it failed to create
     *         the association.
     */
    public String failureReason() {
        return failureReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(botAlias());
        hashCode = 31 * hashCode + Objects.hashCode(botName());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(botConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BotChannelAssociation)) {
            return false;
        }
        BotChannelAssociation other = (BotChannelAssociation) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(botAlias(), other.botAlias()) && Objects.equals(botName(), other.botName())
                && Objects.equals(createdDate(), other.createdDate()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(botConfiguration(), other.botConfiguration())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(failureReason(), other.failureReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BotChannelAssociation").add("Name", name()).add("Description", description())
                .add("BotAlias", botAlias()).add("BotName", botName()).add("CreatedDate", createdDate())
                .add("Type", typeAsString())
                .add("BotConfiguration", botConfiguration() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Status", statusAsString()).add("FailureReason", failureReason()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "botAlias":
            return Optional.ofNullable(clazz.cast(botAlias()));
        case "botName":
            return Optional.ofNullable(clazz.cast(botName()));
        case "createdDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "botConfiguration":
            return Optional.ofNullable(clazz.cast(botConfiguration()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "failureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BotChannelAssociation, T> g) {
        return obj -> g.apply((BotChannelAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BotChannelAssociation> {
        /**
         * <p>
         * The name of the association between the bot and the channel.
         * </p>
         * 
         * @param name
         *        The name of the association between the bot and the channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A text description of the association you are creating.
         * </p>
         * 
         * @param description
         *        A text description of the association you are creating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * An alias pointing to the specific version of the Amazon Lex bot to which this association is being made.
         * </p>
         * 
         * @param botAlias
         *        An alias pointing to the specific version of the Amazon Lex bot to which this association is being
         *        made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botAlias(String botAlias);

        /**
         * <p>
         * The name of the Amazon Lex bot to which this association is being made.
         * </p>
         * <note>
         * <p>
         * Currently, Amazon Lex supports associations with Facebook and Slack, and Twilio.
         * </p>
         * </note>
         * 
         * @param botName
         *        The name of the Amazon Lex bot to which this association is being made. </p> <note>
         *        <p>
         *        Currently, Amazon Lex supports associations with Facebook and Slack, and Twilio.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botName(String botName);

        /**
         * <p>
         * The date that the association between the Amazon Lex bot and the channel was created.
         * </p>
         * 
         * @param createdDate
         *        The date that the association between the Amazon Lex bot and the channel was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * Specifies the type of association by indicating the type of channel being established between the Amazon Lex
         * bot and the external messaging platform.
         * </p>
         * 
         * @param type
         *        Specifies the type of association by indicating the type of channel being established between the
         *        Amazon Lex bot and the external messaging platform.
         * @see ChannelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies the type of association by indicating the type of channel being established between the Amazon Lex
         * bot and the external messaging platform.
         * </p>
         * 
         * @param type
         *        Specifies the type of association by indicating the type of channel being established between the
         *        Amazon Lex bot and the external messaging platform.
         * @see ChannelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelType
         */
        Builder type(ChannelType type);

        /**
         * <p>
         * Provides information necessary to communicate with the messaging platform.
         * </p>
         * 
         * @param botConfiguration
         *        Provides information necessary to communicate with the messaging platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botConfiguration(Map<String, String> botConfiguration);

        /**
         * <p>
         * The status of the bot channel.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATED</code> - The channel has been created and is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - Channel creation is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - There was an error creating the channel. For information about the reason for the
         * failure, see the <code>failureReason</code> field.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the bot channel. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATED</code> - The channel has been created and is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - Channel creation is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - There was an error creating the channel. For information about the reason for
         *        the failure, see the <code>failureReason</code> field.
         *        </p>
         *        </li>
         * @see ChannelStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the bot channel.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATED</code> - The channel has been created and is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - Channel creation is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - There was an error creating the channel. For information about the reason for the
         * failure, see the <code>failureReason</code> field.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the bot channel. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATED</code> - The channel has been created and is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - Channel creation is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - There was an error creating the channel. For information about the reason for
         *        the failure, see the <code>failureReason</code> field.
         *        </p>
         *        </li>
         * @see ChannelStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelStatus
         */
        Builder status(ChannelStatus status);

        /**
         * <p>
         * If <code>status</code> is <code>FAILED</code>, Amazon Lex provides the reason that it failed to create the
         * association.
         * </p>
         * 
         * @param failureReason
         *        If <code>status</code> is <code>FAILED</code>, Amazon Lex provides the reason that it failed to create
         *        the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String botAlias;

        private String botName;

        private Instant createdDate;

        private String type;

        private Map<String, String> botConfiguration = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private String failureReason;

        private BuilderImpl() {
        }

        private BuilderImpl(BotChannelAssociation model) {
            name(model.name);
            description(model.description);
            botAlias(model.botAlias);
            botName(model.botName);
            createdDate(model.createdDate);
            type(model.type);
            botConfiguration(model.botConfiguration);
            status(model.status);
            failureReason(model.failureReason);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getBotAlias() {
            return botAlias;
        }

        @Override
        public final Builder botAlias(String botAlias) {
            this.botAlias = botAlias;
            return this;
        }

        public final void setBotAlias(String botAlias) {
            this.botAlias = botAlias;
        }

        public final String getBotName() {
            return botName;
        }

        @Override
        public final Builder botName(String botName) {
            this.botName = botName;
            return this;
        }

        public final void setBotName(String botName) {
            this.botName = botName;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ChannelType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Map<String, String> getBotConfiguration() {
            return botConfiguration;
        }

        @Override
        public final Builder botConfiguration(Map<String, String> botConfiguration) {
            this.botConfiguration = ChannelConfigurationMapCopier.copy(botConfiguration);
            return this;
        }

        public final void setBotConfiguration(Map<String, String> botConfiguration) {
            this.botConfiguration = ChannelConfigurationMapCopier.copy(botConfiguration);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ChannelStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public BotChannelAssociation build() {
            return new BotChannelAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
