/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about alerts and warnings that Amazon Lex sends during a migration. The alerts include
 * information about how to resolve the issue.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MigrationAlert implements SdkPojo, Serializable, ToCopyableBuilder<MigrationAlert.Builder, MigrationAlert> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(MigrationAlert::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(MigrationAlert::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<List<String>> DETAILS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("details")
            .getter(getter(MigrationAlert::details))
            .setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("details").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> REFERENCE_UR_LS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("referenceURLs")
            .getter(getter(MigrationAlert::referenceURLs))
            .setter(setter(Builder::referenceURLs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("referenceURLs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, MESSAGE_FIELD,
            DETAILS_FIELD, REFERENCE_UR_LS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String message;

    private final List<String> details;

    private final List<String> referenceURLs;

    private MigrationAlert(BuilderImpl builder) {
        this.type = builder.type;
        this.message = builder.message;
        this.details = builder.details;
        this.referenceURLs = builder.referenceURLs;
    }

    /**
     * <p>
     * The type of alert. There are two kinds of alerts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration stops.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WARN</code> - There was an issue with the migration that requires manual changes to the new Amazon Lex V2
     * bot. The migration continues.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MigrationAlertType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of alert. There are two kinds of alerts:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration stops.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WARN</code> - There was an issue with the migration that requires manual changes to the new Amazon
     *         Lex V2 bot. The migration continues.
     *         </p>
     *         </li>
     * @see MigrationAlertType
     */
    public final MigrationAlertType type() {
        return MigrationAlertType.fromValue(type);
    }

    /**
     * <p>
     * The type of alert. There are two kinds of alerts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration stops.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WARN</code> - There was an issue with the migration that requires manual changes to the new Amazon Lex V2
     * bot. The migration continues.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MigrationAlertType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of alert. There are two kinds of alerts:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration stops.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WARN</code> - There was an issue with the migration that requires manual changes to the new Amazon
     *         Lex V2 bot. The migration continues.
     *         </p>
     *         </li>
     * @see MigrationAlertType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * A message that describes why the alert was issued.
     * </p>
     * 
     * @return A message that describes why the alert was issued.
     */
    public final String message() {
        return message;
    }

    /**
     * For responses, this returns true if the service returned a value for the Details property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDetails() {
        return details != null && !(details instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional details about the alert.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDetails} method.
     * </p>
     * 
     * @return Additional details about the alert.
     */
    public final List<String> details() {
        return details;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReferenceURLs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReferenceURLs() {
        return referenceURLs != null && !(referenceURLs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A link to the Amazon Lex documentation that describes how to resolve the alert.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReferenceURLs} method.
     * </p>
     * 
     * @return A link to the Amazon Lex documentation that describes how to resolve the alert.
     */
    public final List<String> referenceURLs() {
        return referenceURLs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasDetails() ? details() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasReferenceURLs() ? referenceURLs() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MigrationAlert)) {
            return false;
        }
        MigrationAlert other = (MigrationAlert) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(message(), other.message())
                && hasDetails() == other.hasDetails() && Objects.equals(details(), other.details())
                && hasReferenceURLs() == other.hasReferenceURLs() && Objects.equals(referenceURLs(), other.referenceURLs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MigrationAlert").add("Type", typeAsString()).add("Message", message())
                .add("Details", hasDetails() ? details() : null)
                .add("ReferenceURLs", hasReferenceURLs() ? referenceURLs() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "details":
            return Optional.ofNullable(clazz.cast(details()));
        case "referenceURLs":
            return Optional.ofNullable(clazz.cast(referenceURLs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("message", MESSAGE_FIELD);
        map.put("details", DETAILS_FIELD);
        map.put("referenceURLs", REFERENCE_UR_LS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MigrationAlert, T> g) {
        return obj -> g.apply((MigrationAlert) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MigrationAlert> {
        /**
         * <p>
         * The type of alert. There are two kinds of alerts:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration stops.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WARN</code> - There was an issue with the migration that requires manual changes to the new Amazon Lex
         * V2 bot. The migration continues.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of alert. There are two kinds of alerts:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration
         *        stops.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WARN</code> - There was an issue with the migration that requires manual changes to the new
         *        Amazon Lex V2 bot. The migration continues.
         *        </p>
         *        </li>
         * @see MigrationAlertType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationAlertType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of alert. There are two kinds of alerts:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration stops.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WARN</code> - There was an issue with the migration that requires manual changes to the new Amazon Lex
         * V2 bot. The migration continues.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of alert. There are two kinds of alerts:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ERROR</code> - There was an issue with the migration that can't be resolved. The migration
         *        stops.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WARN</code> - There was an issue with the migration that requires manual changes to the new
         *        Amazon Lex V2 bot. The migration continues.
         *        </p>
         *        </li>
         * @see MigrationAlertType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationAlertType
         */
        Builder type(MigrationAlertType type);

        /**
         * <p>
         * A message that describes why the alert was issued.
         * </p>
         * 
         * @param message
         *        A message that describes why the alert was issued.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * Additional details about the alert.
         * </p>
         * 
         * @param details
         *        Additional details about the alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(Collection<String> details);

        /**
         * <p>
         * Additional details about the alert.
         * </p>
         * 
         * @param details
         *        Additional details about the alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(String... details);

        /**
         * <p>
         * A link to the Amazon Lex documentation that describes how to resolve the alert.
         * </p>
         * 
         * @param referenceURLs
         *        A link to the Amazon Lex documentation that describes how to resolve the alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceURLs(Collection<String> referenceURLs);

        /**
         * <p>
         * A link to the Amazon Lex documentation that describes how to resolve the alert.
         * </p>
         * 
         * @param referenceURLs
         *        A link to the Amazon Lex documentation that describes how to resolve the alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceURLs(String... referenceURLs);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String message;

        private List<String> details = DefaultSdkAutoConstructList.getInstance();

        private List<String> referenceURLs = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MigrationAlert model) {
            type(model.type);
            message(model.message);
            details(model.details);
            referenceURLs(model.referenceURLs);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(MigrationAlertType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Collection<String> getDetails() {
            if (details instanceof SdkAutoConstructList) {
                return null;
            }
            return details;
        }

        public final void setDetails(Collection<String> details) {
            this.details = MigrationAlertDetailsCopier.copy(details);
        }

        @Override
        public final Builder details(Collection<String> details) {
            this.details = MigrationAlertDetailsCopier.copy(details);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder details(String... details) {
            details(Arrays.asList(details));
            return this;
        }

        public final Collection<String> getReferenceURLs() {
            if (referenceURLs instanceof SdkAutoConstructList) {
                return null;
            }
            return referenceURLs;
        }

        public final void setReferenceURLs(Collection<String> referenceURLs) {
            this.referenceURLs = MigrationAlertReferenceURLsCopier.copy(referenceURLs);
        }

        @Override
        public final Builder referenceURLs(Collection<String> referenceURLs) {
            this.referenceURLs = MigrationAlertReferenceURLsCopier.copy(referenceURLs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceURLs(String... referenceURLs) {
            referenceURLs(Arrays.asList(referenceURLs));
            return this;
        }

        @Override
        public MigrationAlert build() {
            return new MigrationAlert(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
