/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides a list of utterances that have been made to a specific version of your bot. The list contains a maximum of
 * 100 utterances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UtteranceList implements SdkPojo, Serializable, ToCopyableBuilder<UtteranceList.Builder, UtteranceList> {
    private static final SdkField<String> BOT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UtteranceList::botVersion)).setter(setter(Builder::botVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botVersion").build()).build();

    private static final SdkField<List<UtteranceData>> UTTERANCES_FIELD = SdkField
            .<List<UtteranceData>> builder(MarshallingType.LIST)
            .getter(getter(UtteranceList::utterances))
            .setter(setter(Builder::utterances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utterances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtteranceData> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtteranceData::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BOT_VERSION_FIELD,
            UTTERANCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String botVersion;

    private final List<UtteranceData> utterances;

    private UtteranceList(BuilderImpl builder) {
        this.botVersion = builder.botVersion;
        this.utterances = builder.utterances;
    }

    /**
     * <p>
     * The version of the bot that processed the list.
     * </p>
     * 
     * @return The version of the bot that processed the list.
     */
    public String botVersion() {
        return botVersion;
    }

    /**
     * <p>
     * One or more <a>UtteranceData</a> objects that contain information about the utterances that have been made to a
     * bot. The maximum number of object is 100.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more <a>UtteranceData</a> objects that contain information about the utterances that have been
     *         made to a bot. The maximum number of object is 100.
     */
    public List<UtteranceData> utterances() {
        return utterances;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(botVersion());
        hashCode = 31 * hashCode + Objects.hashCode(utterances());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UtteranceList)) {
            return false;
        }
        UtteranceList other = (UtteranceList) obj;
        return Objects.equals(botVersion(), other.botVersion()) && Objects.equals(utterances(), other.utterances());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UtteranceList").add("BotVersion", botVersion()).add("Utterances", utterances()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "botVersion":
            return Optional.ofNullable(clazz.cast(botVersion()));
        case "utterances":
            return Optional.ofNullable(clazz.cast(utterances()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UtteranceList, T> g) {
        return obj -> g.apply((UtteranceList) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UtteranceList> {
        /**
         * <p>
         * The version of the bot that processed the list.
         * </p>
         * 
         * @param botVersion
         *        The version of the bot that processed the list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botVersion(String botVersion);

        /**
         * <p>
         * One or more <a>UtteranceData</a> objects that contain information about the utterances that have been made to
         * a bot. The maximum number of object is 100.
         * </p>
         * 
         * @param utterances
         *        One or more <a>UtteranceData</a> objects that contain information about the utterances that have been
         *        made to a bot. The maximum number of object is 100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utterances(Collection<UtteranceData> utterances);

        /**
         * <p>
         * One or more <a>UtteranceData</a> objects that contain information about the utterances that have been made to
         * a bot. The maximum number of object is 100.
         * </p>
         * 
         * @param utterances
         *        One or more <a>UtteranceData</a> objects that contain information about the utterances that have been
         *        made to a bot. The maximum number of object is 100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utterances(UtteranceData... utterances);

        /**
         * <p>
         * One or more <a>UtteranceData</a> objects that contain information about the utterances that have been made to
         * a bot. The maximum number of object is 100.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UtteranceData>.Builder} avoiding the need
         * to create one manually via {@link List<UtteranceData>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UtteranceData>.Builder#build()} is called immediately and
         * its result is passed to {@link #utterances(List<UtteranceData>)}.
         * 
         * @param utterances
         *        a consumer that will call methods on {@link List<UtteranceData>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utterances(List<UtteranceData>)
         */
        Builder utterances(Consumer<UtteranceData.Builder>... utterances);
    }

    static final class BuilderImpl implements Builder {
        private String botVersion;

        private List<UtteranceData> utterances = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UtteranceList model) {
            botVersion(model.botVersion);
            utterances(model.utterances);
        }

        public final String getBotVersion() {
            return botVersion;
        }

        @Override
        public final Builder botVersion(String botVersion) {
            this.botVersion = botVersion;
            return this;
        }

        public final void setBotVersion(String botVersion) {
            this.botVersion = botVersion;
        }

        public final Collection<UtteranceData.Builder> getUtterances() {
            return utterances != null ? utterances.stream().map(UtteranceData::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder utterances(Collection<UtteranceData> utterances) {
            this.utterances = ListOfUtteranceCopier.copy(utterances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utterances(UtteranceData... utterances) {
            utterances(Arrays.asList(utterances));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utterances(Consumer<UtteranceData.Builder>... utterances) {
            utterances(Stream.of(utterances).map(c -> UtteranceData.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUtterances(Collection<UtteranceData.BuilderImpl> utterances) {
            this.utterances = ListOfUtteranceCopier.copyFromBuilder(utterances);
        }

        @Override
        public UtteranceList build() {
            return new UtteranceList(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
