/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a list of message groups that Amazon Lex sends to a user to elicit a response.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PromptSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<PromptSpecification.Builder, PromptSpecification> {
    private static final SdkField<List<MessageGroup>> MESSAGE_GROUPS_FIELD = SdkField
            .<List<MessageGroup>> builder(MarshallingType.LIST)
            .memberName("messageGroups")
            .getter(getter(PromptSpecification::messageGroups))
            .setter(setter(Builder::messageGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messageGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxRetries").getter(getter(PromptSpecification::maxRetries)).setter(setter(Builder::maxRetries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxRetries").build()).build();

    private static final SdkField<Boolean> ALLOW_INTERRUPT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("allowInterrupt").getter(getter(PromptSpecification::allowInterrupt))
            .setter(setter(Builder::allowInterrupt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowInterrupt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MESSAGE_GROUPS_FIELD,
            MAX_RETRIES_FIELD, ALLOW_INTERRUPT_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<MessageGroup> messageGroups;

    private final Integer maxRetries;

    private final Boolean allowInterrupt;

    private PromptSpecification(BuilderImpl builder) {
        this.messageGroups = builder.messageGroups;
        this.maxRetries = builder.maxRetries;
        this.allowInterrupt = builder.allowInterrupt;
    }

    /**
     * Returns true if the MessageGroups property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasMessageGroups() {
        return messageGroups != null && !(messageGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message to send at
     * runtime.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMessageGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message to
     *         send at runtime.
     */
    public final List<MessageGroup> messageGroups() {
        return messageGroups;
    }

    /**
     * <p>
     * The maximum number of times the bot tries to elicit a resonse from the user using this prompt.
     * </p>
     * 
     * @return The maximum number of times the bot tries to elicit a resonse from the user using this prompt.
     */
    public final Integer maxRetries() {
        return maxRetries;
    }

    /**
     * <p>
     * Indicates whether the user can interrupt a speech prompt from the bot.
     * </p>
     * 
     * @return Indicates whether the user can interrupt a speech prompt from the bot.
     */
    public final Boolean allowInterrupt() {
        return allowInterrupt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageGroups() ? messageGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxRetries());
        hashCode = 31 * hashCode + Objects.hashCode(allowInterrupt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PromptSpecification)) {
            return false;
        }
        PromptSpecification other = (PromptSpecification) obj;
        return hasMessageGroups() == other.hasMessageGroups() && Objects.equals(messageGroups(), other.messageGroups())
                && Objects.equals(maxRetries(), other.maxRetries()) && Objects.equals(allowInterrupt(), other.allowInterrupt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PromptSpecification").add("MessageGroups", hasMessageGroups() ? messageGroups() : null)
                .add("MaxRetries", maxRetries()).add("AllowInterrupt", allowInterrupt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "messageGroups":
            return Optional.ofNullable(clazz.cast(messageGroups()));
        case "maxRetries":
            return Optional.ofNullable(clazz.cast(maxRetries()));
        case "allowInterrupt":
            return Optional.ofNullable(clazz.cast(allowInterrupt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PromptSpecification, T> g) {
        return obj -> g.apply((PromptSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PromptSpecification> {
        /**
         * <p>
         * A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message to send
         * at runtime.
         * </p>
         * 
         * @param messageGroups
         *        A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message
         *        to send at runtime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageGroups(Collection<MessageGroup> messageGroups);

        /**
         * <p>
         * A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message to send
         * at runtime.
         * </p>
         * 
         * @param messageGroups
         *        A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message
         *        to send at runtime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageGroups(MessageGroup... messageGroups);

        /**
         * <p>
         * A collection of messages that Amazon Lex can send to the user. Amazon Lex chooses the actual message to send
         * at runtime.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MessageGroup>.Builder} avoiding the need to
         * create one manually via {@link List<MessageGroup>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MessageGroup>.Builder#build()} is called immediately and its
         * result is passed to {@link #messageGroups(List<MessageGroup>)}.
         * 
         * @param messageGroups
         *        a consumer that will call methods on {@link List<MessageGroup>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messageGroups(List<MessageGroup>)
         */
        Builder messageGroups(Consumer<MessageGroup.Builder>... messageGroups);

        /**
         * <p>
         * The maximum number of times the bot tries to elicit a resonse from the user using this prompt.
         * </p>
         * 
         * @param maxRetries
         *        The maximum number of times the bot tries to elicit a resonse from the user using this prompt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRetries(Integer maxRetries);

        /**
         * <p>
         * Indicates whether the user can interrupt a speech prompt from the bot.
         * </p>
         * 
         * @param allowInterrupt
         *        Indicates whether the user can interrupt a speech prompt from the bot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowInterrupt(Boolean allowInterrupt);
    }

    static final class BuilderImpl implements Builder {
        private List<MessageGroup> messageGroups = DefaultSdkAutoConstructList.getInstance();

        private Integer maxRetries;

        private Boolean allowInterrupt;

        private BuilderImpl() {
        }

        private BuilderImpl(PromptSpecification model) {
            messageGroups(model.messageGroups);
            maxRetries(model.maxRetries);
            allowInterrupt(model.allowInterrupt);
        }

        public final List<MessageGroup.Builder> getMessageGroups() {
            List<MessageGroup.Builder> result = MessageGroupsListCopier.copyToBuilder(this.messageGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder messageGroups(Collection<MessageGroup> messageGroups) {
            this.messageGroups = MessageGroupsListCopier.copy(messageGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messageGroups(MessageGroup... messageGroups) {
            messageGroups(Arrays.asList(messageGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messageGroups(Consumer<MessageGroup.Builder>... messageGroups) {
            messageGroups(Stream.of(messageGroups).map(c -> MessageGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMessageGroups(Collection<MessageGroup.BuilderImpl> messageGroups) {
            this.messageGroups = MessageGroupsListCopier.copyFromBuilder(messageGroups);
        }

        public final Integer getMaxRetries() {
            return maxRetries;
        }

        @Override
        public final Builder maxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public final void setMaxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
        }

        public final Boolean getAllowInterrupt() {
            return allowInterrupt;
        }

        @Override
        public final Builder allowInterrupt(Boolean allowInterrupt) {
            this.allowInterrupt = allowInterrupt;
            return this;
        }

        public final void setAllowInterrupt(Boolean allowInterrupt) {
            this.allowInterrupt = allowInterrupt;
        }

        @Override
        public PromptSpecification build() {
            return new PromptSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
