/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.services.lexmodelsv2.LexModelsV2Client;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotAliasRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotAliasResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotLocaleRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotLocaleResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotVersionRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeBotVersionResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeExportRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeExportResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeImportRequest;
import software.amazon.awssdk.services.lexmodelsv2.model.DescribeImportResponse;
import software.amazon.awssdk.services.lexmodelsv2.model.LexModelsV2Request;
import software.amazon.awssdk.services.lexmodelsv2.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultLexModelsV2Waiter implements LexModelsV2Waiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final LexModelsV2Client client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeBotResponse> botAvailableWaiter;

    private final Waiter<DescribeBotAliasResponse> botAliasAvailableWaiter;

    private final Waiter<DescribeExportResponse> botExportCompletedWaiter;

    private final Waiter<DescribeImportResponse> botImportCompletedWaiter;

    private final Waiter<DescribeBotLocaleResponse> botLocaleBuiltWaiter;

    private final Waiter<DescribeBotLocaleResponse> botLocaleExpressTestingAvailableWaiter;

    private final Waiter<DescribeBotVersionResponse> botVersionAvailableWaiter;

    private final Waiter<DescribeBotLocaleResponse> botLocaleCreatedWaiter;

    private DefaultLexModelsV2Waiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = LexModelsV2Client.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.botAvailableWaiter = Waiter.builder(DescribeBotResponse.class).acceptors(botAvailableWaiterAcceptors())
                .overrideConfiguration(botAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.botAliasAvailableWaiter = Waiter.builder(DescribeBotAliasResponse.class)
                .acceptors(botAliasAvailableWaiterAcceptors())
                .overrideConfiguration(botAliasAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.botExportCompletedWaiter = Waiter.builder(DescribeExportResponse.class)
                .acceptors(botExportCompletedWaiterAcceptors())
                .overrideConfiguration(botExportCompletedWaiterConfig(builder.overrideConfiguration)).build();
        this.botImportCompletedWaiter = Waiter.builder(DescribeImportResponse.class)
                .acceptors(botImportCompletedWaiterAcceptors())
                .overrideConfiguration(botImportCompletedWaiterConfig(builder.overrideConfiguration)).build();
        this.botLocaleBuiltWaiter = Waiter.builder(DescribeBotLocaleResponse.class).acceptors(botLocaleBuiltWaiterAcceptors())
                .overrideConfiguration(botLocaleBuiltWaiterConfig(builder.overrideConfiguration)).build();
        this.botLocaleExpressTestingAvailableWaiter = Waiter.builder(DescribeBotLocaleResponse.class)
                .acceptors(botLocaleExpressTestingAvailableWaiterAcceptors())
                .overrideConfiguration(botLocaleExpressTestingAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.botVersionAvailableWaiter = Waiter.builder(DescribeBotVersionResponse.class)
                .acceptors(botVersionAvailableWaiterAcceptors())
                .overrideConfiguration(botVersionAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.botLocaleCreatedWaiter = Waiter.builder(DescribeBotLocaleResponse.class)
                .acceptors(botLocaleCreatedWaiterAcceptors())
                .overrideConfiguration(botLocaleCreatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeBotAliasResponse> waitUntilBotAliasAvailable(DescribeBotAliasRequest describeBotAliasRequest) {
        return botAliasAvailableWaiter.run(() -> client.describeBotAlias(applyWaitersUserAgent(describeBotAliasRequest)));
    }

    @Override
    public WaiterResponse<DescribeBotAliasResponse> waitUntilBotAliasAvailable(DescribeBotAliasRequest describeBotAliasRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botAliasAvailableWaiter.run(() -> client.describeBotAlias(applyWaitersUserAgent(describeBotAliasRequest)),
                botAliasAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeBotResponse> waitUntilBotAvailable(DescribeBotRequest describeBotRequest) {
        return botAvailableWaiter.run(() -> client.describeBot(applyWaitersUserAgent(describeBotRequest)));
    }

    @Override
    public WaiterResponse<DescribeBotResponse> waitUntilBotAvailable(DescribeBotRequest describeBotRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botAvailableWaiter.run(() -> client.describeBot(applyWaitersUserAgent(describeBotRequest)),
                botAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeExportResponse> waitUntilBotExportCompleted(DescribeExportRequest describeExportRequest) {
        return botExportCompletedWaiter.run(() -> client.describeExport(applyWaitersUserAgent(describeExportRequest)));
    }

    @Override
    public WaiterResponse<DescribeExportResponse> waitUntilBotExportCompleted(DescribeExportRequest describeExportRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botExportCompletedWaiter.run(() -> client.describeExport(applyWaitersUserAgent(describeExportRequest)),
                botExportCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeImportResponse> waitUntilBotImportCompleted(DescribeImportRequest describeImportRequest) {
        return botImportCompletedWaiter.run(() -> client.describeImport(applyWaitersUserAgent(describeImportRequest)));
    }

    @Override
    public WaiterResponse<DescribeImportResponse> waitUntilBotImportCompleted(DescribeImportRequest describeImportRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botImportCompletedWaiter.run(() -> client.describeImport(applyWaitersUserAgent(describeImportRequest)),
                botImportCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeBotLocaleResponse> waitUntilBotLocaleBuilt(DescribeBotLocaleRequest describeBotLocaleRequest) {
        return botLocaleBuiltWaiter.run(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)));
    }

    @Override
    public WaiterResponse<DescribeBotLocaleResponse> waitUntilBotLocaleBuilt(DescribeBotLocaleRequest describeBotLocaleRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botLocaleBuiltWaiter.run(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)),
                botLocaleBuiltWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeBotLocaleResponse> waitUntilBotLocaleCreated(DescribeBotLocaleRequest describeBotLocaleRequest) {
        return botLocaleCreatedWaiter.run(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)));
    }

    @Override
    public WaiterResponse<DescribeBotLocaleResponse> waitUntilBotLocaleCreated(DescribeBotLocaleRequest describeBotLocaleRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return botLocaleCreatedWaiter.run(() -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)),
                botLocaleCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeBotLocaleResponse> waitUntilBotLocaleExpressTestingAvailable(
            DescribeBotLocaleRequest describeBotLocaleRequest) {
        return botLocaleExpressTestingAvailableWaiter.run(() -> client
                .describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)));
    }

    @Override
    public WaiterResponse<DescribeBotLocaleResponse> waitUntilBotLocaleExpressTestingAvailable(
            DescribeBotLocaleRequest describeBotLocaleRequest, WaiterOverrideConfiguration overrideConfig) {
        return botLocaleExpressTestingAvailableWaiter.run(
                () -> client.describeBotLocale(applyWaitersUserAgent(describeBotLocaleRequest)),
                botLocaleExpressTestingAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeBotVersionResponse> waitUntilBotVersionAvailable(
            DescribeBotVersionRequest describeBotVersionRequest) {
        return botVersionAvailableWaiter.run(() -> client.describeBotVersion(applyWaitersUserAgent(describeBotVersionRequest)));
    }

    @Override
    public WaiterResponse<DescribeBotVersionResponse> waitUntilBotVersionAvailable(
            DescribeBotVersionRequest describeBotVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return botVersionAvailableWaiter.run(() -> client.describeBotVersion(applyWaitersUserAgent(describeBotVersionRequest)),
                botVersionAvailableWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeBotResponse>> botAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Available");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Inactive");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotAliasResponse>> botAliasAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotAliasResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botAliasStatus").value(), "Available");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botAliasStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botAliasStatus").value(), "Deleting");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeExportResponse>> botExportCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeExportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("exportStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("exportStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("exportStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImportResponse>> botImportCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("importStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("importStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("importStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotLocaleResponse>> botLocaleBuiltWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotLocaleResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Built");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "NotBuilt");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotLocaleResponse>> botLocaleExpressTestingAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotLocaleResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Built");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "ReadyExpressTesting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "NotBuilt");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotVersionResponse>> botVersionAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Available");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botStatus").value(), "Failed");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBotLocaleResponse>> botLocaleCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBotLocaleResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Built");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "ReadyExpressTesting");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "NotBuilt");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Deleting");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("botLocaleStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration botAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botAliasAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botExportCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botImportCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botLocaleBuiltWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botLocaleExpressTestingAvailableWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botVersionAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration botLocaleCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static LexModelsV2Waiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends LexModelsV2Request> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements LexModelsV2Waiter.Builder {
        private LexModelsV2Client client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public LexModelsV2Waiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public LexModelsV2Waiter.Builder client(LexModelsV2Client client) {
            this.client = client;
            return this;
        }

        public LexModelsV2Waiter build() {
            return new DefaultLexModelsV2Waiter(this);
        }
    }
}
