/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides one or more messages that Amazon Lex should send to the user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageGroup implements SdkPojo, Serializable, ToCopyableBuilder<MessageGroup.Builder, MessageGroup> {
    private static final SdkField<Message> MESSAGE_FIELD = SdkField.<Message> builder(MarshallingType.SDK_POJO)
            .memberName("message").getter(getter(MessageGroup::message)).setter(setter(Builder::message))
            .constructor(Message::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<List<Message>> VARIATIONS_FIELD = SdkField
            .<List<Message>> builder(MarshallingType.LIST)
            .memberName("variations")
            .getter(getter(MessageGroup::variations))
            .setter(setter(Builder::variations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Message> builder(MarshallingType.SDK_POJO)
                                            .constructor(Message::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MESSAGE_FIELD,
            VARIATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Message message;

    private final List<Message> variations;

    private MessageGroup(BuilderImpl builder) {
        this.message = builder.message;
        this.variations = builder.variations;
    }

    /**
     * <p>
     * The primary message that Amazon Lex should send to the user.
     * </p>
     * 
     * @return The primary message that Amazon Lex should send to the user.
     */
    public final Message message() {
        return message;
    }

    /**
     * For responses, this returns true if the service returned a value for the Variations property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVariations() {
        return variations != null && !(variations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary message or
     * one of the variations to send to the user.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariations} method.
     * </p>
     * 
     * @return Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary
     *         message or one of the variations to send to the user.
     */
    public final List<Message> variations() {
        return variations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasVariations() ? variations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageGroup)) {
            return false;
        }
        MessageGroup other = (MessageGroup) obj;
        return Objects.equals(message(), other.message()) && hasVariations() == other.hasVariations()
                && Objects.equals(variations(), other.variations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MessageGroup").add("Message", message())
                .add("Variations", hasVariations() ? variations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "variations":
            return Optional.ofNullable(clazz.cast(variations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MessageGroup, T> g) {
        return obj -> g.apply((MessageGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageGroup> {
        /**
         * <p>
         * The primary message that Amazon Lex should send to the user.
         * </p>
         * 
         * @param message
         *        The primary message that Amazon Lex should send to the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(Message message);

        /**
         * <p>
         * The primary message that Amazon Lex should send to the user.
         * </p>
         * This is a convenience that creates an instance of the {@link Message.Builder} avoiding the need to create one
         * manually via {@link Message#builder()}.
         *
         * When the {@link Consumer} completes, {@link Message.Builder#build()} is called immediately and its result is
         * passed to {@link #message(Message)}.
         * 
         * @param message
         *        a consumer that will call methods on {@link Message.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #message(Message)
         */
        default Builder message(Consumer<Message.Builder> message) {
            return message(Message.builder().applyMutation(message).build());
        }

        /**
         * <p>
         * Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary message
         * or one of the variations to send to the user.
         * </p>
         * 
         * @param variations
         *        Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary
         *        message or one of the variations to send to the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variations(Collection<Message> variations);

        /**
         * <p>
         * Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary message
         * or one of the variations to send to the user.
         * </p>
         * 
         * @param variations
         *        Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary
         *        message or one of the variations to send to the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variations(Message... variations);

        /**
         * <p>
         * Message variations to send to the user. When variations are defined, Amazon Lex chooses the primary message
         * or one of the variations to send to the user.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Message>.Builder} avoiding the need to
         * create one manually via {@link List<Message>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Message>.Builder#build()} is called immediately and its
         * result is passed to {@link #variations(List<Message>)}.
         * 
         * @param variations
         *        a consumer that will call methods on {@link List<Message>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variations(List<Message>)
         */
        Builder variations(Consumer<Message.Builder>... variations);
    }

    static final class BuilderImpl implements Builder {
        private Message message;

        private List<Message> variations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MessageGroup model) {
            message(model.message);
            variations(model.variations);
        }

        public final Message.Builder getMessage() {
            return message != null ? message.toBuilder() : null;
        }

        public final void setMessage(Message.BuilderImpl message) {
            this.message = message != null ? message.build() : null;
        }

        @Override
        @Transient
        public final Builder message(Message message) {
            this.message = message;
            return this;
        }

        public final List<Message.Builder> getVariations() {
            List<Message.Builder> result = MessageVariationsListCopier.copyToBuilder(this.variations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariations(Collection<Message.BuilderImpl> variations) {
            this.variations = MessageVariationsListCopier.copyFromBuilder(variations);
        }

        @Override
        @Transient
        public final Builder variations(Collection<Message> variations) {
            this.variations = MessageVariationsListCopier.copy(variations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder variations(Message... variations) {
            variations(Arrays.asList(variations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder variations(Consumer<Message.Builder>... variations) {
            variations(Stream.of(variations).map(c -> Message.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MessageGroup build() {
            return new MessageGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
