/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains summary information for all the version replication statuses applicable for Global resiliency.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BotVersionReplicaSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<BotVersionReplicaSummary.Builder, BotVersionReplicaSummary> {
    private static final SdkField<String> BOT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("botVersion").getter(getter(BotVersionReplicaSummary::botVersion)).setter(setter(Builder::botVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botVersion").build()).build();

    private static final SdkField<String> BOT_VERSION_REPLICATION_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("botVersionReplicationStatus")
            .getter(getter(BotVersionReplicaSummary::botVersionReplicationStatusAsString))
            .setter(setter(Builder::botVersionReplicationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botVersionReplicationStatus")
                    .build()).build();

    private static final SdkField<Instant> CREATION_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationDateTime").getter(getter(BotVersionReplicaSummary::creationDateTime))
            .setter(setter(Builder::creationDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationDateTime").build()).build();

    private static final SdkField<List<String>> FAILURE_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("failureReasons")
            .getter(getter(BotVersionReplicaSummary::failureReasons))
            .setter(setter(Builder::failureReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BOT_VERSION_FIELD,
            BOT_VERSION_REPLICATION_STATUS_FIELD, CREATION_DATE_TIME_FIELD, FAILURE_REASONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String botVersion;

    private final String botVersionReplicationStatus;

    private final Instant creationDateTime;

    private final List<String> failureReasons;

    private BotVersionReplicaSummary(BuilderImpl builder) {
        this.botVersion = builder.botVersion;
        this.botVersionReplicationStatus = builder.botVersionReplicationStatus;
        this.creationDateTime = builder.creationDateTime;
        this.failureReasons = builder.failureReasons;
    }

    /**
     * <p>
     * The bot version for the summary information for all the version replication statuses.
     * </p>
     * 
     * @return The bot version for the summary information for all the version replication statuses.
     */
    public final String botVersion() {
        return botVersion;
    }

    /**
     * <p>
     * The version replication status for all the replicated bots.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #botVersionReplicationStatus} will return {@link BotVersionReplicationStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #botVersionReplicationStatusAsString}.
     * </p>
     * 
     * @return The version replication status for all the replicated bots.
     * @see BotVersionReplicationStatus
     */
    public final BotVersionReplicationStatus botVersionReplicationStatus() {
        return BotVersionReplicationStatus.fromValue(botVersionReplicationStatus);
    }

    /**
     * <p>
     * The version replication status for all the replicated bots.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #botVersionReplicationStatus} will return {@link BotVersionReplicationStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #botVersionReplicationStatusAsString}.
     * </p>
     * 
     * @return The version replication status for all the replicated bots.
     * @see BotVersionReplicationStatus
     */
    public final String botVersionReplicationStatusAsString() {
        return botVersionReplicationStatus;
    }

    /**
     * <p>
     * The creation date and time of the replication status for all the replicated bots.
     * </p>
     * 
     * @return The creation date and time of the replication status for all the replicated bots.
     */
    public final Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureReasons() {
        return failureReasons != null && !(failureReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reasons for replication failure for all the replicated bots.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureReasons} method.
     * </p>
     * 
     * @return The reasons for replication failure for all the replicated bots.
     */
    public final List<String> failureReasons() {
        return failureReasons;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(botVersion());
        hashCode = 31 * hashCode + Objects.hashCode(botVersionReplicationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureReasons() ? failureReasons() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BotVersionReplicaSummary)) {
            return false;
        }
        BotVersionReplicaSummary other = (BotVersionReplicaSummary) obj;
        return Objects.equals(botVersion(), other.botVersion())
                && Objects.equals(botVersionReplicationStatusAsString(), other.botVersionReplicationStatusAsString())
                && Objects.equals(creationDateTime(), other.creationDateTime())
                && hasFailureReasons() == other.hasFailureReasons() && Objects.equals(failureReasons(), other.failureReasons());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BotVersionReplicaSummary").add("BotVersion", botVersion())
                .add("BotVersionReplicationStatus", botVersionReplicationStatusAsString())
                .add("CreationDateTime", creationDateTime()).add("FailureReasons", hasFailureReasons() ? failureReasons() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "botVersion":
            return Optional.ofNullable(clazz.cast(botVersion()));
        case "botVersionReplicationStatus":
            return Optional.ofNullable(clazz.cast(botVersionReplicationStatusAsString()));
        case "creationDateTime":
            return Optional.ofNullable(clazz.cast(creationDateTime()));
        case "failureReasons":
            return Optional.ofNullable(clazz.cast(failureReasons()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BotVersionReplicaSummary, T> g) {
        return obj -> g.apply((BotVersionReplicaSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BotVersionReplicaSummary> {
        /**
         * <p>
         * The bot version for the summary information for all the version replication statuses.
         * </p>
         * 
         * @param botVersion
         *        The bot version for the summary information for all the version replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botVersion(String botVersion);

        /**
         * <p>
         * The version replication status for all the replicated bots.
         * </p>
         * 
         * @param botVersionReplicationStatus
         *        The version replication status for all the replicated bots.
         * @see BotVersionReplicationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BotVersionReplicationStatus
         */
        Builder botVersionReplicationStatus(String botVersionReplicationStatus);

        /**
         * <p>
         * The version replication status for all the replicated bots.
         * </p>
         * 
         * @param botVersionReplicationStatus
         *        The version replication status for all the replicated bots.
         * @see BotVersionReplicationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BotVersionReplicationStatus
         */
        Builder botVersionReplicationStatus(BotVersionReplicationStatus botVersionReplicationStatus);

        /**
         * <p>
         * The creation date and time of the replication status for all the replicated bots.
         * </p>
         * 
         * @param creationDateTime
         *        The creation date and time of the replication status for all the replicated bots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The reasons for replication failure for all the replicated bots.
         * </p>
         * 
         * @param failureReasons
         *        The reasons for replication failure for all the replicated bots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(Collection<String> failureReasons);

        /**
         * <p>
         * The reasons for replication failure for all the replicated bots.
         * </p>
         * 
         * @param failureReasons
         *        The reasons for replication failure for all the replicated bots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(String... failureReasons);
    }

    static final class BuilderImpl implements Builder {
        private String botVersion;

        private String botVersionReplicationStatus;

        private Instant creationDateTime;

        private List<String> failureReasons = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BotVersionReplicaSummary model) {
            botVersion(model.botVersion);
            botVersionReplicationStatus(model.botVersionReplicationStatus);
            creationDateTime(model.creationDateTime);
            failureReasons(model.failureReasons);
        }

        public final String getBotVersion() {
            return botVersion;
        }

        public final void setBotVersion(String botVersion) {
            this.botVersion = botVersion;
        }

        @Override
        public final Builder botVersion(String botVersion) {
            this.botVersion = botVersion;
            return this;
        }

        public final String getBotVersionReplicationStatus() {
            return botVersionReplicationStatus;
        }

        public final void setBotVersionReplicationStatus(String botVersionReplicationStatus) {
            this.botVersionReplicationStatus = botVersionReplicationStatus;
        }

        @Override
        public final Builder botVersionReplicationStatus(String botVersionReplicationStatus) {
            this.botVersionReplicationStatus = botVersionReplicationStatus;
            return this;
        }

        @Override
        public final Builder botVersionReplicationStatus(BotVersionReplicationStatus botVersionReplicationStatus) {
            this.botVersionReplicationStatus(botVersionReplicationStatus == null ? null : botVersionReplicationStatus.toString());
            return this;
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final Collection<String> getFailureReasons() {
            if (failureReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return failureReasons;
        }

        public final void setFailureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
        }

        @Override
        public final Builder failureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(String... failureReasons) {
            failureReasons(Arrays.asList(failureReasons));
            return this;
        }

        @Override
        public BotVersionReplicaSummary build() {
            return new BotVersionReplicaSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
