/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the test set that is imported.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TestSetImportResourceSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<TestSetImportResourceSpecification.Builder, TestSetImportResourceSpecification> {
    private static final SdkField<String> TEST_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("testSetName").getter(getter(TestSetImportResourceSpecification::testSetName))
            .setter(setter(Builder::testSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("testSetName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(TestSetImportResourceSpecification::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(TestSetImportResourceSpecification::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<TestSetStorageLocation> STORAGE_LOCATION_FIELD = SdkField
            .<TestSetStorageLocation> builder(MarshallingType.SDK_POJO).memberName("storageLocation")
            .getter(getter(TestSetImportResourceSpecification::storageLocation)).setter(setter(Builder::storageLocation))
            .constructor(TestSetStorageLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storageLocation").build()).build();

    private static final SdkField<TestSetImportInputLocation> IMPORT_INPUT_LOCATION_FIELD = SdkField
            .<TestSetImportInputLocation> builder(MarshallingType.SDK_POJO).memberName("importInputLocation")
            .getter(getter(TestSetImportResourceSpecification::importInputLocation)).setter(setter(Builder::importInputLocation))
            .constructor(TestSetImportInputLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("importInputLocation").build())
            .build();

    private static final SdkField<String> MODALITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("modality").getter(getter(TestSetImportResourceSpecification::modalityAsString))
            .setter(setter(Builder::modality))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modality").build()).build();

    private static final SdkField<Map<String, String>> TEST_SET_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("testSetTags")
            .getter(getter(TestSetImportResourceSpecification::testSetTags))
            .setter(setter(Builder::testSetTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("testSetTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TEST_SET_NAME_FIELD,
            DESCRIPTION_FIELD, ROLE_ARN_FIELD, STORAGE_LOCATION_FIELD, IMPORT_INPUT_LOCATION_FIELD, MODALITY_FIELD,
            TEST_SET_TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String testSetName;

    private final String description;

    private final String roleArn;

    private final TestSetStorageLocation storageLocation;

    private final TestSetImportInputLocation importInputLocation;

    private final String modality;

    private final Map<String, String> testSetTags;

    private TestSetImportResourceSpecification(BuilderImpl builder) {
        this.testSetName = builder.testSetName;
        this.description = builder.description;
        this.roleArn = builder.roleArn;
        this.storageLocation = builder.storageLocation;
        this.importInputLocation = builder.importInputLocation;
        this.modality = builder.modality;
        this.testSetTags = builder.testSetTags;
    }

    /**
     * <p>
     * The name of the test set.
     * </p>
     * 
     * @return The name of the test set.
     */
    public final String testSetName() {
        return testSetName;
    }

    /**
     * <p>
     * The description of the test set.
     * </p>
     * 
     * @return The description of the test set.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an IAM role that has permission to access the test set.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an IAM role that has permission to access the test set.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * Contains information about the location that Amazon Lex uses to store the test-set.
     * </p>
     * 
     * @return Contains information about the location that Amazon Lex uses to store the test-set.
     */
    public final TestSetStorageLocation storageLocation() {
        return storageLocation;
    }

    /**
     * <p>
     * Contains information about the input location from where test-set should be imported.
     * </p>
     * 
     * @return Contains information about the input location from where test-set should be imported.
     */
    public final TestSetImportInputLocation importInputLocation() {
        return importInputLocation;
    }

    /**
     * <p>
     * Specifies whether the test-set being imported contains written or spoken data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #modality} will
     * return {@link TestSetModality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modalityAsString}.
     * </p>
     * 
     * @return Specifies whether the test-set being imported contains written or spoken data.
     * @see TestSetModality
     */
    public final TestSetModality modality() {
        return TestSetModality.fromValue(modality);
    }

    /**
     * <p>
     * Specifies whether the test-set being imported contains written or spoken data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #modality} will
     * return {@link TestSetModality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modalityAsString}.
     * </p>
     * 
     * @return Specifies whether the test-set being imported contains written or spoken data.
     * @see TestSetModality
     */
    public final String modalityAsString() {
        return modality;
    }

    /**
     * For responses, this returns true if the service returned a value for the TestSetTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTestSetTags() {
        return testSetTags != null && !(testSetTags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of tags to add to the test set. You can only add tags when you import/generate a new test set. You can't
     * use the <code>UpdateTestSet</code> operation to update tags. To update tags, use the <code>TagResource</code>
     * operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTestSetTags} method.
     * </p>
     * 
     * @return A list of tags to add to the test set. You can only add tags when you import/generate a new test set. You
     *         can't use the <code>UpdateTestSet</code> operation to update tags. To update tags, use the
     *         <code>TagResource</code> operation.
     */
    public final Map<String, String> testSetTags() {
        return testSetTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(testSetName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(storageLocation());
        hashCode = 31 * hashCode + Objects.hashCode(importInputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(modalityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTestSetTags() ? testSetTags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TestSetImportResourceSpecification)) {
            return false;
        }
        TestSetImportResourceSpecification other = (TestSetImportResourceSpecification) obj;
        return Objects.equals(testSetName(), other.testSetName()) && Objects.equals(description(), other.description())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(storageLocation(), other.storageLocation())
                && Objects.equals(importInputLocation(), other.importInputLocation())
                && Objects.equals(modalityAsString(), other.modalityAsString()) && hasTestSetTags() == other.hasTestSetTags()
                && Objects.equals(testSetTags(), other.testSetTags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TestSetImportResourceSpecification").add("TestSetName", testSetName())
                .add("Description", description()).add("RoleArn", roleArn()).add("StorageLocation", storageLocation())
                .add("ImportInputLocation", importInputLocation()).add("Modality", modalityAsString())
                .add("TestSetTags", hasTestSetTags() ? testSetTags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "testSetName":
            return Optional.ofNullable(clazz.cast(testSetName()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "storageLocation":
            return Optional.ofNullable(clazz.cast(storageLocation()));
        case "importInputLocation":
            return Optional.ofNullable(clazz.cast(importInputLocation()));
        case "modality":
            return Optional.ofNullable(clazz.cast(modalityAsString()));
        case "testSetTags":
            return Optional.ofNullable(clazz.cast(testSetTags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("testSetName", TEST_SET_NAME_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("storageLocation", STORAGE_LOCATION_FIELD);
        map.put("importInputLocation", IMPORT_INPUT_LOCATION_FIELD);
        map.put("modality", MODALITY_FIELD);
        map.put("testSetTags", TEST_SET_TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TestSetImportResourceSpecification, T> g) {
        return obj -> g.apply((TestSetImportResourceSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TestSetImportResourceSpecification> {
        /**
         * <p>
         * The name of the test set.
         * </p>
         * 
         * @param testSetName
         *        The name of the test set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder testSetName(String testSetName);

        /**
         * <p>
         * The description of the test set.
         * </p>
         * 
         * @param description
         *        The description of the test set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an IAM role that has permission to access the test set.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of an IAM role that has permission to access the test set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * Contains information about the location that Amazon Lex uses to store the test-set.
         * </p>
         * 
         * @param storageLocation
         *        Contains information about the location that Amazon Lex uses to store the test-set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageLocation(TestSetStorageLocation storageLocation);

        /**
         * <p>
         * Contains information about the location that Amazon Lex uses to store the test-set.
         * </p>
         * This is a convenience method that creates an instance of the {@link TestSetStorageLocation.Builder} avoiding
         * the need to create one manually via {@link TestSetStorageLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TestSetStorageLocation.Builder#build()} is called immediately and
         * its result is passed to {@link #storageLocation(TestSetStorageLocation)}.
         * 
         * @param storageLocation
         *        a consumer that will call methods on {@link TestSetStorageLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageLocation(TestSetStorageLocation)
         */
        default Builder storageLocation(Consumer<TestSetStorageLocation.Builder> storageLocation) {
            return storageLocation(TestSetStorageLocation.builder().applyMutation(storageLocation).build());
        }

        /**
         * <p>
         * Contains information about the input location from where test-set should be imported.
         * </p>
         * 
         * @param importInputLocation
         *        Contains information about the input location from where test-set should be imported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importInputLocation(TestSetImportInputLocation importInputLocation);

        /**
         * <p>
         * Contains information about the input location from where test-set should be imported.
         * </p>
         * This is a convenience method that creates an instance of the {@link TestSetImportInputLocation.Builder}
         * avoiding the need to create one manually via {@link TestSetImportInputLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TestSetImportInputLocation.Builder#build()} is called immediately
         * and its result is passed to {@link #importInputLocation(TestSetImportInputLocation)}.
         * 
         * @param importInputLocation
         *        a consumer that will call methods on {@link TestSetImportInputLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #importInputLocation(TestSetImportInputLocation)
         */
        default Builder importInputLocation(Consumer<TestSetImportInputLocation.Builder> importInputLocation) {
            return importInputLocation(TestSetImportInputLocation.builder().applyMutation(importInputLocation).build());
        }

        /**
         * <p>
         * Specifies whether the test-set being imported contains written or spoken data.
         * </p>
         * 
         * @param modality
         *        Specifies whether the test-set being imported contains written or spoken data.
         * @see TestSetModality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TestSetModality
         */
        Builder modality(String modality);

        /**
         * <p>
         * Specifies whether the test-set being imported contains written or spoken data.
         * </p>
         * 
         * @param modality
         *        Specifies whether the test-set being imported contains written or spoken data.
         * @see TestSetModality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TestSetModality
         */
        Builder modality(TestSetModality modality);

        /**
         * <p>
         * A list of tags to add to the test set. You can only add tags when you import/generate a new test set. You
         * can't use the <code>UpdateTestSet</code> operation to update tags. To update tags, use the
         * <code>TagResource</code> operation.
         * </p>
         * 
         * @param testSetTags
         *        A list of tags to add to the test set. You can only add tags when you import/generate a new test set.
         *        You can't use the <code>UpdateTestSet</code> operation to update tags. To update tags, use the
         *        <code>TagResource</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder testSetTags(Map<String, String> testSetTags);
    }

    static final class BuilderImpl implements Builder {
        private String testSetName;

        private String description;

        private String roleArn;

        private TestSetStorageLocation storageLocation;

        private TestSetImportInputLocation importInputLocation;

        private String modality;

        private Map<String, String> testSetTags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TestSetImportResourceSpecification model) {
            testSetName(model.testSetName);
            description(model.description);
            roleArn(model.roleArn);
            storageLocation(model.storageLocation);
            importInputLocation(model.importInputLocation);
            modality(model.modality);
            testSetTags(model.testSetTags);
        }

        public final String getTestSetName() {
            return testSetName;
        }

        public final void setTestSetName(String testSetName) {
            this.testSetName = testSetName;
        }

        @Override
        public final Builder testSetName(String testSetName) {
            this.testSetName = testSetName;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final TestSetStorageLocation.Builder getStorageLocation() {
            return storageLocation != null ? storageLocation.toBuilder() : null;
        }

        public final void setStorageLocation(TestSetStorageLocation.BuilderImpl storageLocation) {
            this.storageLocation = storageLocation != null ? storageLocation.build() : null;
        }

        @Override
        public final Builder storageLocation(TestSetStorageLocation storageLocation) {
            this.storageLocation = storageLocation;
            return this;
        }

        public final TestSetImportInputLocation.Builder getImportInputLocation() {
            return importInputLocation != null ? importInputLocation.toBuilder() : null;
        }

        public final void setImportInputLocation(TestSetImportInputLocation.BuilderImpl importInputLocation) {
            this.importInputLocation = importInputLocation != null ? importInputLocation.build() : null;
        }

        @Override
        public final Builder importInputLocation(TestSetImportInputLocation importInputLocation) {
            this.importInputLocation = importInputLocation;
            return this;
        }

        public final String getModality() {
            return modality;
        }

        public final void setModality(String modality) {
            this.modality = modality;
        }

        @Override
        public final Builder modality(String modality) {
            this.modality = modality;
            return this;
        }

        @Override
        public final Builder modality(TestSetModality modality) {
            this.modality(modality == null ? null : modality.toString());
            return this;
        }

        public final Map<String, String> getTestSetTags() {
            if (testSetTags instanceof SdkAutoConstructMap) {
                return null;
            }
            return testSetTags;
        }

        public final void setTestSetTags(Map<String, String> testSetTags) {
            this.testSetTags = TagMapCopier.copy(testSetTags);
        }

        @Override
        public final Builder testSetTags(Map<String, String> testSetTags) {
            this.testSetTags = TagMapCopier.copy(testSetTags);
            return this;
        }

        @Override
        public TestSetImportResourceSpecification build() {
            return new TestSetImportResourceSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
