/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceResponse;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationRequest;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationRequestEventStream;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationResponseHandler;

/**
 * Service client for accessing Lex Runtime V2 asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface LexRuntimeV2AsyncClient extends SdkClient {
    String SERVICE_NAME = "lex";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "runtime-v2-lex";

    /**
     * Create a {@link LexRuntimeV2AsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static LexRuntimeV2AsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link LexRuntimeV2AsyncClient}.
     */
    static LexRuntimeV2AsyncClientBuilder builder() {
        return new DefaultLexRuntimeV2AsyncClientBuilder();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     * <p>
     * You can use this operation to restart a conversation with a bot. When you remove a session, the entire history of
     * the session is removed so that you can start again.
     * </p>
     * <p>
     * You don't need to delete a session. Sessions have a time limit and will expire. Set the session time limit when
     * you create the bot. The default is 5 minutes, but you can specify anything between 1 minute and 24 hours.
     * </p>
     * <p>
     * If you specify a bot or alias ID that doesn't exist, you receive a <code>BadRequestException.</code>
     * </p>
     * <p>
     * If the locale doesn't exist in the bot, or if the locale hasn't been enables for the alias, you receive a
     * <code>BadRequestException</code>.
     * </p>
     *
     * @param deleteSessionRequest
     * @return A Java Future containing the result of the DeleteSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/DeleteSession" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteSessionResponse> deleteSession(DeleteSessionRequest deleteSessionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     * <p>
     * You can use this operation to restart a conversation with a bot. When you remove a session, the entire history of
     * the session is removed so that you can start again.
     * </p>
     * <p>
     * You don't need to delete a session. Sessions have a time limit and will expire. Set the session time limit when
     * you create the bot. The default is 5 minutes, but you can specify anything between 1 minute and 24 hours.
     * </p>
     * <p>
     * If you specify a bot or alias ID that doesn't exist, you receive a <code>BadRequestException.</code>
     * </p>
     * <p>
     * If the locale doesn't exist in the bot, or if the locale hasn't been enables for the alias, you receive a
     * <code>BadRequestException</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSessionRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteSessionRequest#builder()}
     * </p>
     *
     * @param deleteSessionRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSessionRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/DeleteSession" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteSessionResponse> deleteSession(Consumer<DeleteSessionRequest.Builder> deleteSessionRequest) {
        return deleteSession(DeleteSessionRequest.builder().applyMutation(deleteSessionRequest).build());
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user.
     * </p>
     * <p>
     * For example, you can use this operation to retrieve session information for a user that has left a long-running
     * session in use.
     * </p>
     * <p>
     * If the bot, alias, or session identifier doesn't exist, Amazon Lex returns a <code>BadRequestException</code>. If
     * the locale doesn't exist or is not enabled for the alias, you receive a <code>BadRequestException</code>.
     * </p>
     *
     * @param getSessionRequest
     * @return A Java Future containing the result of the GetSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetSessionResponse> getSession(GetSessionRequest getSessionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user.
     * </p>
     * <p>
     * For example, you can use this operation to retrieve session information for a user that has left a long-running
     * session in use.
     * </p>
     * <p>
     * If the bot, alias, or session identifier doesn't exist, Amazon Lex returns a <code>BadRequestException</code>. If
     * the locale doesn't exist or is not enabled for the alias, you receive a <code>BadRequestException</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSessionRequest.Builder} avoiding the need to
     * create one manually via {@link GetSessionRequest#builder()}
     * </p>
     *
     * @param getSessionRequest
     *        A {@link Consumer} that will call methods on {@link GetSessionRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetSessionResponse> getSession(Consumer<GetSessionRequest.Builder> getSessionRequest) {
        return getSession(GetSessionRequest.builder().applyMutation(getSessionRequest).build());
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex bot. Use this operation to enable your
     * application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> putSession(PutSessionRequest putSessionRequest,
            AsyncResponseTransformer<PutSessionResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex bot. Use this operation to enable your
     * application to set the state of the bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSessionRequest.Builder} avoiding the need to
     * create one manually via {@link PutSessionRequest#builder()}
     * </p>
     *
     * @param putSessionRequest
     *        A {@link Consumer} that will call methods on {@link PutSessionRequest.Builder} to create a request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> putSession(Consumer<PutSessionRequest.Builder> putSessionRequest,
            AsyncResponseTransformer<PutSessionResponse, ReturnT> asyncResponseTransformer) {
        return putSession(PutSessionRequest.builder().applyMutation(putSessionRequest).build(), asyncResponseTransformer);
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex bot. Use this operation to enable your
     * application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutSessionResponse> putSession(PutSessionRequest putSessionRequest, Path destinationPath) {
        return putSession(putSessionRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex bot. Use this operation to enable your
     * application to set the state of the bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSessionRequest.Builder} avoiding the need to
     * create one manually via {@link PutSessionRequest#builder()}
     * </p>
     *
     * @param putSessionRequest
     *        A {@link Consumer} that will call methods on {@link PutSessionRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutSessionResponse> putSession(Consumer<PutSessionRequest.Builder> putSessionRequest,
            Path destinationPath) {
        return putSession(PutSessionRequest.builder().applyMutation(putSessionRequest).build(), destinationPath);
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. Client applications use this API to send requests to Amazon Lex at runtime.
     * Amazon Lex then interprets the user input using the machine learning model that it build for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex returns the next message to convey to the user and an optional response card to display.
     * </p>
     *
     * @param recognizeTextRequest
     * @return A Java Future containing the result of the RecognizeText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeText" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<RecognizeTextResponse> recognizeText(RecognizeTextRequest recognizeTextRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. Client applications use this API to send requests to Amazon Lex at runtime.
     * Amazon Lex then interprets the user input using the machine learning model that it build for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex returns the next message to convey to the user and an optional response card to display.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RecognizeTextRequest.Builder} avoiding the need to
     * create one manually via {@link RecognizeTextRequest#builder()}
     * </p>
     *
     * @param recognizeTextRequest
     *        A {@link Consumer} that will call methods on {@link RecognizeTextRequest.Builder} to create a request.
     * @return A Java Future containing the result of the RecognizeText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeText" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<RecognizeTextResponse> recognizeText(Consumer<RecognizeTextRequest.Builder> recognizeTextRequest) {
        return recognizeText(RecognizeTextRequest.builder().applyMutation(recognizeTextRequest).build());
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. You can send text or speech. Clients use this API to send text and audio requests
     * to Amazon Lex at runtime. Amazon Lex interprets the user input using the machine learning model built for the
     * bot.
     * </p>
     *
     * @param recognizeUtteranceRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex did not understand the user intent, it sends the <code>clarificationPrompt</code>
     *        configured for the bot. If the intent requires confirmation before taking the fulfillment action, it sends
     *        the <code>confirmationPrompt</code>. Another example: Suppose that the Lambda function successfully
     *        fulfilled the intent, and sent a message to convey to the user. Then Amazon Lex sends that message in the
     *        response.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> recognizeUtterance(RecognizeUtteranceRequest recognizeUtteranceRequest,
            AsyncRequestBody requestBody, AsyncResponseTransformer<RecognizeUtteranceResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. You can send text or speech. Clients use this API to send text and audio requests
     * to Amazon Lex at runtime. Amazon Lex interprets the user input using the machine learning model built for the
     * bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RecognizeUtteranceRequest.Builder} avoiding the
     * need to create one manually via {@link RecognizeUtteranceRequest#builder()}
     * </p>
     *
     * @param recognizeUtteranceRequest
     *        A {@link Consumer} that will call methods on {@link RecognizeUtteranceRequest.Builder} to create a
     *        request.
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex did not understand the user intent, it sends the <code>clarificationPrompt</code>
     *        configured for the bot. If the intent requires confirmation before taking the fulfillment action, it sends
     *        the <code>confirmationPrompt</code>. Another example: Suppose that the Lambda function successfully
     *        fulfilled the intent, and sent a message to convey to the user. Then Amazon Lex sends that message in the
     *        response.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> recognizeUtterance(
            Consumer<RecognizeUtteranceRequest.Builder> recognizeUtteranceRequest, AsyncRequestBody requestBody,
            AsyncResponseTransformer<RecognizeUtteranceResponse, ReturnT> asyncResponseTransformer) {
        return recognizeUtterance(RecognizeUtteranceRequest.builder().applyMutation(recognizeUtteranceRequest).build(),
                requestBody, asyncResponseTransformer);
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. You can send text or speech. Clients use this API to send text and audio requests
     * to Amazon Lex at runtime. Amazon Lex interprets the user input using the machine learning model built for the
     * bot.
     * </p>
     *
     * @param recognizeUtteranceRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex did not understand the user intent, it sends the <code>clarificationPrompt</code>
     *        configured for the bot. If the intent requires confirmation before taking the fulfillment action, it sends
     *        the <code>confirmationPrompt</code>. Another example: Suppose that the Lambda function successfully
     *        fulfilled the intent, and sent a message to convey to the user. Then Amazon Lex sends that message in the
     *        response.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<RecognizeUtteranceResponse> recognizeUtterance(RecognizeUtteranceRequest recognizeUtteranceRequest,
            Path sourcePath, Path destinationPath) {
        return recognizeUtterance(recognizeUtteranceRequest, AsyncRequestBody.fromFile(sourcePath),
                AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. You can send text or speech. Clients use this API to send text and audio requests
     * to Amazon Lex at runtime. Amazon Lex interprets the user input using the machine learning model built for the
     * bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RecognizeUtteranceRequest.Builder} avoiding the
     * need to create one manually via {@link RecognizeUtteranceRequest#builder()}
     * </p>
     *
     * @param recognizeUtteranceRequest
     *        A {@link Consumer} that will call methods on {@link RecognizeUtteranceRequest.Builder} to create a
     *        request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex did not understand the user intent, it sends the <code>clarificationPrompt</code>
     *        configured for the bot. If the intent requires confirmation before taking the fulfillment action, it sends
     *        the <code>confirmationPrompt</code>. Another example: Suppose that the Lambda function successfully
     *        fulfilled the intent, and sent a message to convey to the user. Then Amazon Lex sends that message in the
     *        response.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<RecognizeUtteranceResponse> recognizeUtterance(
            Consumer<RecognizeUtteranceRequest.Builder> recognizeUtteranceRequest, Path sourcePath, Path destinationPath) {
        return recognizeUtterance(RecognizeUtteranceRequest.builder().applyMutation(recognizeUtteranceRequest).build(),
                sourcePath, destinationPath);
    }

    /**
     * <p>
     * Starts an HTTP/2 bidirectional event stream that enables you to send audio, text, or DTMF input in real time.
     * After your application starts a conversation, users send input to Amazon Lex as a stream of events. Amazon Lex
     * processes the incoming events and responds with streaming text or audio events.
     * </p>
     * <p/>
     *
     * @param startConversationRequest
     * @return A Java Future containing the result of the StartConversation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException </li> <li>ValidationException </li> <li>ThrottlingException </li> <li>
     *         InternalServerException </li> <li>SdkException Base class for all exceptions that can be thrown by the
     *         SDK (both service and client). Can be used for catch all scenarios.</li> <li>SdkClientException If any
     *         client side error occurs such as an IO related failure, failure to get credentials, etc.</li> <li>
     *         LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.StartConversation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/StartConversation"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startConversation(StartConversationRequest startConversationRequest,
            Publisher<StartConversationRequestEventStream> requestStream, StartConversationResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts an HTTP/2 bidirectional event stream that enables you to send audio, text, or DTMF input in real time.
     * After your application starts a conversation, users send input to Amazon Lex as a stream of events. Amazon Lex
     * processes the incoming events and responds with streaming text or audio events.
     * </p>
     * <p/>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartConversationRequest.Builder} avoiding the need
     * to create one manually via {@link StartConversationRequest#builder()}
     * </p>
     *
     * @param startConversationRequest
     *        A {@link Consumer} that will call methods on {@link StartConversationRequest.Builder} to create a request.
     * @return A Java Future containing the result of the StartConversation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.StartConversation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/StartConversation"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startConversation(Consumer<StartConversationRequest.Builder> startConversationRequest,
            Publisher<StartConversationRequestEventStream> requestStream, StartConversationResponseHandler asyncResponseHandler) {
        return startConversation(StartConversationRequest.builder().applyMutation(startConversationRequest).build(),
                requestStream, asyncResponseHandler);
    }
}
