/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.lexruntimev2.model.AccessDeniedException;
import software.amazon.awssdk.services.lexruntimev2.model.BadGatewayException;
import software.amazon.awssdk.services.lexruntimev2.model.ConflictException;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.DependencyFailedException;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.InternalServerException;
import software.amazon.awssdk.services.lexruntimev2.model.LexRuntimeV2Exception;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceResponse;
import software.amazon.awssdk.services.lexruntimev2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lexruntimev2.model.ThrottlingException;
import software.amazon.awssdk.services.lexruntimev2.model.ValidationException;

/**
 * Service client for accessing Lex Runtime V2. This can be created using the static {@link #builder()} method.
 *
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface LexRuntimeV2Client extends SdkClient {
    String SERVICE_NAME = "lex";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "runtime-v2-lex";

    /**
     * Create a {@link LexRuntimeV2Client} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static LexRuntimeV2Client create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link LexRuntimeV2Client}.
     */
    static LexRuntimeV2ClientBuilder builder() {
        return new DefaultLexRuntimeV2ClientBuilder();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     * <p>
     * You can use this operation to restart a conversation with a bot. When you remove a session, the entire history of
     * the session is removed so that you can start again.
     * </p>
     * <p>
     * You don't need to delete a session. Sessions have a time limit and will expire. Set the session time limit when
     * you create the bot. The default is 5 minutes, but you can specify anything between 1 minute and 24 hours.
     * </p>
     * <p>
     * If you specify a bot or alias ID that doesn't exist, you receive a <code>BadRequestException.</code>
     * </p>
     * <p>
     * If the locale doesn't exist in the bot, or if the locale hasn't been enables for the alias, you receive a
     * <code>BadRequestException</code>.
     * </p>
     *
     * @param deleteSessionRequest
     * @return Result of the DeleteSession operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/DeleteSession" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteSessionResponse deleteSession(DeleteSessionRequest deleteSessionRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     * <p>
     * You can use this operation to restart a conversation with a bot. When you remove a session, the entire history of
     * the session is removed so that you can start again.
     * </p>
     * <p>
     * You don't need to delete a session. Sessions have a time limit and will expire. Set the session time limit when
     * you create the bot. The default is 5 minutes, but you can specify anything between 1 minute and 24 hours.
     * </p>
     * <p>
     * If you specify a bot or alias ID that doesn't exist, you receive a <code>BadRequestException.</code>
     * </p>
     * <p>
     * If the locale doesn't exist in the bot, or if the locale hasn't been enables for the alias, you receive a
     * <code>BadRequestException</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSessionRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteSessionRequest#builder()}
     * </p>
     *
     * @param deleteSessionRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSessionRequest.Builder} to create a request.
     * @return Result of the DeleteSession operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/DeleteSession" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteSessionResponse deleteSession(Consumer<DeleteSessionRequest.Builder> deleteSessionRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        return deleteSession(DeleteSessionRequest.builder().applyMutation(deleteSessionRequest).build());
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user.
     * </p>
     * <p>
     * For example, you can use this operation to retrieve session information for a user that has left a long-running
     * session in use.
     * </p>
     * <p>
     * If the bot, alias, or session identifier doesn't exist, Amazon Lex V2 returns a <code>BadRequestException</code>.
     * If the locale doesn't exist or is not enabled for the alias, you receive a <code>BadRequestException</code>.
     * </p>
     *
     * @param getSessionRequest
     * @return Result of the GetSession operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSessionResponse getSession(GetSessionRequest getSessionRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user.
     * </p>
     * <p>
     * For example, you can use this operation to retrieve session information for a user that has left a long-running
     * session in use.
     * </p>
     * <p>
     * If the bot, alias, or session identifier doesn't exist, Amazon Lex V2 returns a <code>BadRequestException</code>.
     * If the locale doesn't exist or is not enabled for the alias, you receive a <code>BadRequestException</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSessionRequest.Builder} avoiding the need to
     * create one manually via {@link GetSessionRequest#builder()}
     * </p>
     *
     * @param getSessionRequest
     *        A {@link Consumer} that will call methods on {@link GetSessionRequest.Builder} to create a request.
     * @return Result of the GetSession operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSessionResponse getSession(Consumer<GetSessionRequest.Builder> getSessionRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return getSession(GetSessionRequest.builder().applyMutation(getSessionRequest).build());
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled PutSessionResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT putSession(PutSessionRequest putSessionRequest,
            ResponseTransformer<PutSessionResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSessionRequest.Builder} avoiding the need to
     * create one manually via {@link PutSessionRequest#builder()}
     * </p>
     *
     * @param putSessionRequest
     *        A {@link Consumer} that will call methods on {@link PutSessionRequest.Builder} to create a request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled PutSessionResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT putSession(Consumer<PutSessionRequest.Builder> putSessionRequest,
            ResponseTransformer<PutSessionResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        return putSession(PutSessionRequest.builder().applyMutation(putSessionRequest).build(), responseTransformer);
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see #putSession(PutSessionRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSessionResponse putSession(PutSessionRequest putSessionRequest, Path destinationPath)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return putSession(putSessionRequest, ResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSessionRequest.Builder} avoiding the need to
     * create one manually via {@link PutSessionRequest#builder()}
     * </p>
     *
     * @param putSessionRequest
     *        A {@link Consumer} that will call methods on {@link PutSessionRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see #putSession(PutSessionRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSessionResponse putSession(Consumer<PutSessionRequest.Builder> putSessionRequest, Path destinationPath)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return putSession(PutSessionRequest.builder().applyMutation(putSessionRequest).build(), destinationPath);
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         If the requested content type was audio, the audio version of the message to convey to the user.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see #getObject(putSession, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<PutSessionResponse> putSession(PutSessionRequest putSessionRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        return putSession(putSessionRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSessionRequest.Builder} avoiding the need to
     * create one manually via {@link PutSessionRequest#builder()}
     * </p>
     *
     * @param putSessionRequest
     *        A {@link Consumer} that will call methods on {@link PutSessionRequest.Builder} to create a request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         If the requested content type was audio, the audio version of the message to convey to the user.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see #getObject(putSession, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<PutSessionResponse> putSession(Consumer<PutSessionRequest.Builder> putSessionRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return putSession(PutSessionRequest.builder().applyMutation(putSessionRequest).build());
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         If the requested content type was audio, the audio version of the message to convey to the user.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see #getObject(putSession, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<PutSessionResponse> putSessionAsBytes(PutSessionRequest putSessionRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return putSession(putSessionRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSessionRequest.Builder} avoiding the need to
     * create one manually via {@link PutSessionRequest#builder()}
     * </p>
     *
     * @param putSessionRequest
     *        A {@link Consumer} that will call methods on {@link PutSessionRequest.Builder} to create a request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         If the requested content type was audio, the audio version of the message to convey to the user.
     *         </p>
     *         '.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see #getObject(putSession, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<PutSessionResponse> putSessionAsBytes(Consumer<PutSessionRequest.Builder> putSessionRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return putSessionAsBytes(PutSessionRequest.builder().applyMutation(putSessionRequest).build());
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. Client applications use this API to send requests to Amazon Lex V2 at runtime.
     * Amazon Lex V2 then interprets the user input using the machine learning model that it build for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex V2 returns the next message to convey to the user and an optional response card to
     * display.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeTextRequest
     * @return Result of the RecognizeText operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeText" target="_top">AWS
     *      API Documentation</a>
     */
    default RecognizeTextResponse recognizeText(RecognizeTextRequest recognizeTextRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. Client applications use this API to send requests to Amazon Lex V2 at runtime.
     * Amazon Lex V2 then interprets the user input using the machine learning model that it build for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex V2 returns the next message to convey to the user and an optional response card to
     * display.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RecognizeTextRequest.Builder} avoiding the need to
     * create one manually via {@link RecognizeTextRequest#builder()}
     * </p>
     *
     * @param recognizeTextRequest
     *        A {@link Consumer} that will call methods on {@link RecognizeTextRequest.Builder} to create a request.
     * @return Result of the RecognizeText operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeText" target="_top">AWS
     *      API Documentation</a>
     */
    default RecognizeTextResponse recognizeText(Consumer<RecognizeTextRequest.Builder> recognizeTextRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return recognizeText(RecognizeTextRequest.builder().applyMutation(recognizeTextRequest).build());
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. You can send text or speech. Clients use this API to send text and audio
     * requests to Amazon Lex V2 at runtime. Amazon Lex V2 interprets the user input using the machine learning model
     * built for the bot.
     * </p>
     * <p>
     * The following request fields must be compressed with gzip and then base64 encoded before you send them to Amazon
     * Lex V2.
     * </p>
     * <ul>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following response fields are compressed using gzip and then base64 encoded by Amazon Lex V2. Before you can
     * use these fields, you must decode and decompress them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * inputTranscript
     * </p>
     * </li>
     * <li>
     * <p>
     * interpretations
     * </p>
     * </li>
     * <li>
     * <p>
     * messages
     * </p>
     * </li>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The example contains a Java application that compresses and encodes a Java object to send to Amazon Lex V2, and a
     * second that decodes and decompresses a response from Amazon Lex V2.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeUtteranceRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        RecognizeUtteranceResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex V2 did not understand the user intent, it sends the
     *        <code>clarificationPrompt</code> configured for the bot. If the intent requires confirmation before taking
     *        the fulfillment action, it sends the <code>confirmationPrompt</code>. Another example: Suppose that the
     *        Lambda function successfully fulfilled the intent, and sent a message to convey to the user. Then Amazon
     *        Lex V2 sends that message in the response.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT recognizeUtterance(RecognizeUtteranceRequest recognizeUtteranceRequest, RequestBody requestBody,
            ResponseTransformer<RecognizeUtteranceResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. You can send text or speech. Clients use this API to send text and audio
     * requests to Amazon Lex V2 at runtime. Amazon Lex V2 interprets the user input using the machine learning model
     * built for the bot.
     * </p>
     * <p>
     * The following request fields must be compressed with gzip and then base64 encoded before you send them to Amazon
     * Lex V2.
     * </p>
     * <ul>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following response fields are compressed using gzip and then base64 encoded by Amazon Lex V2. Before you can
     * use these fields, you must decode and decompress them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * inputTranscript
     * </p>
     * </li>
     * <li>
     * <p>
     * interpretations
     * </p>
     * </li>
     * <li>
     * <p>
     * messages
     * </p>
     * </li>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The example contains a Java application that compresses and encodes a Java object to send to Amazon Lex V2, and a
     * second that decodes and decompresses a response from Amazon Lex V2.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RecognizeUtteranceRequest.Builder} avoiding the
     * need to create one manually via {@link RecognizeUtteranceRequest#builder()}
     * </p>
     *
     * @param recognizeUtteranceRequest
     *        A {@link Consumer} that will call methods on {@link RecognizeUtteranceRequest.Builder} to create a
     *        request.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        RecognizeUtteranceResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex V2 did not understand the user intent, it sends the
     *        <code>clarificationPrompt</code> configured for the bot. If the intent requires confirmation before taking
     *        the fulfillment action, it sends the <code>confirmationPrompt</code>. Another example: Suppose that the
     *        Lambda function successfully fulfilled the intent, and sent a message to convey to the user. Then Amazon
     *        Lex V2 sends that message in the response.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT recognizeUtterance(Consumer<RecognizeUtteranceRequest.Builder> recognizeUtteranceRequest,
            RequestBody requestBody, ResponseTransformer<RecognizeUtteranceResponse, ReturnT> responseTransformer)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, ThrottlingException,
            InternalServerException, ConflictException, DependencyFailedException, BadGatewayException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        return recognizeUtterance(RecognizeUtteranceRequest.builder().applyMutation(recognizeUtteranceRequest).build(),
                requestBody, responseTransformer);
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. You can send text or speech. Clients use this API to send text and audio
     * requests to Amazon Lex V2 at runtime. Amazon Lex V2 interprets the user input using the machine learning model
     * built for the bot.
     * </p>
     * <p>
     * The following request fields must be compressed with gzip and then base64 encoded before you send them to Amazon
     * Lex V2.
     * </p>
     * <ul>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following response fields are compressed using gzip and then base64 encoded by Amazon Lex V2. Before you can
     * use these fields, you must decode and decompress them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * inputTranscript
     * </p>
     * </li>
     * <li>
     * <p>
     * interpretations
     * </p>
     * </li>
     * <li>
     * <p>
     * messages
     * </p>
     * </li>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The example contains a Java application that compresses and encodes a Java object to send to Amazon Lex V2, and a
     * second that decodes and decompresses a response from Amazon Lex V2.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeUtteranceRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex V2 did not understand the user intent, it sends the
     *        <code>clarificationPrompt</code> configured for the bot. If the intent requires confirmation before taking
     *        the fulfillment action, it sends the <code>confirmationPrompt</code>. Another example: Suppose that the
     *        Lambda function successfully fulfilled the intent, and sent a message to convey to the user. Then Amazon
     *        Lex V2 sends that message in the response.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeUtterance
     * @see #recognizeUtterance(RecognizeUtteranceRequest, RequestBody)
     * @see #recognizeUtterance(RecognizeUtteranceRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default RecognizeUtteranceResponse recognizeUtterance(RecognizeUtteranceRequest recognizeUtteranceRequest, Path sourcePath,
            Path destinationPath) throws AccessDeniedException, ResourceNotFoundException, ValidationException,
            ThrottlingException, InternalServerException, ConflictException, DependencyFailedException, BadGatewayException,
            AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        return recognizeUtterance(recognizeUtteranceRequest, RequestBody.fromFile(sourcePath),
                ResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. You can send text or speech. Clients use this API to send text and audio
     * requests to Amazon Lex V2 at runtime. Amazon Lex V2 interprets the user input using the machine learning model
     * built for the bot.
     * </p>
     * <p>
     * The following request fields must be compressed with gzip and then base64 encoded before you send them to Amazon
     * Lex V2.
     * </p>
     * <ul>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following response fields are compressed using gzip and then base64 encoded by Amazon Lex V2. Before you can
     * use these fields, you must decode and decompress them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * inputTranscript
     * </p>
     * </li>
     * <li>
     * <p>
     * interpretations
     * </p>
     * </li>
     * <li>
     * <p>
     * messages
     * </p>
     * </li>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The example contains a Java application that compresses and encodes a Java object to send to Amazon Lex V2, and a
     * second that decodes and decompresses a response from Amazon Lex V2.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RecognizeUtteranceRequest.Builder} avoiding the
     * need to create one manually via {@link RecognizeUtteranceRequest#builder()}
     * </p>
     *
     * @param recognizeUtteranceRequest
     *        A {@link Consumer} that will call methods on {@link RecognizeUtteranceRequest.Builder} to create a
     *        request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex V2 did not understand the user intent, it sends the
     *        <code>clarificationPrompt</code> configured for the bot. If the intent requires confirmation before taking
     *        the fulfillment action, it sends the <code>confirmationPrompt</code>. Another example: Suppose that the
     *        Lambda function successfully fulfilled the intent, and sent a message to convey to the user. Then Amazon
     *        Lex V2 sends that message in the response.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeUtterance
     * @see #recognizeUtterance(RecognizeUtteranceRequest, RequestBody)
     * @see #recognizeUtterance(RecognizeUtteranceRequest, ResponseTransformer)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    default RecognizeUtteranceResponse recognizeUtterance(Consumer<RecognizeUtteranceRequest.Builder> recognizeUtteranceRequest,
            Path sourcePath, Path destinationPath) throws AccessDeniedException, ResourceNotFoundException, ValidationException,
            ThrottlingException, InternalServerException, ConflictException, DependencyFailedException, BadGatewayException,
            AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        return recognizeUtterance(RecognizeUtteranceRequest.builder().applyMutation(recognizeUtteranceRequest).build(),
                sourcePath, destinationPath);
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
