/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.auth.signer.EventStreamAws4Signer;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.client.handler.AwsClientHandlerUtils;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionJsonMarshaller;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.runtime.transform.AsyncStreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lexruntimev2.internal.LexRuntimeV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.lexruntimev2.model.AccessDeniedException;
import software.amazon.awssdk.services.lexruntimev2.model.BadGatewayException;
import software.amazon.awssdk.services.lexruntimev2.model.ConflictException;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.DependencyFailedException;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.InternalServerException;
import software.amazon.awssdk.services.lexruntimev2.model.LexRuntimeV2Exception;
import software.amazon.awssdk.services.lexruntimev2.model.LexRuntimeV2Request;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceResponse;
import software.amazon.awssdk.services.lexruntimev2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationRequest;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationRequestEventStream;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationResponse;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationResponseEventStream;
import software.amazon.awssdk.services.lexruntimev2.model.StartConversationResponseHandler;
import software.amazon.awssdk.services.lexruntimev2.model.ThrottlingException;
import software.amazon.awssdk.services.lexruntimev2.model.ValidationException;
import software.amazon.awssdk.services.lexruntimev2.model.startconversationrequesteventstream.DefaultAudioInputEvent;
import software.amazon.awssdk.services.lexruntimev2.model.startconversationrequesteventstream.DefaultConfigurationEvent;
import software.amazon.awssdk.services.lexruntimev2.model.startconversationrequesteventstream.DefaultDTMFInputEvent;
import software.amazon.awssdk.services.lexruntimev2.model.startconversationrequesteventstream.DefaultDisconnectionEvent;
import software.amazon.awssdk.services.lexruntimev2.model.startconversationrequesteventstream.DefaultPlaybackCompletionEvent;
import software.amazon.awssdk.services.lexruntimev2.model.startconversationrequesteventstream.DefaultTextInputEvent;
import software.amazon.awssdk.services.lexruntimev2.transform.AudioInputEventMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.ConfigurationEventMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.DTMFInputEventMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.DeleteSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.DisconnectionEventMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.PlaybackCompletionEventMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.PutSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.RecognizeTextRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.RecognizeUtteranceRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.StartConversationRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.TextInputEventMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link LexRuntimeV2AsyncClient}.
 *
 * @see LexRuntimeV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLexRuntimeV2AsyncClient implements LexRuntimeV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLexRuntimeV2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LexRuntimeV2ServiceClientConfiguration serviceClientConfiguration;

    private final Executor executor;

    protected DefaultLexRuntimeV2AsyncClient(LexRuntimeV2ServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     * <p>
     * You can use this operation to restart a conversation with a bot. When you remove a session, the entire history of
     * the session is removed so that you can start again.
     * </p>
     * <p>
     * You don't need to delete a session. Sessions have a time limit and will expire. Set the session time limit when
     * you create the bot. The default is 5 minutes, but you can specify anything between 1 minute and 24 hours.
     * </p>
     * <p>
     * If you specify a bot or alias ID that doesn't exist, you receive a <code>BadRequestException.</code>
     * </p>
     * <p>
     * If the locale doesn't exist in the bot, or if the locale hasn't been enables for the alias, you receive a
     * <code>BadRequestException</code>.
     * </p>
     *
     * @param deleteSessionRequest
     * @return A Java Future containing the result of the DeleteSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/DeleteSession" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSessionResponse> deleteSession(DeleteSessionRequest deleteSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSessionRequest, DeleteSessionResponse>()
                            .withOperationName("DeleteSession")
                            .withMarshaller(new DeleteSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSessionRequest));
            CompletableFuture<DeleteSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user.
     * </p>
     * <p>
     * For example, you can use this operation to retrieve session information for a user that has left a long-running
     * session in use.
     * </p>
     * <p>
     * If the bot, alias, or session identifier doesn't exist, Amazon Lex V2 returns a <code>BadRequestException</code>.
     * If the locale doesn't exist or is not enabled for the alias, you receive a <code>BadRequestException</code>.
     * </p>
     *
     * @param getSessionRequest
     * @return A Java Future containing the result of the GetSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSessionResponse> getSession(GetSessionRequest getSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>().withOperationName("GetSession")
                            .withMarshaller(new GetSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSessionRequest));
            CompletableFuture<GetSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> putSession(PutSessionRequest putSessionRequest,
            AsyncResponseTransformer<PutSessionResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSession");
            Pair<AsyncResponseTransformer<PutSessionResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<PutSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<PutSessionRequest, PutSessionResponse>().withOperationName("PutSession")
                            .withMarshaller(new PutSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putSessionRequest), asyncResponseTransformer);
            AsyncResponseTransformer<PutSessionResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<PutSessionResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. Client applications use this API to send requests to Amazon Lex V2 at runtime.
     * Amazon Lex V2 then interprets the user input using the machine learning model that it build for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex V2 returns the next message to convey to the user and an optional response card to
     * display.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeTextRequest
     * @return A Java Future containing the result of the RecognizeText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeText" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RecognizeTextResponse> recognizeText(RecognizeTextRequest recognizeTextRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(recognizeTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, recognizeTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecognizeText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RecognizeTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RecognizeTextResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RecognizeTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RecognizeTextRequest, RecognizeTextResponse>()
                            .withOperationName("RecognizeText")
                            .withMarshaller(new RecognizeTextRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(recognizeTextRequest));
            CompletableFuture<RecognizeTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. You can send text or speech. Clients use this API to send text and audio
     * requests to Amazon Lex V2 at runtime. Amazon Lex V2 interprets the user input using the machine learning model
     * built for the bot.
     * </p>
     * <p>
     * The following request fields must be compressed with gzip and then base64 encoded before you send them to Amazon
     * Lex V2.
     * </p>
     * <ul>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following response fields are compressed using gzip and then base64 encoded by Amazon Lex V2. Before you can
     * use these fields, you must decode and decompress them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * inputTranscript
     * </p>
     * </li>
     * <li>
     * <p>
     * interpretations
     * </p>
     * </li>
     * <li>
     * <p>
     * messages
     * </p>
     * </li>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The example contains a Java application that compresses and encodes a Java object to send to Amazon Lex V2, and a
     * second that decodes and decompresses a response from Amazon Lex V2.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeUtteranceRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex V2 did not understand the user intent, it sends the
     *        <code>clarificationPrompt</code> configured for the bot. If the intent requires confirmation before taking
     *        the fulfillment action, it sends the <code>confirmationPrompt</code>. Another example: Suppose that the
     *        Lambda function successfully fulfilled the intent, and sent a message to convey to the user. Then Amazon
     *        Lex V2 sends that message in the response.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>ConflictException</li>
     *         <li>DependencyFailedException</li>
     *         <li>BadGatewayException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> recognizeUtterance(RecognizeUtteranceRequest recognizeUtteranceRequest,
            AsyncRequestBody requestBody, AsyncResponseTransformer<RecognizeUtteranceResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(recognizeUtteranceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, recognizeUtteranceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecognizeUtterance");
            Pair<AsyncResponseTransformer<RecognizeUtteranceResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            recognizeUtteranceRequest = applySignerOverride(recognizeUtteranceRequest, Aws4UnsignedPayloadSigner.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<RecognizeUtteranceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RecognizeUtteranceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<RecognizeUtteranceRequest, RecognizeUtteranceResponse>()
                            .withOperationName("RecognizeUtterance")
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new RecognizeUtteranceRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).transferEncoding(true).useHttp2(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncRequestBody(requestBody).withInput(recognizeUtteranceRequest), asyncResponseTransformer);
            AsyncResponseTransformer<RecognizeUtteranceResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<RecognizeUtteranceResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an HTTP/2 bidirectional event stream that enables you to send audio, text, or DTMF input in real time.
     * After your application starts a conversation, users send input to Amazon Lex V2 as a stream of events. Amazon Lex
     * V2 processes the incoming events and responds with streaming text or audio events.
     * </p>
     * <p>
     * Audio input must be in the following format:
     * <code>audio/lpcm sample-rate=8000 sample-size-bits=16 channel-count=1; is-big-endian=false</code>.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     * <p>
     * If the optional update message is configured, it is played at the specified frequency while the Lambda function
     * is running and the update message state is active. If the fulfillment update message is not active, the Lambda
     * function runs with a 30 second timeout.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-update.html">Update message
     * </a>
     * </p>
     * <p>
     * The <code>StartConversation</code> operation is supported only in the following SDKs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/goto/SdkForCpp/runtime.lex.v2-2020-08-07/StartConversation">AWS SDK for
     * C++</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/goto/SdkForJavaV2/runtime.lex.v2-2020-08-07/StartConversation">AWS SDK for
     * Java V2</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/goto/SdkForRubyV3/runtime.lex.v2-2020-08-07/StartConversation">AWS SDK for
     * Ruby V3</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startConversationRequest
     * @return A Java Future containing the result of the StartConversation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>ThrottlingException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeV2AsyncClient.StartConversation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/StartConversation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startConversation(StartConversationRequest startConversationRequest,
            Publisher<StartConversationRequestEventStream> requestStream, StartConversationResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startConversationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startConversationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartConversation");
            startConversationRequest = applySignerOverride(startConversationRequest, EventStreamAws4Signer.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartConversationResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartConversationResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends StartConversationResponseEventStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier
                                    .builder()
                                    .putSdkPojoSupplier("PlaybackInterruptionEvent",
                                            StartConversationResponseEventStream::playbackInterruptionEventBuilder)
                                    .putSdkPojoSupplier("TranscriptEvent",
                                            StartConversationResponseEventStream::transcriptEventBuilder)
                                    .putSdkPojoSupplier("IntentResultEvent",
                                            StartConversationResponseEventStream::intentResultEventBuilder)
                                    .putSdkPojoSupplier("TextResponseEvent",
                                            StartConversationResponseEventStream::textResponseEventBuilder)
                                    .putSdkPojoSupplier("AudioResponseEvent",
                                            StartConversationResponseEventStream::audioResponseEventBuilder)
                                    .putSdkPojoSupplier("HeartbeatEvent",
                                            StartConversationResponseEventStream::heartbeatEventBuilder)
                                    .defaultSdkPojoSupplier(
                                            () -> new SdkPojoBuilder(StartConversationResponseEventStream.UNKNOWN)).build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultAudioInputEvent.class, new AudioInputEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultDTMFInputEvent.class, new DTMFInputEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultTextInputEvent.class, new TextInputEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultPlaybackCompletionEvent.class, new PlaybackCompletionEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultDisconnectionEvent.class, new DisconnectionEventMarshaller(protocolFactory)).build();
            SdkPublisher<StartConversationRequestEventStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartConversationResponse, StartConversationResponseEventStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartConversationResponse, StartConversationResponseEventStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartConversationResponse, StartConversationResponseEventStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartConversationResponse, StartConversationResponseEventStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<StartConversationRequest, StartConversationResponse>()
                            .withOperationName("StartConversation")
                            .withMarshaller(new StartConversationRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startConversationRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LexRuntimeV2ServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LexRuntimeV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .contentType("application/json")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LexRuntimeV2Request> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private static boolean isSignerOverridden(SdkClientConfiguration clientConfiguration) {
        return Boolean.TRUE.equals(clientConfiguration.option(SdkClientOption.SIGNER_OVERRIDDEN));
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        LexRuntimeV2ServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = LexRuntimeV2ServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
