/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The current intent that Amazon Lex V2 is attempting to fulfill.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Intent implements SdkPojo, Serializable, ToCopyableBuilder<Intent.Builder, Intent> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Intent::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Map<String, Slot>> SLOTS_FIELD = SdkField
            .<Map<String, Slot>> builder(MarshallingType.MAP)
            .memberName("slots")
            .getter(getter(Intent::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Slot> builder(MarshallingType.SDK_POJO)
                                            .constructor(Slot::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Intent::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> CONFIRMATION_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("confirmationState").getter(getter(Intent::confirmationStateAsString))
            .setter(setter(Builder::confirmationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confirmationState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SLOTS_FIELD,
            STATE_FIELD, CONFIRMATION_STATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Map<String, Slot> slots;

    private final String state;

    private final String confirmationState;

    private Intent(BuilderImpl builder) {
        this.name = builder.name;
        this.slots = builder.slots;
        this.state = builder.state;
        this.confirmationState = builder.confirmationState;
    }

    /**
     * <p>
     * The name of the intent.
     * </p>
     * 
     * @return The name of the intent.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Slots property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasSlots() {
        return slots != null && !(slots instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a slot has not
     * been filled, the value is null.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSlots} method.
     * </p>
     * 
     * @return A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a slot
     *         has not been filled, the value is null.
     */
    public final Map<String, Slot> slots() {
        return slots;
    }

    /**
     * <p>
     * Indicates the fulfillment state for the intent. The meanings of each value are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> – The bot failed to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to fulfill the
     * intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is ready to
     * fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming conversations).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IntentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Indicates the fulfillment state for the intent. The meanings of each value are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> – The bot failed to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to
     *         fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is ready
     *         to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming
     *         conversations).
     *         </p>
     *         </li>
     * @see IntentState
     */
    public final IntentState state() {
        return IntentState.fromValue(state);
    }

    /**
     * <p>
     * Indicates the fulfillment state for the intent. The meanings of each value are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> – The bot failed to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to fulfill the
     * intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is ready to
     * fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming conversations).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IntentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Indicates the fulfillment state for the intent. The meanings of each value are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> – The bot failed to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to
     *         fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is ready
     *         to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming
     *         conversations).
     *         </p>
     *         </li>
     * @see IntentState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or <code>None</code> if the
     * confirmation stage has not yet been reached.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confirmationState}
     * will return {@link ConfirmationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #confirmationStateAsString}.
     * </p>
     * 
     * @return Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or <code>None</code>
     *         if the confirmation stage has not yet been reached.
     * @see ConfirmationState
     */
    public final ConfirmationState confirmationState() {
        return ConfirmationState.fromValue(confirmationState);
    }

    /**
     * <p>
     * Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or <code>None</code> if the
     * confirmation stage has not yet been reached.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confirmationState}
     * will return {@link ConfirmationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #confirmationStateAsString}.
     * </p>
     * 
     * @return Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or <code>None</code>
     *         if the confirmation stage has not yet been reached.
     * @see ConfirmationState
     */
    public final String confirmationStateAsString() {
        return confirmationState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasSlots() ? slots() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Intent)) {
            return false;
        }
        Intent other = (Intent) obj;
        return Objects.equals(name(), other.name()) && hasSlots() == other.hasSlots() && Objects.equals(slots(), other.slots())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(confirmationStateAsString(), other.confirmationStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Intent").add("Name", name()).add("Slots", hasSlots() ? slots() : null)
                .add("State", stateAsString()).add("ConfirmationState", confirmationStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "confirmationState":
            return Optional.ofNullable(clazz.cast(confirmationStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("slots", SLOTS_FIELD);
        map.put("state", STATE_FIELD);
        map.put("confirmationState", CONFIRMATION_STATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Intent, T> g) {
        return obj -> g.apply((Intent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Intent> {
        /**
         * <p>
         * The name of the intent.
         * </p>
         * 
         * @param name
         *        The name of the intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a slot has
         * not been filled, the value is null.
         * </p>
         * 
         * @param slots
         *        A map of all of the slots for the intent. The name of the slot maps to the value of the slot. If a
         *        slot has not been filled, the value is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, Slot> slots);

        /**
         * <p>
         * Indicates the fulfillment state for the intent. The meanings of each value are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> – The bot failed to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to fulfill
         * the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is ready to
         * fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming conversations).
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        Indicates the fulfillment state for the intent. The meanings of each value are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> – The bot failed to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to
         *        fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is
         *        ready to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming
         *        conversations).
         *        </p>
         *        </li>
         * @see IntentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntentState
         */
        Builder state(String state);

        /**
         * <p>
         * Indicates the fulfillment state for the intent. The meanings of each value are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> – The bot failed to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to fulfill
         * the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is ready to
         * fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming conversations).
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        Indicates the fulfillment state for the intent. The meanings of each value are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> – The bot failed to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> – The bot has completed fulfillment of the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FulfillmentInProgress</code> – The bot is in the middle of fulfilling the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InProgress</code> – The bot is in the middle of eliciting the slot values that are necessary to
         *        fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> – The bot has elicited all the slot values for the intent and is
         *        ready to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Waiting</code> – The bot is waiting for a response from the user (limited to streaming
         *        conversations).
         *        </p>
         *        </li>
         * @see IntentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntentState
         */
        Builder state(IntentState state);

        /**
         * <p>
         * Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or <code>None</code> if
         * the confirmation stage has not yet been reached.
         * </p>
         * 
         * @param confirmationState
         *        Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or
         *        <code>None</code> if the confirmation stage has not yet been reached.
         * @see ConfirmationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationState
         */
        Builder confirmationState(String confirmationState);

        /**
         * <p>
         * Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or <code>None</code> if
         * the confirmation stage has not yet been reached.
         * </p>
         * 
         * @param confirmationState
         *        Indicates whether the intent has been <code>Confirmed</code>, <code>Denied</code>, or
         *        <code>None</code> if the confirmation stage has not yet been reached.
         * @see ConfirmationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationState
         */
        Builder confirmationState(ConfirmationState confirmationState);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Map<String, Slot> slots = DefaultSdkAutoConstructMap.getInstance();

        private String state;

        private String confirmationState;

        private BuilderImpl() {
        }

        private BuilderImpl(Intent model) {
            name(model.name);
            slots(model.slots);
            state(model.state);
            confirmationState(model.confirmationState);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, Slot.Builder> getSlots() {
            Map<String, Slot.Builder> result = SlotsCopier.copyToBuilder(this.slots);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setSlots(Map<String, Slot.BuilderImpl> slots) {
            this.slots = SlotsCopier.copyFromBuilder(slots);
        }

        @Override
        public final Builder slots(Map<String, Slot> slots) {
            this.slots = SlotsCopier.copy(slots);
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IntentState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getConfirmationState() {
            return confirmationState;
        }

        public final void setConfirmationState(String confirmationState) {
            this.confirmationState = confirmationState;
        }

        @Override
        public final Builder confirmationState(String confirmationState) {
            this.confirmationState = confirmationState;
            return this;
        }

        @Override
        public final Builder confirmationState(ConfirmationState confirmationState) {
            this.confirmationState(confirmationState == null ? null : confirmationState.toString());
            return this;
        }

        @Override
        public Intent build() {
            return new Intent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
