/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanager;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanager.model.AccessDeniedException;
import software.amazon.awssdk.services.licensemanager.model.AuthorizationException;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.FailedDependencyException;
import software.amazon.awssdk.services.licensemanager.model.FilterLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.model.InvalidParameterValueException;
import software.amazon.awssdk.services.licensemanager.model.InvalidResourceStateException;
import software.amazon.awssdk.services.licensemanager.model.LicenseManagerException;
import software.amazon.awssdk.services.licensemanager.model.LicenseUsageException;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryRequest;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryResponse;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.RateLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.ServerInternalException;
import software.amazon.awssdk.services.licensemanager.model.TagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.TagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListAssociationsForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListFailuresForLicenseConfigurationOperationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListResourceInventoryRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListUsageForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateServiceSettingsRequestMarshaller;

/**
 * Internal implementation of {@link LicenseManagerClient}.
 *
 * @see LicenseManagerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerClient implements LicenseManagerClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLicenseManagerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a license configuration.
     * </p>
     * <p>
     * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
     * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
     * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), host affinity (how
     * long a VM must be associated with a host), and the number of licenses purchased and used.
     * </p>
     *
     * @param createLicenseConfigurationRequest
     * @return Result of the CreateLicenseConfiguration operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws ResourceLimitExceededException
     *         Your resource limits have been exceeded.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.CreateLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateLicenseConfigurationResponse createLicenseConfiguration(
            CreateLicenseConfigurationRequest createLicenseConfigurationRequest) throws InvalidParameterValueException,
            ServerInternalException, ResourceLimitExceededException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateLicenseConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateLicenseConfigurationRequest, CreateLicenseConfigurationResponse>()
                        .withOperationName("CreateLicenseConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createLicenseConfigurationRequest)
                        .withMarshaller(new CreateLicenseConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified license configuration.
     * </p>
     * <p>
     * You cannot delete a license configuration that is in use.
     * </p>
     *
     * @param deleteLicenseConfigurationRequest
     * @return Result of the DeleteLicenseConfiguration operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.DeleteLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLicenseConfigurationResponse deleteLicenseConfiguration(
            DeleteLicenseConfigurationRequest deleteLicenseConfigurationRequest) throws InvalidParameterValueException,
            ServerInternalException, AuthorizationException, AccessDeniedException, RateLimitExceededException,
            AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteLicenseConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteLicenseConfigurationRequest, DeleteLicenseConfigurationResponse>()
                        .withOperationName("DeleteLicenseConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteLicenseConfigurationRequest)
                        .withMarshaller(new DeleteLicenseConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets detailed information about the specified license configuration.
     * </p>
     *
     * @param getLicenseConfigurationRequest
     * @return Result of the GetLicenseConfiguration operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.GetLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetLicenseConfigurationResponse getLicenseConfiguration(GetLicenseConfigurationRequest getLicenseConfigurationRequest)
            throws InvalidParameterValueException, ServerInternalException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLicenseConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetLicenseConfigurationRequest, GetLicenseConfigurationResponse>()
                .withOperationName("GetLicenseConfiguration").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getLicenseConfigurationRequest)
                .withMarshaller(new GetLicenseConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the License Manager settings for the current Region.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return Result of the GetServiceSettings operation returned by the service.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.GetServiceSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServiceSettingsResponse getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest)
            throws ServerInternalException, AuthorizationException, AccessDeniedException, RateLimitExceededException,
            AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetServiceSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                .withOperationName("GetServiceSettings").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getServiceSettingsRequest)
                .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the resource associations for the specified license configuration.
     * </p>
     * <p>
     * Resource associations need not consume licenses from a license configuration. For example, an AMI or a stopped
     * instance might not consume a license (depending on the license rules).
     * </p>
     *
     * @param listAssociationsForLicenseConfigurationRequest
     * @return Result of the ListAssociationsForLicenseConfiguration operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws FilterLimitExceededException
     *         The request uses too many filters or too many filter values.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListAssociationsForLicenseConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListAssociationsForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociationsForLicenseConfigurationResponse listAssociationsForLicenseConfiguration(
            ListAssociationsForLicenseConfigurationRequest listAssociationsForLicenseConfigurationRequest)
            throws InvalidParameterValueException, FilterLimitExceededException, ServerInternalException, AuthorizationException,
            AccessDeniedException, RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssociationsForLicenseConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListAssociationsForLicenseConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListAssociationsForLicenseConfigurationRequest, ListAssociationsForLicenseConfigurationResponse>()
                        .withOperationName("ListAssociationsForLicenseConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listAssociationsForLicenseConfigurationRequest)
                        .withMarshaller(new ListAssociationsForLicenseConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the license configuration operations that failed.
     * </p>
     *
     * @param listFailuresForLicenseConfigurationOperationsRequest
     * @return Result of the ListFailuresForLicenseConfigurationOperations operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListFailuresForLicenseConfigurationOperations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListFailuresForLicenseConfigurationOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListFailuresForLicenseConfigurationOperationsResponse listFailuresForLicenseConfigurationOperations(
            ListFailuresForLicenseConfigurationOperationsRequest listFailuresForLicenseConfigurationOperationsRequest)
            throws InvalidParameterValueException, ServerInternalException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFailuresForLicenseConfigurationOperationsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListFailuresForLicenseConfigurationOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListFailuresForLicenseConfigurationOperationsRequest, ListFailuresForLicenseConfigurationOperationsResponse>()
                        .withOperationName("ListFailuresForLicenseConfigurationOperations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(listFailuresForLicenseConfigurationOperationsRequest)
                        .withMarshaller(new ListFailuresForLicenseConfigurationOperationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the license configurations for your account.
     * </p>
     *
     * @param listLicenseConfigurationsRequest
     * @return Result of the ListLicenseConfigurations operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws FilterLimitExceededException
     *         The request uses too many filters or too many filter values.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListLicenseConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLicenseConfigurationsResponse listLicenseConfigurations(
            ListLicenseConfigurationsRequest listLicenseConfigurationsRequest) throws InvalidParameterValueException,
            ServerInternalException, FilterLimitExceededException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLicenseConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListLicenseConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListLicenseConfigurationsRequest, ListLicenseConfigurationsResponse>()
                        .withOperationName("ListLicenseConfigurations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listLicenseConfigurationsRequest)
                        .withMarshaller(new ListLicenseConfigurationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the license configurations for the specified resource.
     * </p>
     *
     * @param listLicenseSpecificationsForResourceRequest
     * @return Result of the ListLicenseSpecificationsForResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListLicenseSpecificationsForResource
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLicenseSpecificationsForResourceResponse listLicenseSpecificationsForResource(
            ListLicenseSpecificationsForResourceRequest listLicenseSpecificationsForResourceRequest)
            throws InvalidParameterValueException, ServerInternalException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListLicenseSpecificationsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListLicenseSpecificationsForResourceRequest, ListLicenseSpecificationsForResourceResponse>()
                        .withOperationName("ListLicenseSpecificationsForResource").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listLicenseSpecificationsForResourceRequest)
                        .withMarshaller(new ListLicenseSpecificationsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists resources managed using Systems Manager inventory.
     * </p>
     *
     * @param listResourceInventoryRequest
     * @return Result of the ListResourceInventory operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws FilterLimitExceededException
     *         The request uses too many filters or too many filter values.
     * @throws FailedDependencyException
     *         A dependency required to run the API is missing.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListResourceInventory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListResourceInventory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceInventoryResponse listResourceInventory(ListResourceInventoryRequest listResourceInventoryRequest)
            throws InvalidParameterValueException, ServerInternalException, FilterLimitExceededException,
            FailedDependencyException, AuthorizationException, AccessDeniedException, RateLimitExceededException,
            AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceInventoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourceInventoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListResourceInventoryRequest, ListResourceInventoryResponse>()
                .withOperationName("ListResourceInventory").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listResourceInventoryRequest)
                .withMarshaller(new ListResourceInventoryRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the tags for the specified license configuration.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterValueException, ServerInternalException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all license usage records for a license configuration, displaying license consumption details by resource
     * at a selected point in time. Use this action to audit the current license consumption for any license inventory
     * and configuration.
     * </p>
     *
     * @param listUsageForLicenseConfigurationRequest
     * @return Result of the ListUsageForLicenseConfiguration operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws FilterLimitExceededException
     *         The request uses too many filters or too many filter values.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.ListUsageForLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListUsageForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListUsageForLicenseConfigurationResponse listUsageForLicenseConfiguration(
            ListUsageForLicenseConfigurationRequest listUsageForLicenseConfigurationRequest)
            throws InvalidParameterValueException, FilterLimitExceededException, ServerInternalException, AuthorizationException,
            AccessDeniedException, RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsageForLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListUsageForLicenseConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListUsageForLicenseConfigurationRequest, ListUsageForLicenseConfigurationResponse>()
                        .withOperationName("ListUsageForLicenseConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listUsageForLicenseConfigurationRequest)
                        .withMarshaller(new ListUsageForLicenseConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds the specified tags to the specified license configuration.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterValueException,
            ServerInternalException, AuthorizationException, AccessDeniedException, RateLimitExceededException,
            AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the specified tags from the specified license configuration.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterValueException,
            ServerInternalException, AuthorizationException, AccessDeniedException, RateLimitExceededException,
            AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Modifies the attributes of an existing license configuration.
     * </p>
     * <p>
     * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
     * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
     * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), host affinity (how
     * long a VM must be associated with a host), and the number of licenses purchased and used.
     * </p>
     *
     * @param updateLicenseConfigurationRequest
     * @return Result of the UpdateLicenseConfiguration operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.UpdateLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLicenseConfigurationResponse updateLicenseConfiguration(
            UpdateLicenseConfigurationRequest updateLicenseConfigurationRequest) throws InvalidParameterValueException,
            ServerInternalException, AuthorizationException, AccessDeniedException, RateLimitExceededException,
            AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateLicenseConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateLicenseConfigurationRequest, UpdateLicenseConfigurationResponse>()
                        .withOperationName("UpdateLicenseConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateLicenseConfigurationRequest)
                        .withMarshaller(new UpdateLicenseConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds or removes the specified license configurations for the specified AWS resource.
     * </p>
     * <p>
     * You can update the license specifications of AMIs, instances, and hosts. You cannot update the license
     * specifications for launch templates and AWS CloudFormation templates, as they send license configurations to the
     * operation that creates the resource.
     * </p>
     *
     * @param updateLicenseSpecificationsForResourceRequest
     * @return Result of the UpdateLicenseSpecificationsForResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws InvalidResourceStateException
     *         License Manager cannot allocate a license to a resource because of its state. </p>
     *         <p>
     *         For example, you cannot allocate a license to an instance in the process of shutting down.
     * @throws LicenseUsageException
     *         You do not have enough licenses available to support a new resource launch.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.UpdateLicenseSpecificationsForResource
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLicenseSpecificationsForResourceResponse updateLicenseSpecificationsForResource(
            UpdateLicenseSpecificationsForResourceRequest updateLicenseSpecificationsForResourceRequest)
            throws InvalidParameterValueException, InvalidResourceStateException, LicenseUsageException, ServerInternalException,
            AuthorizationException, AccessDeniedException, RateLimitExceededException, AwsServiceException, SdkClientException,
            LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, UpdateLicenseSpecificationsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateLicenseSpecificationsForResourceRequest, UpdateLicenseSpecificationsForResourceResponse>()
                        .withOperationName("UpdateLicenseSpecificationsForResource").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateLicenseSpecificationsForResourceRequest)
                        .withMarshaller(new UpdateLicenseSpecificationsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates License Manager settings for the current Region.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return Result of the UpdateServiceSettings operation returned by the service.
     * @throws InvalidParameterValueException
     *         One or more parameter values are not valid.
     * @throws ServerInternalException
     *         The server experienced an internal error. Try again.
     * @throws AuthorizationException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws AccessDeniedException
     *         Access to resource denied.
     * @throws RateLimitExceededException
     *         Too many requests have been submitted. Try again after a brief wait.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerClient.UpdateServiceSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServiceSettingsResponse updateServiceSettings(UpdateServiceSettingsRequest updateServiceSettingsRequest)
            throws InvalidParameterValueException, ServerInternalException, AuthorizationException, AccessDeniedException,
            RateLimitExceededException, AwsServiceException, SdkClientException, LicenseManagerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServiceSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                .withOperationName("UpdateServiceSettings").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateServiceSettingsRequest)
                .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedDependencyException")
                                .exceptionBuilderSupplier(FailedDependencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationException")
                                .exceptionBuilderSupplier(AuthorizationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RateLimitExceededException")
                                .exceptionBuilderSupplier(RateLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterLimitExceededException")
                                .exceptionBuilderSupplier(FilterLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LicenseUsageException")
                                .exceptionBuilderSupplier(LicenseUsageException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
