/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanager;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanager.model.AccessDeniedException;
import software.amazon.awssdk.services.licensemanager.model.AuthorizationException;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.FailedDependencyException;
import software.amazon.awssdk.services.licensemanager.model.FilterLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.model.InvalidParameterValueException;
import software.amazon.awssdk.services.licensemanager.model.InvalidResourceStateException;
import software.amazon.awssdk.services.licensemanager.model.LicenseManagerException;
import software.amazon.awssdk.services.licensemanager.model.LicenseUsageException;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryRequest;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryResponse;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.RateLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.ServerInternalException;
import software.amazon.awssdk.services.licensemanager.model.TagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.TagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListAssociationsForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListFailuresForLicenseConfigurationOperationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListResourceInventoryRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListUsageForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LicenseManagerAsyncClient}.
 *
 * @see LicenseManagerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerAsyncClient implements LicenseManagerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLicenseManagerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLicenseManagerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a license configuration.
     * </p>
     * <p>
     * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
     * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
     * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), host affinity (how
     * long a VM must be associated with a host), and the number of licenses purchased and used.
     * </p>
     *
     * @param createLicenseConfigurationRequest
     * @return A Java Future containing the result of the CreateLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseConfigurationResponse> createLicenseConfiguration(
            CreateLicenseConfigurationRequest createLicenseConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseConfigurationRequest, CreateLicenseConfigurationResponse>()
                            .withOperationName("CreateLicenseConfiguration")
                            .withMarshaller(new CreateLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    createLicenseConfigurationRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified license configuration.
     * </p>
     * <p>
     * You cannot delete a license configuration that is in use.
     * </p>
     *
     * @param deleteLicenseConfigurationRequest
     * @return A Java Future containing the result of the DeleteLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLicenseConfigurationResponse> deleteLicenseConfiguration(
            DeleteLicenseConfigurationRequest deleteLicenseConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLicenseConfigurationRequest, DeleteLicenseConfigurationResponse>()
                            .withOperationName("DeleteLicenseConfiguration")
                            .withMarshaller(new DeleteLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    deleteLicenseConfigurationRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified license configuration.
     * </p>
     *
     * @param getLicenseConfigurationRequest
     * @return A Java Future containing the result of the GetLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseConfigurationResponse> getLicenseConfiguration(
            GetLicenseConfigurationRequest getLicenseConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseConfigurationRequest, GetLicenseConfigurationResponse>()
                            .withOperationName("GetLicenseConfiguration")
                            .withMarshaller(new GetLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseConfigurationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the License Manager settings for the current Region.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return A Java Future containing the result of the GetServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetServiceSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceSettingsResponse> getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                            .withOperationName("GetServiceSettings")
                            .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getServiceSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getServiceSettingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource associations for the specified license configuration.
     * </p>
     * <p>
     * Resource associations need not consume licenses from a license configuration. For example, an AMI or a stopped
     * instance might not consume a license (depending on the license rules).
     * </p>
     *
     * @param listAssociationsForLicenseConfigurationRequest
     * @return A Java Future containing the result of the ListAssociationsForLicenseConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListAssociationsForLicenseConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListAssociationsForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociationsForLicenseConfigurationResponse> listAssociationsForLicenseConfiguration(
            ListAssociationsForLicenseConfigurationRequest listAssociationsForLicenseConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociationsForLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociationsForLicenseConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListAssociationsForLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssociationsForLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociationsForLicenseConfigurationRequest, ListAssociationsForLicenseConfigurationResponse>()
                            .withOperationName("ListAssociationsForLicenseConfiguration")
                            .withMarshaller(new ListAssociationsForLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociationsForLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAssociationsForLicenseConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListAssociationsForLicenseConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                                requestOverrideConfig);
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listAssociationsForLicenseConfigurationRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license configuration operations that failed.
     * </p>
     *
     * @param listFailuresForLicenseConfigurationOperationsRequest
     * @return A Java Future containing the result of the ListFailuresForLicenseConfigurationOperations operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListFailuresForLicenseConfigurationOperations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListFailuresForLicenseConfigurationOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> listFailuresForLicenseConfigurationOperations(
            ListFailuresForLicenseConfigurationOperationsRequest listFailuresForLicenseConfigurationOperationsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFailuresForLicenseConfigurationOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFailuresForLicenseConfigurationOperationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListFailuresForLicenseConfigurationOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFailuresForLicenseConfigurationOperationsRequest, ListFailuresForLicenseConfigurationOperationsResponse>()
                            .withOperationName("ListFailuresForLicenseConfigurationOperations")
                            .withMarshaller(new ListFailuresForLicenseConfigurationOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listFailuresForLicenseConfigurationOperationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFailuresForLicenseConfigurationOperationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listFailuresForLicenseConfigurationOperationsRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license configurations for your account.
     * </p>
     *
     * @param listLicenseConfigurationsRequest
     * @return A Java Future containing the result of the ListLicenseConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseConfigurationsResponse> listLicenseConfigurations(
            ListLicenseConfigurationsRequest listLicenseConfigurationsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLicenseConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseConfigurationsRequest, ListLicenseConfigurationsResponse>()
                            .withOperationName("ListLicenseConfigurations")
                            .withMarshaller(new ListLicenseConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLicenseConfigurationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLicenseConfigurationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListLicenseConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listLicenseConfigurationsRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the license configurations for the specified resource.
     * </p>
     *
     * @param listLicenseSpecificationsForResourceRequest
     * @return A Java Future containing the result of the ListLicenseSpecificationsForResource operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseSpecificationsForResource
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseSpecificationsForResourceResponse> listLicenseSpecificationsForResource(
            ListLicenseSpecificationsForResourceRequest listLicenseSpecificationsForResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseSpecificationsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListLicenseSpecificationsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseSpecificationsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseSpecificationsForResourceRequest, ListLicenseSpecificationsForResourceResponse>()
                            .withOperationName("ListLicenseSpecificationsForResource")
                            .withMarshaller(new ListLicenseSpecificationsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLicenseSpecificationsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLicenseSpecificationsForResourceRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListLicenseSpecificationsForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                                requestOverrideConfig);
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listLicenseSpecificationsForResourceRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists resources managed using Systems Manager inventory.
     * </p>
     *
     * @param listResourceInventoryRequest
     * @return A Java Future containing the result of the ListResourceInventory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>FailedDependencyException A dependency required to run the API is missing.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListResourceInventory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListResourceInventory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceInventoryResponse> listResourceInventory(
            ListResourceInventoryRequest listResourceInventoryRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceInventory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceInventoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceInventoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceInventoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceInventoryRequest, ListResourceInventoryResponse>()
                            .withOperationName("ListResourceInventory")
                            .withMarshaller(new ListResourceInventoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourceInventoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResourceInventoryRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListResourceInventoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceInventoryRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified license configuration.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all license usage records for a license configuration, displaying license consumption details by resource
     * at a selected point in time. Use this action to audit the current license consumption for any license inventory
     * and configuration.
     * </p>
     *
     * @param listUsageForLicenseConfigurationRequest
     * @return A Java Future containing the result of the ListUsageForLicenseConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListUsageForLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListUsageForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsageForLicenseConfigurationResponse> listUsageForLicenseConfiguration(
            ListUsageForLicenseConfigurationRequest listUsageForLicenseConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsageForLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsageForLicenseConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListUsageForLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsageForLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsageForLicenseConfigurationRequest, ListUsageForLicenseConfigurationResponse>()
                            .withOperationName("ListUsageForLicenseConfiguration")
                            .withMarshaller(new ListUsageForLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUsageForLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listUsageForLicenseConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListUsageForLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listUsageForLicenseConfigurationRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified license configuration.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified license configuration.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the attributes of an existing license configuration.
     * </p>
     * <p>
     * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
     * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
     * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), host affinity (how
     * long a VM must be associated with a host), and the number of licenses purchased and used.
     * </p>
     *
     * @param updateLicenseConfigurationRequest
     * @return A Java Future containing the result of the UpdateLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseConfigurationResponse> updateLicenseConfiguration(
            UpdateLicenseConfigurationRequest updateLicenseConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseConfigurationRequest, UpdateLicenseConfigurationResponse>()
                            .withOperationName("UpdateLicenseConfiguration")
                            .withMarshaller(new UpdateLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    updateLicenseConfigurationRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or removes the specified license configurations for the specified AWS resource.
     * </p>
     * <p>
     * You can update the license specifications of AMIs, instances, and hosts. You cannot update the license
     * specifications for launch templates and AWS CloudFormation templates, as they send license configurations to the
     * operation that creates the resource.
     * </p>
     *
     * @param updateLicenseSpecificationsForResourceRequest
     * @return A Java Future containing the result of the UpdateLicenseSpecificationsForResource operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>InvalidResourceStateException License Manager cannot allocate a license to a resource because of its
     *         state. </p>
     *         <p>
     *         For example, you cannot allocate a license to an instance in the process of shutting down.</li>
     *         <li>LicenseUsageException You do not have enough licenses available to support a new resource launch.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseSpecificationsForResource
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> updateLicenseSpecificationsForResource(
            UpdateLicenseSpecificationsForResourceRequest updateLicenseSpecificationsForResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseSpecificationsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateLicenseSpecificationsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseSpecificationsForResourceRequest, UpdateLicenseSpecificationsForResourceResponse>()
                            .withOperationName("UpdateLicenseSpecificationsForResource")
                            .withMarshaller(new UpdateLicenseSpecificationsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLicenseSpecificationsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateLicenseSpecificationsForResourceRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                                requestOverrideConfig);
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    updateLicenseSpecificationsForResourceRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates License Manager settings for the current Region.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return A Java Future containing the result of the UpdateServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateServiceSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceSettingsResponse> updateServiceSettings(
            UpdateServiceSettingsRequest updateServiceSettingsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                            .withOperationName("UpdateServiceSettings")
                            .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateServiceSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateServiceSettingsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedDependencyException")
                                .exceptionBuilderSupplier(FailedDependencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationException")
                                .exceptionBuilderSupplier(AuthorizationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RateLimitExceededException")
                                .exceptionBuilderSupplier(RateLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterLimitExceededException")
                                .exceptionBuilderSupplier(FilterLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LicenseUsageException")
                                .exceptionBuilderSupplier(LicenseUsageException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
