/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanager;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanager.model.AcceptGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.AcceptGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.AccessDeniedException;
import software.amazon.awssdk.services.licensemanager.model.AuthorizationException;
import software.amazon.awssdk.services.licensemanager.model.CheckInLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CheckInLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.CheckoutBorrowLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CheckoutBorrowLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.CheckoutLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CheckoutLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.ConflictException;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantVersionRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantVersionResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseVersionRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseVersionResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateTokenRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateTokenResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteTokenRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteTokenResponse;
import software.amazon.awssdk.services.licensemanager.model.EntitlementNotAllowedException;
import software.amazon.awssdk.services.licensemanager.model.ExtendLicenseConsumptionRequest;
import software.amazon.awssdk.services.licensemanager.model.ExtendLicenseConsumptionResponse;
import software.amazon.awssdk.services.licensemanager.model.FailedDependencyException;
import software.amazon.awssdk.services.licensemanager.model.FilterLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.GetAccessTokenRequest;
import software.amazon.awssdk.services.licensemanager.model.GetAccessTokenResponse;
import software.amazon.awssdk.services.licensemanager.model.GetGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.GetGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseUsageRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseUsageResponse;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.model.InvalidParameterValueException;
import software.amazon.awssdk.services.licensemanager.model.InvalidResourceStateException;
import software.amazon.awssdk.services.licensemanager.model.LicenseManagerException;
import software.amazon.awssdk.services.licensemanager.model.LicenseUsageException;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.ListDistributedGrantsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListDistributedGrantsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseVersionsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseVersionsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicensesRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicensesResponse;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedGrantsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedGrantsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedLicensesRequest;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedLicensesResponse;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryRequest;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryResponse;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListTokensRequest;
import software.amazon.awssdk.services.licensemanager.model.ListTokensResponse;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.NoEntitlementsAllowedException;
import software.amazon.awssdk.services.licensemanager.model.RateLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.RedirectException;
import software.amazon.awssdk.services.licensemanager.model.RejectGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.RejectGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.ResourceNotFoundException;
import software.amazon.awssdk.services.licensemanager.model.ServerInternalException;
import software.amazon.awssdk.services.licensemanager.model.TagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.TagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UnsupportedDigitalSignatureMethodException;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.model.ValidationException;
import software.amazon.awssdk.services.licensemanager.transform.AcceptGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CheckInLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CheckoutBorrowLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CheckoutLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateGrantVersionRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseVersionRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateTokenRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteTokenRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ExtendLicenseConsumptionRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetAccessTokenRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseUsageRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListAssociationsForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListDistributedGrantsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListFailuresForLicenseConfigurationOperationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseVersionsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicensesRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListReceivedGrantsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListReceivedLicensesRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListResourceInventoryRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListTokensRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListUsageForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.RejectGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LicenseManagerAsyncClient}.
 *
 * @see LicenseManagerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerAsyncClient implements LicenseManagerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLicenseManagerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLicenseManagerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts the specified grant.
     * </p>
     *
     * @param acceptGrantRequest
     * @return A Java Future containing the result of the AcceptGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.AcceptGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/AcceptGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptGrantResponse> acceptGrant(AcceptGrantRequest acceptGrantRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AcceptGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptGrantRequest, AcceptGrantResponse>()
                            .withOperationName("AcceptGrant").withMarshaller(new AcceptGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptGrantRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = acceptGrantRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AcceptGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks in the specified license. Check in a license when it is no longer in use.
     * </p>
     *
     * @param checkInLicenseRequest
     * @return A Java Future containing the result of the CheckInLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CheckInLicense
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CheckInLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckInLicenseResponse> checkInLicense(CheckInLicenseRequest checkInLicenseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkInLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckInLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckInLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckInLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckInLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckInLicenseRequest, CheckInLicenseResponse>()
                            .withOperationName("CheckInLicense")
                            .withMarshaller(new CheckInLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(checkInLicenseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = checkInLicenseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CheckInLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks out the specified license for offline use.
     * </p>
     *
     * @param checkoutBorrowLicenseRequest
     * @return A Java Future containing the result of the CheckoutBorrowLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>NoEntitlementsAllowedException There are no entitlements found for this license, or the entitlement
     *         maximum count is reached.</li>
     *         <li>EntitlementNotAllowedException The entitlement is not allowed.</li>
     *         <li>UnsupportedDigitalSignatureMethodException The digital signature method is unsupported. Try your
     *         request again.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CheckoutBorrowLicense
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CheckoutBorrowLicense"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckoutBorrowLicenseResponse> checkoutBorrowLicense(
            CheckoutBorrowLicenseRequest checkoutBorrowLicenseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkoutBorrowLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckoutBorrowLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckoutBorrowLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckoutBorrowLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckoutBorrowLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckoutBorrowLicenseRequest, CheckoutBorrowLicenseResponse>()
                            .withOperationName("CheckoutBorrowLicense")
                            .withMarshaller(new CheckoutBorrowLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(checkoutBorrowLicenseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = checkoutBorrowLicenseRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CheckoutBorrowLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks out the specified license.
     * </p>
     *
     * @param checkoutLicenseRequest
     * @return A Java Future containing the result of the CheckoutLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>NoEntitlementsAllowedException There are no entitlements found for this license, or the entitlement
     *         maximum count is reached.</li>
     *         <li>UnsupportedDigitalSignatureMethodException The digital signature method is unsupported. Try your
     *         request again.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CheckoutLicense
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CheckoutLicense"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckoutLicenseResponse> checkoutLicense(CheckoutLicenseRequest checkoutLicenseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkoutLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckoutLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckoutLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckoutLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckoutLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckoutLicenseRequest, CheckoutLicenseResponse>()
                            .withOperationName("CheckoutLicense")
                            .withMarshaller(new CheckoutLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(checkoutLicenseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = checkoutLicenseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CheckoutLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a grant for the specified license. A grant shares the use of license entitlements with specific AWS
     * accounts.
     * </p>
     *
     * @param createGrantRequest
     * @return A Java Future containing the result of the CreateGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGrantResponse> createGrant(CreateGrantRequest createGrantRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGrantRequest, CreateGrantResponse>()
                            .withOperationName("CreateGrant").withMarshaller(new CreateGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGrantRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGrantRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of the specified grant.
     * </p>
     *
     * @param createGrantVersionRequest
     * @return A Java Future containing the result of the CreateGrantVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateGrantVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateGrantVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGrantVersionResponse> createGrantVersion(CreateGrantVersionRequest createGrantVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGrantVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGrantVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGrantVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGrantVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGrantVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGrantVersionRequest, CreateGrantVersionResponse>()
                            .withOperationName("CreateGrantVersion")
                            .withMarshaller(new CreateGrantVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGrantVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGrantVersionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateGrantVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a license.
     * </p>
     *
     * @param createLicenseRequest
     * @return A Java Future containing the result of the CreateLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicense
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseResponse> createLicense(CreateLicenseRequest createLicenseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseRequest, CreateLicenseResponse>()
                            .withOperationName("CreateLicense")
                            .withMarshaller(new CreateLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLicenseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLicenseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a license configuration.
     * </p>
     * <p>
     * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
     * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
     * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), license affinity to
     * host (how long a license must be associated with a host), and the number of licenses purchased and used.
     * </p>
     *
     * @param createLicenseConfigurationRequest
     * @return A Java Future containing the result of the CreateLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseConfigurationResponse> createLicenseConfiguration(
            CreateLicenseConfigurationRequest createLicenseConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseConfigurationRequest, CreateLicenseConfigurationResponse>()
                            .withOperationName("CreateLicenseConfiguration")
                            .withMarshaller(new CreateLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of the specified license.
     * </p>
     *
     * @param createLicenseVersionRequest
     * @return A Java Future containing the result of the CreateLicenseVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseVersionResponse> createLicenseVersion(
            CreateLicenseVersionRequest createLicenseVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLicenseVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLicenseVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseVersionRequest, CreateLicenseVersionResponse>()
                            .withOperationName("CreateLicenseVersion")
                            .withMarshaller(new CreateLicenseVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLicenseVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLicenseVersionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateLicenseVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a long-lived token.
     * </p>
     * <p>
     * A refresh token is a JWT token used to get an access token. With an access token, you can call
     * AssumeRoleWithWebIdentity to get role credentials that you can use to call License Manager to manage the
     * specified license.
     * </p>
     *
     * @param createTokenRequest
     * @return A Java Future containing the result of the CreateToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTokenResponse> createToken(CreateTokenRequest createTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTokenRequest, CreateTokenResponse>()
                            .withOperationName("CreateToken").withMarshaller(new CreateTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTokenRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createTokenRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified grant.
     * </p>
     *
     * @param deleteGrantRequest
     * @return A Java Future containing the result of the DeleteGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGrantResponse> deleteGrant(DeleteGrantRequest deleteGrantRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGrantRequest, DeleteGrantResponse>()
                            .withOperationName("DeleteGrant").withMarshaller(new DeleteGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGrantRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteGrantRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified license.
     * </p>
     *
     * @param deleteLicenseRequest
     * @return A Java Future containing the result of the DeleteLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteLicense
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLicenseResponse> deleteLicense(DeleteLicenseRequest deleteLicenseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLicenseRequest, DeleteLicenseResponse>()
                            .withOperationName("DeleteLicense")
                            .withMarshaller(new DeleteLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLicenseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLicenseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified license configuration.
     * </p>
     * <p>
     * You cannot delete a license configuration that is in use.
     * </p>
     *
     * @param deleteLicenseConfigurationRequest
     * @return A Java Future containing the result of the DeleteLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLicenseConfigurationResponse> deleteLicenseConfiguration(
            DeleteLicenseConfigurationRequest deleteLicenseConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLicenseConfigurationRequest, DeleteLicenseConfigurationResponse>()
                            .withOperationName("DeleteLicenseConfiguration")
                            .withMarshaller(new DeleteLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified token. Must be called in the license home Region.
     * </p>
     *
     * @param deleteTokenRequest
     * @return A Java Future containing the result of the DeleteToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTokenResponse> deleteToken(DeleteTokenRequest deleteTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTokenRequest, DeleteTokenResponse>()
                            .withOperationName("DeleteToken").withMarshaller(new DeleteTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTokenRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTokenRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Extends the expiration date for license consumption.
     * </p>
     *
     * @param extendLicenseConsumptionRequest
     * @return A Java Future containing the result of the ExtendLicenseConsumption operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ExtendLicenseConsumption
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ExtendLicenseConsumption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExtendLicenseConsumptionResponse> extendLicenseConsumption(
            ExtendLicenseConsumptionRequest extendLicenseConsumptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, extendLicenseConsumptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExtendLicenseConsumption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExtendLicenseConsumptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExtendLicenseConsumptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExtendLicenseConsumptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExtendLicenseConsumptionRequest, ExtendLicenseConsumptionResponse>()
                            .withOperationName("ExtendLicenseConsumption")
                            .withMarshaller(new ExtendLicenseConsumptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(extendLicenseConsumptionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = extendLicenseConsumptionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ExtendLicenseConsumptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a temporary access token to use with AssumeRoleWithWebIdentity. Access tokens are valid for one hour.
     * </p>
     *
     * @param getAccessTokenRequest
     * @return A Java Future containing the result of the GetAccessToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetAccessToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetAccessToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessTokenResponse> getAccessToken(GetAccessTokenRequest getAccessTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccessTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccessTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccessTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessTokenRequest, GetAccessTokenResponse>()
                            .withOperationName("GetAccessToken")
                            .withMarshaller(new GetAccessTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessTokenRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessTokenRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAccessTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified grant.
     * </p>
     *
     * @param getGrantRequest
     * @return A Java Future containing the result of the GetGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetGrant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGrantResponse> getGrant(GetGrantRequest getGrantRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGrantRequest, GetGrantResponse>().withOperationName("GetGrant")
                            .withMarshaller(new GetGrantRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getGrantRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getGrantRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified license.
     * </p>
     *
     * @param getLicenseRequest
     * @return A Java Future containing the result of the GetLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicense
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicense" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseResponse> getLicense(GetLicenseRequest getLicenseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseRequest, GetLicenseResponse>().withOperationName("GetLicense")
                            .withMarshaller(new GetLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLicenseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLicenseRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified license configuration.
     * </p>
     *
     * @param getLicenseConfigurationRequest
     * @return A Java Future containing the result of the GetLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseConfigurationResponse> getLicenseConfiguration(
            GetLicenseConfigurationRequest getLicenseConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseConfigurationRequest, GetLicenseConfigurationResponse>()
                            .withOperationName("GetLicenseConfiguration")
                            .withMarshaller(new GetLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the usage of the specified license.
     * </p>
     *
     * @param getLicenseUsageRequest
     * @return A Java Future containing the result of the GetLicenseUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseUsageResponse> getLicenseUsage(GetLicenseUsageRequest getLicenseUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseUsageRequest, GetLicenseUsageResponse>()
                            .withOperationName("GetLicenseUsage")
                            .withMarshaller(new GetLicenseUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLicenseUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLicenseUsageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetLicenseUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the License Manager settings for the current Region.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return A Java Future containing the result of the GetServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetServiceSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceSettingsResponse> getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                            .withOperationName("GetServiceSettings")
                            .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getServiceSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getServiceSettingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource associations for the specified license configuration.
     * </p>
     * <p>
     * Resource associations need not consume licenses from a license configuration. For example, an AMI or a stopped
     * instance might not consume a license (depending on the license rules).
     * </p>
     *
     * @param listAssociationsForLicenseConfigurationRequest
     * @return A Java Future containing the result of the ListAssociationsForLicenseConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListAssociationsForLicenseConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListAssociationsForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociationsForLicenseConfigurationResponse> listAssociationsForLicenseConfiguration(
            ListAssociationsForLicenseConfigurationRequest listAssociationsForLicenseConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAssociationsForLicenseConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociationsForLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociationsForLicenseConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListAssociationsForLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssociationsForLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociationsForLicenseConfigurationRequest, ListAssociationsForLicenseConfigurationResponse>()
                            .withOperationName("ListAssociationsForLicenseConfiguration")
                            .withMarshaller(new ListAssociationsForLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociationsForLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAssociationsForLicenseConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListAssociationsForLicenseConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the grants distributed for the specified license.
     * </p>
     *
     * @param listDistributedGrantsRequest
     * @return A Java Future containing the result of the ListDistributedGrants operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListDistributedGrants
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListDistributedGrants"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDistributedGrantsResponse> listDistributedGrants(
            ListDistributedGrantsRequest listDistributedGrantsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDistributedGrantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDistributedGrants");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDistributedGrantsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDistributedGrantsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDistributedGrantsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDistributedGrantsRequest, ListDistributedGrantsResponse>()
                            .withOperationName("ListDistributedGrants")
                            .withMarshaller(new ListDistributedGrantsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDistributedGrantsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDistributedGrantsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListDistributedGrantsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license configuration operations that failed.
     * </p>
     *
     * @param listFailuresForLicenseConfigurationOperationsRequest
     * @return A Java Future containing the result of the ListFailuresForLicenseConfigurationOperations operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListFailuresForLicenseConfigurationOperations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListFailuresForLicenseConfigurationOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> listFailuresForLicenseConfigurationOperations(
            ListFailuresForLicenseConfigurationOperationsRequest listFailuresForLicenseConfigurationOperationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listFailuresForLicenseConfigurationOperationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFailuresForLicenseConfigurationOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFailuresForLicenseConfigurationOperationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListFailuresForLicenseConfigurationOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFailuresForLicenseConfigurationOperationsRequest, ListFailuresForLicenseConfigurationOperationsResponse>()
                            .withOperationName("ListFailuresForLicenseConfigurationOperations")
                            .withMarshaller(new ListFailuresForLicenseConfigurationOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listFailuresForLicenseConfigurationOperationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFailuresForLicenseConfigurationOperationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license configurations for your account.
     * </p>
     *
     * @param listLicenseConfigurationsRequest
     * @return A Java Future containing the result of the ListLicenseConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseConfigurationsResponse> listLicenseConfigurations(
            ListLicenseConfigurationsRequest listLicenseConfigurationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicenseConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLicenseConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseConfigurationsRequest, ListLicenseConfigurationsResponse>()
                            .withOperationName("ListLicenseConfigurations")
                            .withMarshaller(new ListLicenseConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLicenseConfigurationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLicenseConfigurationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListLicenseConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the license configurations for the specified resource.
     * </p>
     *
     * @param listLicenseSpecificationsForResourceRequest
     * @return A Java Future containing the result of the ListLicenseSpecificationsForResource operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseSpecificationsForResource
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseSpecificationsForResourceResponse> listLicenseSpecificationsForResource(
            ListLicenseSpecificationsForResourceRequest listLicenseSpecificationsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listLicenseSpecificationsForResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseSpecificationsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListLicenseSpecificationsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseSpecificationsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseSpecificationsForResourceRequest, ListLicenseSpecificationsForResourceResponse>()
                            .withOperationName("ListLicenseSpecificationsForResource")
                            .withMarshaller(new ListLicenseSpecificationsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLicenseSpecificationsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLicenseSpecificationsForResourceRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListLicenseSpecificationsForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all versions of the specified license.
     * </p>
     *
     * @param listLicenseVersionsRequest
     * @return A Java Future containing the result of the ListLicenseVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseVersionsResponse> listLicenseVersions(
            ListLicenseVersionsRequest listLicenseVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicenseVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLicenseVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseVersionsRequest, ListLicenseVersionsResponse>()
                            .withOperationName("ListLicenseVersions")
                            .withMarshaller(new ListLicenseVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLicenseVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLicenseVersionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListLicenseVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the licenses for your account.
     * </p>
     *
     * @param listLicensesRequest
     * @return A Java Future containing the result of the ListLicenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicensesResponse> listLicenses(ListLicensesRequest listLicensesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicensesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLicensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicensesRequest, ListLicensesResponse>()
                            .withOperationName("ListLicenses").withMarshaller(new ListLicensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLicensesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLicensesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListLicensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists grants that are received but not accepted.
     * </p>
     *
     * @param listReceivedGrantsRequest
     * @return A Java Future containing the result of the ListReceivedGrants operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListReceivedGrants
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListReceivedGrants"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReceivedGrantsResponse> listReceivedGrants(ListReceivedGrantsRequest listReceivedGrantsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReceivedGrantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReceivedGrants");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReceivedGrantsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReceivedGrantsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReceivedGrantsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReceivedGrantsRequest, ListReceivedGrantsResponse>()
                            .withOperationName("ListReceivedGrants")
                            .withMarshaller(new ListReceivedGrantsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listReceivedGrantsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listReceivedGrantsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListReceivedGrantsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists received licenses.
     * </p>
     *
     * @param listReceivedLicensesRequest
     * @return A Java Future containing the result of the ListReceivedLicenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListReceivedLicenses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListReceivedLicenses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReceivedLicensesResponse> listReceivedLicenses(
            ListReceivedLicensesRequest listReceivedLicensesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReceivedLicensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReceivedLicenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReceivedLicensesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReceivedLicensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReceivedLicensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReceivedLicensesRequest, ListReceivedLicensesResponse>()
                            .withOperationName("ListReceivedLicenses")
                            .withMarshaller(new ListReceivedLicensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listReceivedLicensesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listReceivedLicensesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListReceivedLicensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists resources managed using Systems Manager inventory.
     * </p>
     *
     * @param listResourceInventoryRequest
     * @return A Java Future containing the result of the ListResourceInventory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>FailedDependencyException A dependency required to run the API is missing.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListResourceInventory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListResourceInventory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceInventoryResponse> listResourceInventory(
            ListResourceInventoryRequest listResourceInventoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceInventoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceInventory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceInventoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceInventoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceInventoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceInventoryRequest, ListResourceInventoryResponse>()
                            .withOperationName("ListResourceInventory")
                            .withMarshaller(new ListResourceInventoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourceInventoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResourceInventoryRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListResourceInventoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified license configuration.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your tokens.
     * </p>
     *
     * @param listTokensRequest
     * @return A Java Future containing the result of the ListTokens operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListTokens
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListTokens" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTokensResponse> listTokens(ListTokensRequest listTokensRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTokensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTokens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTokensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTokensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTokensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTokensRequest, ListTokensResponse>().withOperationName("ListTokens")
                            .withMarshaller(new ListTokensRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTokensRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTokensRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListTokensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all license usage records for a license configuration, displaying license consumption details by resource
     * at a selected point in time. Use this action to audit the current license consumption for any license inventory
     * and configuration.
     * </p>
     *
     * @param listUsageForLicenseConfigurationRequest
     * @return A Java Future containing the result of the ListUsageForLicenseConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListUsageForLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListUsageForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsageForLicenseConfigurationResponse> listUsageForLicenseConfiguration(
            ListUsageForLicenseConfigurationRequest listUsageForLicenseConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listUsageForLicenseConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsageForLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsageForLicenseConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListUsageForLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsageForLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsageForLicenseConfigurationRequest, ListUsageForLicenseConfigurationResponse>()
                            .withOperationName("ListUsageForLicenseConfiguration")
                            .withMarshaller(new ListUsageForLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUsageForLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listUsageForLicenseConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListUsageForLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects the specified grant.
     * </p>
     *
     * @param rejectGrantRequest
     * @return A Java Future containing the result of the RejectGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.RejectGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/RejectGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectGrantResponse> rejectGrant(RejectGrantRequest rejectGrantRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RejectGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectGrantRequest, RejectGrantResponse>()
                            .withOperationName("RejectGrant").withMarshaller(new RejectGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectGrantRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = rejectGrantRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RejectGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified license configuration.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified license configuration.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the attributes of an existing license configuration.
     * </p>
     *
     * @param updateLicenseConfigurationRequest
     * @return A Java Future containing the result of the UpdateLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseConfigurationResponse> updateLicenseConfiguration(
            UpdateLicenseConfigurationRequest updateLicenseConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseConfigurationRequest, UpdateLicenseConfigurationResponse>()
                            .withOperationName("UpdateLicenseConfiguration")
                            .withMarshaller(new UpdateLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLicenseConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateLicenseConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or removes the specified license configurations for the specified AWS resource.
     * </p>
     * <p>
     * You can update the license specifications of AMIs, instances, and hosts. You cannot update the license
     * specifications for launch templates and AWS CloudFormation templates, as they send license configurations to the
     * operation that creates the resource.
     * </p>
     *
     * @param updateLicenseSpecificationsForResourceRequest
     * @return A Java Future containing the result of the UpdateLicenseSpecificationsForResource operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>InvalidResourceStateException License Manager cannot allocate a license to a resource because of its
     *         state. </p>
     *         <p>
     *         For example, you cannot allocate a license to an instance in the process of shutting down.</li>
     *         <li>LicenseUsageException You do not have enough licenses available to support a new resource launch.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseSpecificationsForResource
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> updateLicenseSpecificationsForResource(
            UpdateLicenseSpecificationsForResourceRequest updateLicenseSpecificationsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateLicenseSpecificationsForResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseSpecificationsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateLicenseSpecificationsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseSpecificationsForResourceRequest, UpdateLicenseSpecificationsForResourceResponse>()
                            .withOperationName("UpdateLicenseSpecificationsForResource")
                            .withMarshaller(new UpdateLicenseSpecificationsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLicenseSpecificationsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateLicenseSpecificationsForResourceRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates License Manager settings for the current Region.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return A Java Future containing the result of the UpdateServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The AWS user account does not have permission to perform the action. Check the
     *         IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateServiceSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceSettingsResponse> updateServiceSettings(
            UpdateServiceSettingsRequest updateServiceSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                            .withOperationName("UpdateServiceSettings")
                            .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateServiceSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateServiceSettingsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedDependencyException")
                                .exceptionBuilderSupplier(FailedDependencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RateLimitExceededException")
                                .exceptionBuilderSupplier(RateLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterLimitExceededException")
                                .exceptionBuilderSupplier(FilterLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RedirectException")
                                .exceptionBuilderSupplier(RedirectException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDigitalSignatureMethodException")
                                .exceptionBuilderSupplier(UnsupportedDigitalSignatureMethodException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationException")
                                .exceptionBuilderSupplier(AuthorizationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntitlementNotAllowedException")
                                .exceptionBuilderSupplier(EntitlementNotAllowedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoEntitlementsAllowedException")
                                .exceptionBuilderSupplier(NoEntitlementsAllowedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LicenseUsageException")
                                .exceptionBuilderSupplier(LicenseUsageException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
