/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
 * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
 * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), host affinity (how long
 * a VM must be associated with a host), and the number of licenses purchased and used.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LicenseConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LicenseConfiguration.Builder, LicenseConfiguration> {
    private static final SdkField<String> LICENSE_CONFIGURATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LicenseConfigurationId").getter(getter(LicenseConfiguration::licenseConfigurationId))
            .setter(setter(Builder::licenseConfigurationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseConfigurationId").build())
            .build();

    private static final SdkField<String> LICENSE_CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LicenseConfigurationArn").getter(getter(LicenseConfiguration::licenseConfigurationArn))
            .setter(setter(Builder::licenseConfigurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseConfigurationArn").build())
            .build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(LicenseConfiguration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(LicenseConfiguration::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> LICENSE_COUNTING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LicenseCountingType").getter(getter(LicenseConfiguration::licenseCountingTypeAsString))
            .setter(setter(Builder::licenseCountingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseCountingType").build())
            .build();

    private static final SdkField<List<String>> LICENSE_RULES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LicenseRules")
            .getter(getter(LicenseConfiguration::licenseRules))
            .setter(setter(Builder::licenseRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> LICENSE_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("LicenseCount").getter(getter(LicenseConfiguration::licenseCount)).setter(setter(Builder::licenseCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseCount").build()).build();

    private static final SdkField<Boolean> LICENSE_COUNT_HARD_LIMIT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("LicenseCountHardLimit").getter(getter(LicenseConfiguration::licenseCountHardLimit))
            .setter(setter(Builder::licenseCountHardLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseCountHardLimit").build())
            .build();

    private static final SdkField<Boolean> DISASSOCIATE_WHEN_NOT_FOUND_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("DisassociateWhenNotFound")
            .getter(getter(LicenseConfiguration::disassociateWhenNotFound)).setter(setter(Builder::disassociateWhenNotFound))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisassociateWhenNotFound").build())
            .build();

    private static final SdkField<Long> CONSUMED_LICENSES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ConsumedLicenses").getter(getter(LicenseConfiguration::consumedLicenses))
            .setter(setter(Builder::consumedLicenses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsumedLicenses").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(LicenseConfiguration::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerAccountId").getter(getter(LicenseConfiguration::ownerAccountId))
            .setter(setter(Builder::ownerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerAccountId").build()).build();

    private static final SdkField<List<ConsumedLicenseSummary>> CONSUMED_LICENSE_SUMMARY_LIST_FIELD = SdkField
            .<List<ConsumedLicenseSummary>> builder(MarshallingType.LIST)
            .memberName("ConsumedLicenseSummaryList")
            .getter(getter(LicenseConfiguration::consumedLicenseSummaryList))
            .setter(setter(Builder::consumedLicenseSummaryList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsumedLicenseSummaryList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConsumedLicenseSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConsumedLicenseSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ManagedResourceSummary>> MANAGED_RESOURCE_SUMMARY_LIST_FIELD = SdkField
            .<List<ManagedResourceSummary>> builder(MarshallingType.LIST)
            .memberName("ManagedResourceSummaryList")
            .getter(getter(LicenseConfiguration::managedResourceSummaryList))
            .setter(setter(Builder::managedResourceSummaryList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedResourceSummaryList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ManagedResourceSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(ManagedResourceSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ProductInformation>> PRODUCT_INFORMATION_LIST_FIELD = SdkField
            .<List<ProductInformation>> builder(MarshallingType.LIST)
            .memberName("ProductInformationList")
            .getter(getter(LicenseConfiguration::productInformationList))
            .setter(setter(Builder::productInformationList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductInformationList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProductInformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProductInformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AutomatedDiscoveryInformation> AUTOMATED_DISCOVERY_INFORMATION_FIELD = SdkField
            .<AutomatedDiscoveryInformation> builder(MarshallingType.SDK_POJO)
            .memberName("AutomatedDiscoveryInformation")
            .getter(getter(LicenseConfiguration::automatedDiscoveryInformation))
            .setter(setter(Builder::automatedDiscoveryInformation))
            .constructor(AutomatedDiscoveryInformation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomatedDiscoveryInformation")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            LICENSE_CONFIGURATION_ID_FIELD, LICENSE_CONFIGURATION_ARN_FIELD, NAME_FIELD, DESCRIPTION_FIELD,
            LICENSE_COUNTING_TYPE_FIELD, LICENSE_RULES_FIELD, LICENSE_COUNT_FIELD, LICENSE_COUNT_HARD_LIMIT_FIELD,
            DISASSOCIATE_WHEN_NOT_FOUND_FIELD, CONSUMED_LICENSES_FIELD, STATUS_FIELD, OWNER_ACCOUNT_ID_FIELD,
            CONSUMED_LICENSE_SUMMARY_LIST_FIELD, MANAGED_RESOURCE_SUMMARY_LIST_FIELD, PRODUCT_INFORMATION_LIST_FIELD,
            AUTOMATED_DISCOVERY_INFORMATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String licenseConfigurationId;

    private final String licenseConfigurationArn;

    private final String name;

    private final String description;

    private final String licenseCountingType;

    private final List<String> licenseRules;

    private final Long licenseCount;

    private final Boolean licenseCountHardLimit;

    private final Boolean disassociateWhenNotFound;

    private final Long consumedLicenses;

    private final String status;

    private final String ownerAccountId;

    private final List<ConsumedLicenseSummary> consumedLicenseSummaryList;

    private final List<ManagedResourceSummary> managedResourceSummaryList;

    private final List<ProductInformation> productInformationList;

    private final AutomatedDiscoveryInformation automatedDiscoveryInformation;

    private LicenseConfiguration(BuilderImpl builder) {
        this.licenseConfigurationId = builder.licenseConfigurationId;
        this.licenseConfigurationArn = builder.licenseConfigurationArn;
        this.name = builder.name;
        this.description = builder.description;
        this.licenseCountingType = builder.licenseCountingType;
        this.licenseRules = builder.licenseRules;
        this.licenseCount = builder.licenseCount;
        this.licenseCountHardLimit = builder.licenseCountHardLimit;
        this.disassociateWhenNotFound = builder.disassociateWhenNotFound;
        this.consumedLicenses = builder.consumedLicenses;
        this.status = builder.status;
        this.ownerAccountId = builder.ownerAccountId;
        this.consumedLicenseSummaryList = builder.consumedLicenseSummaryList;
        this.managedResourceSummaryList = builder.managedResourceSummaryList;
        this.productInformationList = builder.productInformationList;
        this.automatedDiscoveryInformation = builder.automatedDiscoveryInformation;
    }

    /**
     * <p>
     * Unique ID of the license configuration.
     * </p>
     * 
     * @return Unique ID of the license configuration.
     */
    public final String licenseConfigurationId() {
        return licenseConfigurationId;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) of the license configuration.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) of the license configuration.
     */
    public final String licenseConfigurationArn() {
        return licenseConfigurationArn;
    }

    /**
     * <p>
     * Name of the license configuration.
     * </p>
     * 
     * @return Name of the license configuration.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Description of the license configuration.
     * </p>
     * 
     * @return Description of the license configuration.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Dimension to use to track the license inventory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #licenseCountingType} will return {@link LicenseCountingType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #licenseCountingTypeAsString}.
     * </p>
     * 
     * @return Dimension to use to track the license inventory.
     * @see LicenseCountingType
     */
    public final LicenseCountingType licenseCountingType() {
        return LicenseCountingType.fromValue(licenseCountingType);
    }

    /**
     * <p>
     * Dimension to use to track the license inventory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #licenseCountingType} will return {@link LicenseCountingType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #licenseCountingTypeAsString}.
     * </p>
     * 
     * @return Dimension to use to track the license inventory.
     * @see LicenseCountingType
     */
    public final String licenseCountingTypeAsString() {
        return licenseCountingType;
    }

    /**
     * Returns true if the LicenseRules property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLicenseRules() {
        return licenseRules != null && !(licenseRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * License rules.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLicenseRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return License rules.
     */
    public final List<String> licenseRules() {
        return licenseRules;
    }

    /**
     * <p>
     * Number of licenses managed by the license configuration.
     * </p>
     * 
     * @return Number of licenses managed by the license configuration.
     */
    public final Long licenseCount() {
        return licenseCount;
    }

    /**
     * <p>
     * Number of available licenses as a hard limit.
     * </p>
     * 
     * @return Number of available licenses as a hard limit.
     */
    public final Boolean licenseCountHardLimit() {
        return licenseCountHardLimit;
    }

    /**
     * <p>
     * When true, disassociates a resource when software is uninstalled.
     * </p>
     * 
     * @return When true, disassociates a resource when software is uninstalled.
     */
    public final Boolean disassociateWhenNotFound() {
        return disassociateWhenNotFound;
    }

    /**
     * <p>
     * Number of licenses consumed.
     * </p>
     * 
     * @return Number of licenses consumed.
     */
    public final Long consumedLicenses() {
        return consumedLicenses;
    }

    /**
     * <p>
     * Status of the license configuration.
     * </p>
     * 
     * @return Status of the license configuration.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * Account ID of the license configuration's owner.
     * </p>
     * 
     * @return Account ID of the license configuration's owner.
     */
    public final String ownerAccountId() {
        return ownerAccountId;
    }

    /**
     * Returns true if the ConsumedLicenseSummaryList property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public final boolean hasConsumedLicenseSummaryList() {
        return consumedLicenseSummaryList != null && !(consumedLicenseSummaryList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Summaries for licenses consumed by various resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConsumedLicenseSummaryList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Summaries for licenses consumed by various resources.
     */
    public final List<ConsumedLicenseSummary> consumedLicenseSummaryList() {
        return consumedLicenseSummaryList;
    }

    /**
     * Returns true if the ManagedResourceSummaryList property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public final boolean hasManagedResourceSummaryList() {
        return managedResourceSummaryList != null && !(managedResourceSummaryList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Summaries for managed resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasManagedResourceSummaryList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Summaries for managed resources.
     */
    public final List<ManagedResourceSummary> managedResourceSummaryList() {
        return managedResourceSummaryList;
    }

    /**
     * Returns true if the ProductInformationList property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasProductInformationList() {
        return productInformationList != null && !(productInformationList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Product information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProductInformationList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Product information.
     */
    public final List<ProductInformation> productInformationList() {
        return productInformationList;
    }

    /**
     * <p>
     * Automated discovery information.
     * </p>
     * 
     * @return Automated discovery information.
     */
    public final AutomatedDiscoveryInformation automatedDiscoveryInformation() {
        return automatedDiscoveryInformation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(licenseConfigurationId());
        hashCode = 31 * hashCode + Objects.hashCode(licenseConfigurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(licenseCountingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLicenseRules() ? licenseRules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(licenseCount());
        hashCode = 31 * hashCode + Objects.hashCode(licenseCountHardLimit());
        hashCode = 31 * hashCode + Objects.hashCode(disassociateWhenNotFound());
        hashCode = 31 * hashCode + Objects.hashCode(consumedLicenses());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasConsumedLicenseSummaryList() ? consumedLicenseSummaryList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasManagedResourceSummaryList() ? managedResourceSummaryList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasProductInformationList() ? productInformationList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(automatedDiscoveryInformation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LicenseConfiguration)) {
            return false;
        }
        LicenseConfiguration other = (LicenseConfiguration) obj;
        return Objects.equals(licenseConfigurationId(), other.licenseConfigurationId())
                && Objects.equals(licenseConfigurationArn(), other.licenseConfigurationArn())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(licenseCountingTypeAsString(), other.licenseCountingTypeAsString())
                && hasLicenseRules() == other.hasLicenseRules() && Objects.equals(licenseRules(), other.licenseRules())
                && Objects.equals(licenseCount(), other.licenseCount())
                && Objects.equals(licenseCountHardLimit(), other.licenseCountHardLimit())
                && Objects.equals(disassociateWhenNotFound(), other.disassociateWhenNotFound())
                && Objects.equals(consumedLicenses(), other.consumedLicenses()) && Objects.equals(status(), other.status())
                && Objects.equals(ownerAccountId(), other.ownerAccountId())
                && hasConsumedLicenseSummaryList() == other.hasConsumedLicenseSummaryList()
                && Objects.equals(consumedLicenseSummaryList(), other.consumedLicenseSummaryList())
                && hasManagedResourceSummaryList() == other.hasManagedResourceSummaryList()
                && Objects.equals(managedResourceSummaryList(), other.managedResourceSummaryList())
                && hasProductInformationList() == other.hasProductInformationList()
                && Objects.equals(productInformationList(), other.productInformationList())
                && Objects.equals(automatedDiscoveryInformation(), other.automatedDiscoveryInformation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LicenseConfiguration").add("LicenseConfigurationId", licenseConfigurationId())
                .add("LicenseConfigurationArn", licenseConfigurationArn()).add("Name", name()).add("Description", description())
                .add("LicenseCountingType", licenseCountingTypeAsString())
                .add("LicenseRules", hasLicenseRules() ? licenseRules() : null).add("LicenseCount", licenseCount())
                .add("LicenseCountHardLimit", licenseCountHardLimit())
                .add("DisassociateWhenNotFound", disassociateWhenNotFound()).add("ConsumedLicenses", consumedLicenses())
                .add("Status", status()).add("OwnerAccountId", ownerAccountId())
                .add("ConsumedLicenseSummaryList", hasConsumedLicenseSummaryList() ? consumedLicenseSummaryList() : null)
                .add("ManagedResourceSummaryList", hasManagedResourceSummaryList() ? managedResourceSummaryList() : null)
                .add("ProductInformationList", hasProductInformationList() ? productInformationList() : null)
                .add("AutomatedDiscoveryInformation", automatedDiscoveryInformation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LicenseConfigurationId":
            return Optional.ofNullable(clazz.cast(licenseConfigurationId()));
        case "LicenseConfigurationArn":
            return Optional.ofNullable(clazz.cast(licenseConfigurationArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "LicenseCountingType":
            return Optional.ofNullable(clazz.cast(licenseCountingTypeAsString()));
        case "LicenseRules":
            return Optional.ofNullable(clazz.cast(licenseRules()));
        case "LicenseCount":
            return Optional.ofNullable(clazz.cast(licenseCount()));
        case "LicenseCountHardLimit":
            return Optional.ofNullable(clazz.cast(licenseCountHardLimit()));
        case "DisassociateWhenNotFound":
            return Optional.ofNullable(clazz.cast(disassociateWhenNotFound()));
        case "ConsumedLicenses":
            return Optional.ofNullable(clazz.cast(consumedLicenses()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "OwnerAccountId":
            return Optional.ofNullable(clazz.cast(ownerAccountId()));
        case "ConsumedLicenseSummaryList":
            return Optional.ofNullable(clazz.cast(consumedLicenseSummaryList()));
        case "ManagedResourceSummaryList":
            return Optional.ofNullable(clazz.cast(managedResourceSummaryList()));
        case "ProductInformationList":
            return Optional.ofNullable(clazz.cast(productInformationList()));
        case "AutomatedDiscoveryInformation":
            return Optional.ofNullable(clazz.cast(automatedDiscoveryInformation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LicenseConfiguration, T> g) {
        return obj -> g.apply((LicenseConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LicenseConfiguration> {
        /**
         * <p>
         * Unique ID of the license configuration.
         * </p>
         * 
         * @param licenseConfigurationId
         *        Unique ID of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseConfigurationId(String licenseConfigurationId);

        /**
         * <p>
         * Amazon Resource Name (ARN) of the license configuration.
         * </p>
         * 
         * @param licenseConfigurationArn
         *        Amazon Resource Name (ARN) of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseConfigurationArn(String licenseConfigurationArn);

        /**
         * <p>
         * Name of the license configuration.
         * </p>
         * 
         * @param name
         *        Name of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Description of the license configuration.
         * </p>
         * 
         * @param description
         *        Description of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Dimension to use to track the license inventory.
         * </p>
         * 
         * @param licenseCountingType
         *        Dimension to use to track the license inventory.
         * @see LicenseCountingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseCountingType
         */
        Builder licenseCountingType(String licenseCountingType);

        /**
         * <p>
         * Dimension to use to track the license inventory.
         * </p>
         * 
         * @param licenseCountingType
         *        Dimension to use to track the license inventory.
         * @see LicenseCountingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseCountingType
         */
        Builder licenseCountingType(LicenseCountingType licenseCountingType);

        /**
         * <p>
         * License rules.
         * </p>
         * 
         * @param licenseRules
         *        License rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseRules(Collection<String> licenseRules);

        /**
         * <p>
         * License rules.
         * </p>
         * 
         * @param licenseRules
         *        License rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseRules(String... licenseRules);

        /**
         * <p>
         * Number of licenses managed by the license configuration.
         * </p>
         * 
         * @param licenseCount
         *        Number of licenses managed by the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseCount(Long licenseCount);

        /**
         * <p>
         * Number of available licenses as a hard limit.
         * </p>
         * 
         * @param licenseCountHardLimit
         *        Number of available licenses as a hard limit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseCountHardLimit(Boolean licenseCountHardLimit);

        /**
         * <p>
         * When true, disassociates a resource when software is uninstalled.
         * </p>
         * 
         * @param disassociateWhenNotFound
         *        When true, disassociates a resource when software is uninstalled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disassociateWhenNotFound(Boolean disassociateWhenNotFound);

        /**
         * <p>
         * Number of licenses consumed.
         * </p>
         * 
         * @param consumedLicenses
         *        Number of licenses consumed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLicenses(Long consumedLicenses);

        /**
         * <p>
         * Status of the license configuration.
         * </p>
         * 
         * @param status
         *        Status of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Account ID of the license configuration's owner.
         * </p>
         * 
         * @param ownerAccountId
         *        Account ID of the license configuration's owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccountId(String ownerAccountId);

        /**
         * <p>
         * Summaries for licenses consumed by various resources.
         * </p>
         * 
         * @param consumedLicenseSummaryList
         *        Summaries for licenses consumed by various resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLicenseSummaryList(Collection<ConsumedLicenseSummary> consumedLicenseSummaryList);

        /**
         * <p>
         * Summaries for licenses consumed by various resources.
         * </p>
         * 
         * @param consumedLicenseSummaryList
         *        Summaries for licenses consumed by various resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLicenseSummaryList(ConsumedLicenseSummary... consumedLicenseSummaryList);

        /**
         * <p>
         * Summaries for licenses consumed by various resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ConsumedLicenseSummary>.Builder} avoiding
         * the need to create one manually via {@link List<ConsumedLicenseSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ConsumedLicenseSummary>.Builder#build()} is called
         * immediately and its result is passed to {@link #consumedLicenseSummaryList(List<ConsumedLicenseSummary>)}.
         * 
         * @param consumedLicenseSummaryList
         *        a consumer that will call methods on {@link List<ConsumedLicenseSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #consumedLicenseSummaryList(List<ConsumedLicenseSummary>)
         */
        Builder consumedLicenseSummaryList(Consumer<ConsumedLicenseSummary.Builder>... consumedLicenseSummaryList);

        /**
         * <p>
         * Summaries for managed resources.
         * </p>
         * 
         * @param managedResourceSummaryList
         *        Summaries for managed resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedResourceSummaryList(Collection<ManagedResourceSummary> managedResourceSummaryList);

        /**
         * <p>
         * Summaries for managed resources.
         * </p>
         * 
         * @param managedResourceSummaryList
         *        Summaries for managed resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedResourceSummaryList(ManagedResourceSummary... managedResourceSummaryList);

        /**
         * <p>
         * Summaries for managed resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ManagedResourceSummary>.Builder} avoiding
         * the need to create one manually via {@link List<ManagedResourceSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ManagedResourceSummary>.Builder#build()} is called
         * immediately and its result is passed to {@link #managedResourceSummaryList(List<ManagedResourceSummary>)}.
         * 
         * @param managedResourceSummaryList
         *        a consumer that will call methods on {@link List<ManagedResourceSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #managedResourceSummaryList(List<ManagedResourceSummary>)
         */
        Builder managedResourceSummaryList(Consumer<ManagedResourceSummary.Builder>... managedResourceSummaryList);

        /**
         * <p>
         * Product information.
         * </p>
         * 
         * @param productInformationList
         *        Product information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productInformationList(Collection<ProductInformation> productInformationList);

        /**
         * <p>
         * Product information.
         * </p>
         * 
         * @param productInformationList
         *        Product information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productInformationList(ProductInformation... productInformationList);

        /**
         * <p>
         * Product information.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ProductInformation>.Builder} avoiding the
         * need to create one manually via {@link List<ProductInformation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProductInformation>.Builder#build()} is called immediately
         * and its result is passed to {@link #productInformationList(List<ProductInformation>)}.
         * 
         * @param productInformationList
         *        a consumer that will call methods on {@link List<ProductInformation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #productInformationList(List<ProductInformation>)
         */
        Builder productInformationList(Consumer<ProductInformation.Builder>... productInformationList);

        /**
         * <p>
         * Automated discovery information.
         * </p>
         * 
         * @param automatedDiscoveryInformation
         *        Automated discovery information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automatedDiscoveryInformation(AutomatedDiscoveryInformation automatedDiscoveryInformation);

        /**
         * <p>
         * Automated discovery information.
         * </p>
         * This is a convenience that creates an instance of the {@link AutomatedDiscoveryInformation.Builder} avoiding
         * the need to create one manually via {@link AutomatedDiscoveryInformation#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutomatedDiscoveryInformation.Builder#build()} is called
         * immediately and its result is passed to {@link #automatedDiscoveryInformation(AutomatedDiscoveryInformation)}
         * .
         * 
         * @param automatedDiscoveryInformation
         *        a consumer that will call methods on {@link AutomatedDiscoveryInformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #automatedDiscoveryInformation(AutomatedDiscoveryInformation)
         */
        default Builder automatedDiscoveryInformation(
                Consumer<AutomatedDiscoveryInformation.Builder> automatedDiscoveryInformation) {
            return automatedDiscoveryInformation(AutomatedDiscoveryInformation.builder()
                    .applyMutation(automatedDiscoveryInformation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String licenseConfigurationId;

        private String licenseConfigurationArn;

        private String name;

        private String description;

        private String licenseCountingType;

        private List<String> licenseRules = DefaultSdkAutoConstructList.getInstance();

        private Long licenseCount;

        private Boolean licenseCountHardLimit;

        private Boolean disassociateWhenNotFound;

        private Long consumedLicenses;

        private String status;

        private String ownerAccountId;

        private List<ConsumedLicenseSummary> consumedLicenseSummaryList = DefaultSdkAutoConstructList.getInstance();

        private List<ManagedResourceSummary> managedResourceSummaryList = DefaultSdkAutoConstructList.getInstance();

        private List<ProductInformation> productInformationList = DefaultSdkAutoConstructList.getInstance();

        private AutomatedDiscoveryInformation automatedDiscoveryInformation;

        private BuilderImpl() {
        }

        private BuilderImpl(LicenseConfiguration model) {
            licenseConfigurationId(model.licenseConfigurationId);
            licenseConfigurationArn(model.licenseConfigurationArn);
            name(model.name);
            description(model.description);
            licenseCountingType(model.licenseCountingType);
            licenseRules(model.licenseRules);
            licenseCount(model.licenseCount);
            licenseCountHardLimit(model.licenseCountHardLimit);
            disassociateWhenNotFound(model.disassociateWhenNotFound);
            consumedLicenses(model.consumedLicenses);
            status(model.status);
            ownerAccountId(model.ownerAccountId);
            consumedLicenseSummaryList(model.consumedLicenseSummaryList);
            managedResourceSummaryList(model.managedResourceSummaryList);
            productInformationList(model.productInformationList);
            automatedDiscoveryInformation(model.automatedDiscoveryInformation);
        }

        public final String getLicenseConfigurationId() {
            return licenseConfigurationId;
        }

        @Override
        public final Builder licenseConfigurationId(String licenseConfigurationId) {
            this.licenseConfigurationId = licenseConfigurationId;
            return this;
        }

        public final void setLicenseConfigurationId(String licenseConfigurationId) {
            this.licenseConfigurationId = licenseConfigurationId;
        }

        public final String getLicenseConfigurationArn() {
            return licenseConfigurationArn;
        }

        @Override
        public final Builder licenseConfigurationArn(String licenseConfigurationArn) {
            this.licenseConfigurationArn = licenseConfigurationArn;
            return this;
        }

        public final void setLicenseConfigurationArn(String licenseConfigurationArn) {
            this.licenseConfigurationArn = licenseConfigurationArn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getLicenseCountingType() {
            return licenseCountingType;
        }

        @Override
        public final Builder licenseCountingType(String licenseCountingType) {
            this.licenseCountingType = licenseCountingType;
            return this;
        }

        @Override
        public final Builder licenseCountingType(LicenseCountingType licenseCountingType) {
            this.licenseCountingType(licenseCountingType == null ? null : licenseCountingType.toString());
            return this;
        }

        public final void setLicenseCountingType(String licenseCountingType) {
            this.licenseCountingType = licenseCountingType;
        }

        public final Collection<String> getLicenseRules() {
            if (licenseRules instanceof SdkAutoConstructList) {
                return null;
            }
            return licenseRules;
        }

        @Override
        public final Builder licenseRules(Collection<String> licenseRules) {
            this.licenseRules = StringListCopier.copy(licenseRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenseRules(String... licenseRules) {
            licenseRules(Arrays.asList(licenseRules));
            return this;
        }

        public final void setLicenseRules(Collection<String> licenseRules) {
            this.licenseRules = StringListCopier.copy(licenseRules);
        }

        public final Long getLicenseCount() {
            return licenseCount;
        }

        @Override
        public final Builder licenseCount(Long licenseCount) {
            this.licenseCount = licenseCount;
            return this;
        }

        public final void setLicenseCount(Long licenseCount) {
            this.licenseCount = licenseCount;
        }

        public final Boolean getLicenseCountHardLimit() {
            return licenseCountHardLimit;
        }

        @Override
        public final Builder licenseCountHardLimit(Boolean licenseCountHardLimit) {
            this.licenseCountHardLimit = licenseCountHardLimit;
            return this;
        }

        public final void setLicenseCountHardLimit(Boolean licenseCountHardLimit) {
            this.licenseCountHardLimit = licenseCountHardLimit;
        }

        public final Boolean getDisassociateWhenNotFound() {
            return disassociateWhenNotFound;
        }

        @Override
        public final Builder disassociateWhenNotFound(Boolean disassociateWhenNotFound) {
            this.disassociateWhenNotFound = disassociateWhenNotFound;
            return this;
        }

        public final void setDisassociateWhenNotFound(Boolean disassociateWhenNotFound) {
            this.disassociateWhenNotFound = disassociateWhenNotFound;
        }

        public final Long getConsumedLicenses() {
            return consumedLicenses;
        }

        @Override
        public final Builder consumedLicenses(Long consumedLicenses) {
            this.consumedLicenses = consumedLicenses;
            return this;
        }

        public final void setConsumedLicenses(Long consumedLicenses) {
            this.consumedLicenses = consumedLicenses;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getOwnerAccountId() {
            return ownerAccountId;
        }

        @Override
        public final Builder ownerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
            return this;
        }

        public final void setOwnerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
        }

        public final List<ConsumedLicenseSummary.Builder> getConsumedLicenseSummaryList() {
            List<ConsumedLicenseSummary.Builder> result = ConsumedLicenseSummaryListCopier
                    .copyToBuilder(this.consumedLicenseSummaryList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder consumedLicenseSummaryList(Collection<ConsumedLicenseSummary> consumedLicenseSummaryList) {
            this.consumedLicenseSummaryList = ConsumedLicenseSummaryListCopier.copy(consumedLicenseSummaryList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder consumedLicenseSummaryList(ConsumedLicenseSummary... consumedLicenseSummaryList) {
            consumedLicenseSummaryList(Arrays.asList(consumedLicenseSummaryList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder consumedLicenseSummaryList(Consumer<ConsumedLicenseSummary.Builder>... consumedLicenseSummaryList) {
            consumedLicenseSummaryList(Stream.of(consumedLicenseSummaryList)
                    .map(c -> ConsumedLicenseSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setConsumedLicenseSummaryList(Collection<ConsumedLicenseSummary.BuilderImpl> consumedLicenseSummaryList) {
            this.consumedLicenseSummaryList = ConsumedLicenseSummaryListCopier.copyFromBuilder(consumedLicenseSummaryList);
        }

        public final List<ManagedResourceSummary.Builder> getManagedResourceSummaryList() {
            List<ManagedResourceSummary.Builder> result = ManagedResourceSummaryListCopier
                    .copyToBuilder(this.managedResourceSummaryList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder managedResourceSummaryList(Collection<ManagedResourceSummary> managedResourceSummaryList) {
            this.managedResourceSummaryList = ManagedResourceSummaryListCopier.copy(managedResourceSummaryList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedResourceSummaryList(ManagedResourceSummary... managedResourceSummaryList) {
            managedResourceSummaryList(Arrays.asList(managedResourceSummaryList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedResourceSummaryList(Consumer<ManagedResourceSummary.Builder>... managedResourceSummaryList) {
            managedResourceSummaryList(Stream.of(managedResourceSummaryList)
                    .map(c -> ManagedResourceSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setManagedResourceSummaryList(Collection<ManagedResourceSummary.BuilderImpl> managedResourceSummaryList) {
            this.managedResourceSummaryList = ManagedResourceSummaryListCopier.copyFromBuilder(managedResourceSummaryList);
        }

        public final List<ProductInformation.Builder> getProductInformationList() {
            List<ProductInformation.Builder> result = ProductInformationListCopier.copyToBuilder(this.productInformationList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder productInformationList(Collection<ProductInformation> productInformationList) {
            this.productInformationList = ProductInformationListCopier.copy(productInformationList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productInformationList(ProductInformation... productInformationList) {
            productInformationList(Arrays.asList(productInformationList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productInformationList(Consumer<ProductInformation.Builder>... productInformationList) {
            productInformationList(Stream.of(productInformationList)
                    .map(c -> ProductInformation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setProductInformationList(Collection<ProductInformation.BuilderImpl> productInformationList) {
            this.productInformationList = ProductInformationListCopier.copyFromBuilder(productInformationList);
        }

        public final AutomatedDiscoveryInformation.Builder getAutomatedDiscoveryInformation() {
            return automatedDiscoveryInformation != null ? automatedDiscoveryInformation.toBuilder() : null;
        }

        @Override
        public final Builder automatedDiscoveryInformation(AutomatedDiscoveryInformation automatedDiscoveryInformation) {
            this.automatedDiscoveryInformation = automatedDiscoveryInformation;
            return this;
        }

        public final void setAutomatedDiscoveryInformation(AutomatedDiscoveryInformation.BuilderImpl automatedDiscoveryInformation) {
            this.automatedDiscoveryInformation = automatedDiscoveryInformation != null ? automatedDiscoveryInformation.build()
                    : null;
        }

        @Override
        public LicenseConfiguration build() {
            return new LicenseConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
