/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
 * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
 * VCPU), tenancy (shared tenancy, Amazon EC2 Dedicated Instance, Amazon EC2 Dedicated Host, or any of these), host
 * affinity (how long a VM must be associated with a host), the number of licenses purchased and used.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LicenseConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LicenseConfiguration.Builder, LicenseConfiguration> {
    private static final SdkField<String> LICENSE_CONFIGURATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::licenseConfigurationId)).setter(setter(Builder::licenseConfigurationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseConfigurationId").build())
            .build();

    private static final SdkField<String> LICENSE_CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::licenseConfigurationArn)).setter(setter(Builder::licenseConfigurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseConfigurationArn").build())
            .build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> LICENSE_COUNTING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::licenseCountingTypeAsString)).setter(setter(Builder::licenseCountingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseCountingType").build())
            .build();

    private static final SdkField<List<String>> LICENSE_RULES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(LicenseConfiguration::licenseRules))
            .setter(setter(Builder::licenseRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> LICENSE_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(LicenseConfiguration::licenseCount)).setter(setter(Builder::licenseCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseCount").build()).build();

    private static final SdkField<Boolean> LICENSE_COUNT_HARD_LIMIT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(LicenseConfiguration::licenseCountHardLimit)).setter(setter(Builder::licenseCountHardLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseCountHardLimit").build())
            .build();

    private static final SdkField<Long> CONSUMED_LICENSES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(LicenseConfiguration::consumedLicenses)).setter(setter(Builder::consumedLicenses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsumedLicenses").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LicenseConfiguration::ownerAccountId)).setter(setter(Builder::ownerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerAccountId").build()).build();

    private static final SdkField<List<ConsumedLicenseSummary>> CONSUMED_LICENSE_SUMMARY_LIST_FIELD = SdkField
            .<List<ConsumedLicenseSummary>> builder(MarshallingType.LIST)
            .getter(getter(LicenseConfiguration::consumedLicenseSummaryList))
            .setter(setter(Builder::consumedLicenseSummaryList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsumedLicenseSummaryList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConsumedLicenseSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConsumedLicenseSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ManagedResourceSummary>> MANAGED_RESOURCE_SUMMARY_LIST_FIELD = SdkField
            .<List<ManagedResourceSummary>> builder(MarshallingType.LIST)
            .getter(getter(LicenseConfiguration::managedResourceSummaryList))
            .setter(setter(Builder::managedResourceSummaryList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedResourceSummaryList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ManagedResourceSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(ManagedResourceSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            LICENSE_CONFIGURATION_ID_FIELD, LICENSE_CONFIGURATION_ARN_FIELD, NAME_FIELD, DESCRIPTION_FIELD,
            LICENSE_COUNTING_TYPE_FIELD, LICENSE_RULES_FIELD, LICENSE_COUNT_FIELD, LICENSE_COUNT_HARD_LIMIT_FIELD,
            CONSUMED_LICENSES_FIELD, STATUS_FIELD, OWNER_ACCOUNT_ID_FIELD, CONSUMED_LICENSE_SUMMARY_LIST_FIELD,
            MANAGED_RESOURCE_SUMMARY_LIST_FIELD));

    private static final long serialVersionUID = 1L;

    private final String licenseConfigurationId;

    private final String licenseConfigurationArn;

    private final String name;

    private final String description;

    private final String licenseCountingType;

    private final List<String> licenseRules;

    private final Long licenseCount;

    private final Boolean licenseCountHardLimit;

    private final Long consumedLicenses;

    private final String status;

    private final String ownerAccountId;

    private final List<ConsumedLicenseSummary> consumedLicenseSummaryList;

    private final List<ManagedResourceSummary> managedResourceSummaryList;

    private LicenseConfiguration(BuilderImpl builder) {
        this.licenseConfigurationId = builder.licenseConfigurationId;
        this.licenseConfigurationArn = builder.licenseConfigurationArn;
        this.name = builder.name;
        this.description = builder.description;
        this.licenseCountingType = builder.licenseCountingType;
        this.licenseRules = builder.licenseRules;
        this.licenseCount = builder.licenseCount;
        this.licenseCountHardLimit = builder.licenseCountHardLimit;
        this.consumedLicenses = builder.consumedLicenses;
        this.status = builder.status;
        this.ownerAccountId = builder.ownerAccountId;
        this.consumedLicenseSummaryList = builder.consumedLicenseSummaryList;
        this.managedResourceSummaryList = builder.managedResourceSummaryList;
    }

    /**
     * <p>
     * Unique ID of the <code>LicenseConfiguration</code> object.
     * </p>
     * 
     * @return Unique ID of the <code>LicenseConfiguration</code> object.
     */
    public String licenseConfigurationId() {
        return licenseConfigurationId;
    }

    /**
     * <p>
     * ARN of the <code>LicenseConfiguration</code> object.
     * </p>
     * 
     * @return ARN of the <code>LicenseConfiguration</code> object.
     */
    public String licenseConfigurationArn() {
        return licenseConfigurationArn;
    }

    /**
     * <p>
     * Name of the license configuration.
     * </p>
     * 
     * @return Name of the license configuration.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Description of the license configuration.
     * </p>
     * 
     * @return Description of the license configuration.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * Dimension to use to track license inventory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #licenseCountingType} will return {@link LicenseCountingType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #licenseCountingTypeAsString}.
     * </p>
     * 
     * @return Dimension to use to track license inventory.
     * @see LicenseCountingType
     */
    public LicenseCountingType licenseCountingType() {
        return LicenseCountingType.fromValue(licenseCountingType);
    }

    /**
     * <p>
     * Dimension to use to track license inventory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #licenseCountingType} will return {@link LicenseCountingType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #licenseCountingTypeAsString}.
     * </p>
     * 
     * @return Dimension to use to track license inventory.
     * @see LicenseCountingType
     */
    public String licenseCountingTypeAsString() {
        return licenseCountingType;
    }

    /**
     * <p>
     * Array of configured License Manager rules.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Array of configured License Manager rules.
     */
    public List<String> licenseRules() {
        return licenseRules;
    }

    /**
     * <p>
     * Number of licenses managed by the license configuration.
     * </p>
     * 
     * @return Number of licenses managed by the license configuration.
     */
    public Long licenseCount() {
        return licenseCount;
    }

    /**
     * <p>
     * Sets the number of available licenses as a hard limit.
     * </p>
     * 
     * @return Sets the number of available licenses as a hard limit.
     */
    public Boolean licenseCountHardLimit() {
        return licenseCountHardLimit;
    }

    /**
     * <p>
     * Number of licenses consumed.
     * </p>
     * 
     * @return Number of licenses consumed.
     */
    public Long consumedLicenses() {
        return consumedLicenses;
    }

    /**
     * <p>
     * Status of the license configuration.
     * </p>
     * 
     * @return Status of the license configuration.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * Account ID of the license configuration's owner.
     * </p>
     * 
     * @return Account ID of the license configuration's owner.
     */
    public String ownerAccountId() {
        return ownerAccountId;
    }

    /**
     * <p>
     * List of summaries for licenses consumed by various resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return List of summaries for licenses consumed by various resources.
     */
    public List<ConsumedLicenseSummary> consumedLicenseSummaryList() {
        return consumedLicenseSummaryList;
    }

    /**
     * <p>
     * List of summaries for managed resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return List of summaries for managed resources.
     */
    public List<ManagedResourceSummary> managedResourceSummaryList() {
        return managedResourceSummaryList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(licenseConfigurationId());
        hashCode = 31 * hashCode + Objects.hashCode(licenseConfigurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(licenseCountingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(licenseRules());
        hashCode = 31 * hashCode + Objects.hashCode(licenseCount());
        hashCode = 31 * hashCode + Objects.hashCode(licenseCountHardLimit());
        hashCode = 31 * hashCode + Objects.hashCode(consumedLicenses());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(consumedLicenseSummaryList());
        hashCode = 31 * hashCode + Objects.hashCode(managedResourceSummaryList());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LicenseConfiguration)) {
            return false;
        }
        LicenseConfiguration other = (LicenseConfiguration) obj;
        return Objects.equals(licenseConfigurationId(), other.licenseConfigurationId())
                && Objects.equals(licenseConfigurationArn(), other.licenseConfigurationArn())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(licenseCountingTypeAsString(), other.licenseCountingTypeAsString())
                && Objects.equals(licenseRules(), other.licenseRules()) && Objects.equals(licenseCount(), other.licenseCount())
                && Objects.equals(licenseCountHardLimit(), other.licenseCountHardLimit())
                && Objects.equals(consumedLicenses(), other.consumedLicenses()) && Objects.equals(status(), other.status())
                && Objects.equals(ownerAccountId(), other.ownerAccountId())
                && Objects.equals(consumedLicenseSummaryList(), other.consumedLicenseSummaryList())
                && Objects.equals(managedResourceSummaryList(), other.managedResourceSummaryList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LicenseConfiguration").add("LicenseConfigurationId", licenseConfigurationId())
                .add("LicenseConfigurationArn", licenseConfigurationArn()).add("Name", name()).add("Description", description())
                .add("LicenseCountingType", licenseCountingTypeAsString()).add("LicenseRules", licenseRules())
                .add("LicenseCount", licenseCount()).add("LicenseCountHardLimit", licenseCountHardLimit())
                .add("ConsumedLicenses", consumedLicenses()).add("Status", status()).add("OwnerAccountId", ownerAccountId())
                .add("ConsumedLicenseSummaryList", consumedLicenseSummaryList())
                .add("ManagedResourceSummaryList", managedResourceSummaryList()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LicenseConfigurationId":
            return Optional.ofNullable(clazz.cast(licenseConfigurationId()));
        case "LicenseConfigurationArn":
            return Optional.ofNullable(clazz.cast(licenseConfigurationArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "LicenseCountingType":
            return Optional.ofNullable(clazz.cast(licenseCountingTypeAsString()));
        case "LicenseRules":
            return Optional.ofNullable(clazz.cast(licenseRules()));
        case "LicenseCount":
            return Optional.ofNullable(clazz.cast(licenseCount()));
        case "LicenseCountHardLimit":
            return Optional.ofNullable(clazz.cast(licenseCountHardLimit()));
        case "ConsumedLicenses":
            return Optional.ofNullable(clazz.cast(consumedLicenses()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "OwnerAccountId":
            return Optional.ofNullable(clazz.cast(ownerAccountId()));
        case "ConsumedLicenseSummaryList":
            return Optional.ofNullable(clazz.cast(consumedLicenseSummaryList()));
        case "ManagedResourceSummaryList":
            return Optional.ofNullable(clazz.cast(managedResourceSummaryList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LicenseConfiguration, T> g) {
        return obj -> g.apply((LicenseConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LicenseConfiguration> {
        /**
         * <p>
         * Unique ID of the <code>LicenseConfiguration</code> object.
         * </p>
         * 
         * @param licenseConfigurationId
         *        Unique ID of the <code>LicenseConfiguration</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseConfigurationId(String licenseConfigurationId);

        /**
         * <p>
         * ARN of the <code>LicenseConfiguration</code> object.
         * </p>
         * 
         * @param licenseConfigurationArn
         *        ARN of the <code>LicenseConfiguration</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseConfigurationArn(String licenseConfigurationArn);

        /**
         * <p>
         * Name of the license configuration.
         * </p>
         * 
         * @param name
         *        Name of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Description of the license configuration.
         * </p>
         * 
         * @param description
         *        Description of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Dimension to use to track license inventory.
         * </p>
         * 
         * @param licenseCountingType
         *        Dimension to use to track license inventory.
         * @see LicenseCountingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseCountingType
         */
        Builder licenseCountingType(String licenseCountingType);

        /**
         * <p>
         * Dimension to use to track license inventory.
         * </p>
         * 
         * @param licenseCountingType
         *        Dimension to use to track license inventory.
         * @see LicenseCountingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseCountingType
         */
        Builder licenseCountingType(LicenseCountingType licenseCountingType);

        /**
         * <p>
         * Array of configured License Manager rules.
         * </p>
         * 
         * @param licenseRules
         *        Array of configured License Manager rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseRules(Collection<String> licenseRules);

        /**
         * <p>
         * Array of configured License Manager rules.
         * </p>
         * 
         * @param licenseRules
         *        Array of configured License Manager rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseRules(String... licenseRules);

        /**
         * <p>
         * Number of licenses managed by the license configuration.
         * </p>
         * 
         * @param licenseCount
         *        Number of licenses managed by the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseCount(Long licenseCount);

        /**
         * <p>
         * Sets the number of available licenses as a hard limit.
         * </p>
         * 
         * @param licenseCountHardLimit
         *        Sets the number of available licenses as a hard limit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseCountHardLimit(Boolean licenseCountHardLimit);

        /**
         * <p>
         * Number of licenses consumed.
         * </p>
         * 
         * @param consumedLicenses
         *        Number of licenses consumed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLicenses(Long consumedLicenses);

        /**
         * <p>
         * Status of the license configuration.
         * </p>
         * 
         * @param status
         *        Status of the license configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Account ID of the license configuration's owner.
         * </p>
         * 
         * @param ownerAccountId
         *        Account ID of the license configuration's owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccountId(String ownerAccountId);

        /**
         * <p>
         * List of summaries for licenses consumed by various resources.
         * </p>
         * 
         * @param consumedLicenseSummaryList
         *        List of summaries for licenses consumed by various resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLicenseSummaryList(Collection<ConsumedLicenseSummary> consumedLicenseSummaryList);

        /**
         * <p>
         * List of summaries for licenses consumed by various resources.
         * </p>
         * 
         * @param consumedLicenseSummaryList
         *        List of summaries for licenses consumed by various resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLicenseSummaryList(ConsumedLicenseSummary... consumedLicenseSummaryList);

        /**
         * <p>
         * List of summaries for licenses consumed by various resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ConsumedLicenseSummary>.Builder} avoiding
         * the need to create one manually via {@link List<ConsumedLicenseSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ConsumedLicenseSummary>.Builder#build()} is called
         * immediately and its result is passed to {@link #consumedLicenseSummaryList(List<ConsumedLicenseSummary>)}.
         * 
         * @param consumedLicenseSummaryList
         *        a consumer that will call methods on {@link List<ConsumedLicenseSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #consumedLicenseSummaryList(List<ConsumedLicenseSummary>)
         */
        Builder consumedLicenseSummaryList(Consumer<ConsumedLicenseSummary.Builder>... consumedLicenseSummaryList);

        /**
         * <p>
         * List of summaries for managed resources.
         * </p>
         * 
         * @param managedResourceSummaryList
         *        List of summaries for managed resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedResourceSummaryList(Collection<ManagedResourceSummary> managedResourceSummaryList);

        /**
         * <p>
         * List of summaries for managed resources.
         * </p>
         * 
         * @param managedResourceSummaryList
         *        List of summaries for managed resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedResourceSummaryList(ManagedResourceSummary... managedResourceSummaryList);

        /**
         * <p>
         * List of summaries for managed resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ManagedResourceSummary>.Builder} avoiding
         * the need to create one manually via {@link List<ManagedResourceSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ManagedResourceSummary>.Builder#build()} is called
         * immediately and its result is passed to {@link #managedResourceSummaryList(List<ManagedResourceSummary>)}.
         * 
         * @param managedResourceSummaryList
         *        a consumer that will call methods on {@link List<ManagedResourceSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #managedResourceSummaryList(List<ManagedResourceSummary>)
         */
        Builder managedResourceSummaryList(Consumer<ManagedResourceSummary.Builder>... managedResourceSummaryList);
    }

    static final class BuilderImpl implements Builder {
        private String licenseConfigurationId;

        private String licenseConfigurationArn;

        private String name;

        private String description;

        private String licenseCountingType;

        private List<String> licenseRules = DefaultSdkAutoConstructList.getInstance();

        private Long licenseCount;

        private Boolean licenseCountHardLimit;

        private Long consumedLicenses;

        private String status;

        private String ownerAccountId;

        private List<ConsumedLicenseSummary> consumedLicenseSummaryList = DefaultSdkAutoConstructList.getInstance();

        private List<ManagedResourceSummary> managedResourceSummaryList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LicenseConfiguration model) {
            licenseConfigurationId(model.licenseConfigurationId);
            licenseConfigurationArn(model.licenseConfigurationArn);
            name(model.name);
            description(model.description);
            licenseCountingType(model.licenseCountingType);
            licenseRules(model.licenseRules);
            licenseCount(model.licenseCount);
            licenseCountHardLimit(model.licenseCountHardLimit);
            consumedLicenses(model.consumedLicenses);
            status(model.status);
            ownerAccountId(model.ownerAccountId);
            consumedLicenseSummaryList(model.consumedLicenseSummaryList);
            managedResourceSummaryList(model.managedResourceSummaryList);
        }

        public final String getLicenseConfigurationId() {
            return licenseConfigurationId;
        }

        @Override
        public final Builder licenseConfigurationId(String licenseConfigurationId) {
            this.licenseConfigurationId = licenseConfigurationId;
            return this;
        }

        public final void setLicenseConfigurationId(String licenseConfigurationId) {
            this.licenseConfigurationId = licenseConfigurationId;
        }

        public final String getLicenseConfigurationArn() {
            return licenseConfigurationArn;
        }

        @Override
        public final Builder licenseConfigurationArn(String licenseConfigurationArn) {
            this.licenseConfigurationArn = licenseConfigurationArn;
            return this;
        }

        public final void setLicenseConfigurationArn(String licenseConfigurationArn) {
            this.licenseConfigurationArn = licenseConfigurationArn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getLicenseCountingTypeAsString() {
            return licenseCountingType;
        }

        @Override
        public final Builder licenseCountingType(String licenseCountingType) {
            this.licenseCountingType = licenseCountingType;
            return this;
        }

        @Override
        public final Builder licenseCountingType(LicenseCountingType licenseCountingType) {
            this.licenseCountingType(licenseCountingType == null ? null : licenseCountingType.toString());
            return this;
        }

        public final void setLicenseCountingType(String licenseCountingType) {
            this.licenseCountingType = licenseCountingType;
        }

        public final Collection<String> getLicenseRules() {
            return licenseRules;
        }

        @Override
        public final Builder licenseRules(Collection<String> licenseRules) {
            this.licenseRules = StringListCopier.copy(licenseRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenseRules(String... licenseRules) {
            licenseRules(Arrays.asList(licenseRules));
            return this;
        }

        public final void setLicenseRules(Collection<String> licenseRules) {
            this.licenseRules = StringListCopier.copy(licenseRules);
        }

        public final Long getLicenseCount() {
            return licenseCount;
        }

        @Override
        public final Builder licenseCount(Long licenseCount) {
            this.licenseCount = licenseCount;
            return this;
        }

        public final void setLicenseCount(Long licenseCount) {
            this.licenseCount = licenseCount;
        }

        public final Boolean getLicenseCountHardLimit() {
            return licenseCountHardLimit;
        }

        @Override
        public final Builder licenseCountHardLimit(Boolean licenseCountHardLimit) {
            this.licenseCountHardLimit = licenseCountHardLimit;
            return this;
        }

        public final void setLicenseCountHardLimit(Boolean licenseCountHardLimit) {
            this.licenseCountHardLimit = licenseCountHardLimit;
        }

        public final Long getConsumedLicenses() {
            return consumedLicenses;
        }

        @Override
        public final Builder consumedLicenses(Long consumedLicenses) {
            this.consumedLicenses = consumedLicenses;
            return this;
        }

        public final void setConsumedLicenses(Long consumedLicenses) {
            this.consumedLicenses = consumedLicenses;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getOwnerAccountId() {
            return ownerAccountId;
        }

        @Override
        public final Builder ownerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
            return this;
        }

        public final void setOwnerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
        }

        public final Collection<ConsumedLicenseSummary.Builder> getConsumedLicenseSummaryList() {
            return consumedLicenseSummaryList != null ? consumedLicenseSummaryList.stream()
                    .map(ConsumedLicenseSummary::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder consumedLicenseSummaryList(Collection<ConsumedLicenseSummary> consumedLicenseSummaryList) {
            this.consumedLicenseSummaryList = ConsumedLicenseSummaryListCopier.copy(consumedLicenseSummaryList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder consumedLicenseSummaryList(ConsumedLicenseSummary... consumedLicenseSummaryList) {
            consumedLicenseSummaryList(Arrays.asList(consumedLicenseSummaryList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder consumedLicenseSummaryList(Consumer<ConsumedLicenseSummary.Builder>... consumedLicenseSummaryList) {
            consumedLicenseSummaryList(Stream.of(consumedLicenseSummaryList)
                    .map(c -> ConsumedLicenseSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setConsumedLicenseSummaryList(Collection<ConsumedLicenseSummary.BuilderImpl> consumedLicenseSummaryList) {
            this.consumedLicenseSummaryList = ConsumedLicenseSummaryListCopier.copyFromBuilder(consumedLicenseSummaryList);
        }

        public final Collection<ManagedResourceSummary.Builder> getManagedResourceSummaryList() {
            return managedResourceSummaryList != null ? managedResourceSummaryList.stream()
                    .map(ManagedResourceSummary::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder managedResourceSummaryList(Collection<ManagedResourceSummary> managedResourceSummaryList) {
            this.managedResourceSummaryList = ManagedResourceSummaryListCopier.copy(managedResourceSummaryList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedResourceSummaryList(ManagedResourceSummary... managedResourceSummaryList) {
            managedResourceSummaryList(Arrays.asList(managedResourceSummaryList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedResourceSummaryList(Consumer<ManagedResourceSummary.Builder>... managedResourceSummaryList) {
            managedResourceSummaryList(Stream.of(managedResourceSummaryList)
                    .map(c -> ManagedResourceSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setManagedResourceSummaryList(Collection<ManagedResourceSummary.BuilderImpl> managedResourceSummaryList) {
            this.managedResourceSummaryList = ManagedResourceSummaryListCopier.copyFromBuilder(managedResourceSummaryList);
        }

        @Override
        public LicenseConfiguration build() {
            return new LicenseConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
