/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanagerlinuxsubscriptions;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.InternalServerException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.LicenseManagerLinuxSubscriptionsException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.LicenseManagerLinuxSubscriptionsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ThrottlingException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ValidationException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.paginators.ListLinuxSubscriptionInstancesPublisher;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.paginators.ListLinuxSubscriptionsPublisher;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.ListLinuxSubscriptionInstancesRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.ListLinuxSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LicenseManagerLinuxSubscriptionsAsyncClient}.
 *
 * @see LicenseManagerLinuxSubscriptionsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerLinuxSubscriptionsAsyncClient implements LicenseManagerLinuxSubscriptionsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLicenseManagerLinuxSubscriptionsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LicenseManagerLinuxSubscriptionsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLicenseManagerLinuxSubscriptionsAsyncClient(
            LicenseManagerLinuxSubscriptionsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Lists the Linux subscriptions service settings.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return A Java Future containing the result of the GetServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.GetServiceSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceSettingsResponse> getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                            .withOperationName("GetServiceSettings")
                            .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getServiceSettingsRequest));
            CompletableFuture<GetServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the running Amazon EC2 instances that were discovered with commercial Linux subscriptions.
     * </p>
     *
     * @param listLinuxSubscriptionInstancesRequest
     *        NextToken length limit is half of ddb accepted limit. Increase this limit if parameters in request
     *        increases.
     * @return A Java Future containing the result of the ListLinuxSubscriptionInstances operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListLinuxSubscriptionInstances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListLinuxSubscriptionInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLinuxSubscriptionInstancesResponse> listLinuxSubscriptionInstances(
            ListLinuxSubscriptionInstancesRequest listLinuxSubscriptionInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listLinuxSubscriptionInstancesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinuxSubscriptionInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLinuxSubscriptionInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLinuxSubscriptionInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLinuxSubscriptionInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLinuxSubscriptionInstancesRequest, ListLinuxSubscriptionInstancesResponse>()
                            .withOperationName("ListLinuxSubscriptionInstances")
                            .withMarshaller(new ListLinuxSubscriptionInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLinuxSubscriptionInstancesRequest));
            CompletableFuture<ListLinuxSubscriptionInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the running Amazon EC2 instances that were discovered with commercial Linux subscriptions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLinuxSubscriptionInstances(software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.paginators.ListLinuxSubscriptionInstancesPublisher publisher = client.listLinuxSubscriptionInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.paginators.ListLinuxSubscriptionInstancesPublisher publisher = client.listLinuxSubscriptionInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLinuxSubscriptionInstances(software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLinuxSubscriptionInstancesRequest
     *        NextToken length limit is half of ddb accepted limit. Increase this limit if parameters in request
     *        increases.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListLinuxSubscriptionInstances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListLinuxSubscriptionInstances"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLinuxSubscriptionInstancesPublisher listLinuxSubscriptionInstancesPaginator(
            ListLinuxSubscriptionInstancesRequest listLinuxSubscriptionInstancesRequest) {
        return new ListLinuxSubscriptionInstancesPublisher(this, applyPaginatorUserAgent(listLinuxSubscriptionInstancesRequest));
    }

    /**
     * <p>
     * Lists the Linux subscriptions that have been discovered. If you have linked your organization, the returned
     * results will include data aggregated across your accounts in Organizations.
     * </p>
     *
     * @param listLinuxSubscriptionsRequest
     *        NextToken length limit is half of ddb accepted limit. Increase this limit if parameters in request
     *        increases.
     * @return A Java Future containing the result of the ListLinuxSubscriptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListLinuxSubscriptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListLinuxSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLinuxSubscriptionsResponse> listLinuxSubscriptions(
            ListLinuxSubscriptionsRequest listLinuxSubscriptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLinuxSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinuxSubscriptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLinuxSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLinuxSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLinuxSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLinuxSubscriptionsRequest, ListLinuxSubscriptionsResponse>()
                            .withOperationName("ListLinuxSubscriptions")
                            .withMarshaller(new ListLinuxSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLinuxSubscriptionsRequest));
            CompletableFuture<ListLinuxSubscriptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Linux subscriptions that have been discovered. If you have linked your organization, the returned
     * results will include data aggregated across your accounts in Organizations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLinuxSubscriptions(software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.paginators.ListLinuxSubscriptionsPublisher publisher = client.listLinuxSubscriptionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.paginators.ListLinuxSubscriptionsPublisher publisher = client.listLinuxSubscriptionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLinuxSubscriptions(software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLinuxSubscriptionsRequest
     *        NextToken length limit is half of ddb accepted limit. Increase this limit if parameters in request
     *        increases.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListLinuxSubscriptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListLinuxSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLinuxSubscriptionsPublisher listLinuxSubscriptionsPaginator(
            ListLinuxSubscriptionsRequest listLinuxSubscriptionsRequest) {
        return new ListLinuxSubscriptionsPublisher(this, applyPaginatorUserAgent(listLinuxSubscriptionsRequest));
    }

    /**
     * <p>
     * Updates the service settings for Linux subscriptions.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return A Java Future containing the result of the UpdateServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.UpdateServiceSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceSettingsResponse> updateServiceSettings(
            UpdateServiceSettingsRequest updateServiceSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                            .withOperationName("UpdateServiceSettings")
                            .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateServiceSettingsRequest));
            CompletableFuture<UpdateServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LicenseManagerLinuxSubscriptionsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerLinuxSubscriptionsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LicenseManagerLinuxSubscriptionsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
