/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanagerusersubscriptions;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.AccessDeniedException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.AssociateUserRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.AssociateUserResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ConflictException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DeregisterIdentityProviderRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DeregisterIdentityProviderResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DisassociateUserRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DisassociateUserResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.InternalServerException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.LicenseManagerUserSubscriptionsException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.LicenseManagerUserSubscriptionsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.RegisterIdentityProviderRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.RegisterIdentityProviderResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ResourceNotFoundException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StartProductSubscriptionRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StartProductSubscriptionResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StopProductSubscriptionRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StopProductSubscriptionResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ThrottlingException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.UpdateIdentityProviderSettingsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.UpdateIdentityProviderSettingsResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ValidationException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListIdentityProvidersPublisher;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListInstancesPublisher;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListProductSubscriptionsPublisher;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListUserAssociationsPublisher;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.AssociateUserRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.DeregisterIdentityProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.DisassociateUserRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListIdentityProvidersRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListProductSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListUserAssociationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.RegisterIdentityProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.StartProductSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.StopProductSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.UpdateIdentityProviderSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LicenseManagerUserSubscriptionsAsyncClient}.
 *
 * @see LicenseManagerUserSubscriptionsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerUserSubscriptionsAsyncClient implements LicenseManagerUserSubscriptionsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLicenseManagerUserSubscriptionsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLicenseManagerUserSubscriptionsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates the user to an EC2 instance to utilize user-based subscriptions.
     * </p>
     * <note>
     * <p>
     * Your estimated bill for charges on the number of users and related costs will take 48 hours to appear for billing
     * periods that haven't closed (marked as <b>Pending</b> billing status) in Amazon Web Services Billing. For more
     * information, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/invoice.html">Viewing your
     * monthly charges</a> in the <i>Amazon Web Services Billing User Guide</i>.
     * </p>
     * </note>
     *
     * @param associateUserRequest
     * @return A Java Future containing the result of the AssociateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.AssociateUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/AssociateUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateUserResponse> associateUser(AssociateUserRequest associateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AssociateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateUserRequest, AssociateUserResponse>()
                            .withOperationName("AssociateUser")
                            .withMarshaller(new AssociateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateUserRequest));
            CompletableFuture<AssociateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters the identity provider from providing user-based subscriptions.
     * </p>
     *
     * @param deregisterIdentityProviderRequest
     * @return A Java Future containing the result of the DeregisterIdentityProvider operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.DeregisterIdentityProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/DeregisterIdentityProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterIdentityProviderResponse> deregisterIdentityProvider(
            DeregisterIdentityProviderRequest deregisterIdentityProviderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterIdentityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterIdentityProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterIdentityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterIdentityProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterIdentityProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterIdentityProviderRequest, DeregisterIdentityProviderResponse>()
                            .withOperationName("DeregisterIdentityProvider")
                            .withMarshaller(new DeregisterIdentityProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterIdentityProviderRequest));
            CompletableFuture<DeregisterIdentityProviderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the user from an EC2 instance providing user-based subscriptions.
     * </p>
     *
     * @param disassociateUserRequest
     * @return A Java Future containing the result of the DisassociateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.DisassociateUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/DisassociateUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateUserResponse> disassociateUser(DisassociateUserRequest disassociateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateUserRequest, DisassociateUserResponse>()
                            .withOperationName("DisassociateUser")
                            .withMarshaller(new DisassociateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateUserRequest));
            CompletableFuture<DisassociateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the identity providers for user-based subscriptions.
     * </p>
     *
     * @param listIdentityProvidersRequest
     * @return A Java Future containing the result of the ListIdentityProviders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListIdentityProviders
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListIdentityProviders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIdentityProvidersResponse> listIdentityProviders(
            ListIdentityProvidersRequest listIdentityProvidersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentityProvidersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentityProviders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIdentityProvidersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIdentityProvidersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIdentityProvidersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIdentityProvidersRequest, ListIdentityProvidersResponse>()
                            .withOperationName("ListIdentityProviders")
                            .withMarshaller(new ListIdentityProvidersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIdentityProvidersRequest));
            CompletableFuture<ListIdentityProvidersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the identity providers for user-based subscriptions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listIdentityProviders(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListIdentityProvidersPublisher publisher = client.listIdentityProvidersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListIdentityProvidersPublisher publisher = client.listIdentityProvidersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listIdentityProviders(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listIdentityProvidersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListIdentityProviders
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListIdentityProviders"
     *      target="_top">AWS API Documentation</a>
     */
    public ListIdentityProvidersPublisher listIdentityProvidersPaginator(ListIdentityProvidersRequest listIdentityProvidersRequest) {
        return new ListIdentityProvidersPublisher(this, applyPaginatorUserAgent(listIdentityProvidersRequest));
    }

    /**
     * <p>
     * Lists the EC2 instances providing user-based subscriptions.
     * </p>
     *
     * @param listInstancesRequest
     * @return A Java Future containing the result of the ListInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListInstances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInstancesResponse> listInstances(ListInstancesRequest listInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                            .withOperationName("ListInstances")
                            .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInstancesRequest));
            CompletableFuture<ListInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the EC2 instances providing user-based subscriptions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInstances(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListInstancesPublisher publisher = client.listInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListInstancesPublisher publisher = client.listInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstances(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInstancesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListInstances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListInstances"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInstancesPublisher listInstancesPaginator(ListInstancesRequest listInstancesRequest) {
        return new ListInstancesPublisher(this, applyPaginatorUserAgent(listInstancesRequest));
    }

    /**
     * <p>
     * Lists the user-based subscription products available from an identity provider.
     * </p>
     *
     * @param listProductSubscriptionsRequest
     * @return A Java Future containing the result of the ListProductSubscriptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListProductSubscriptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListProductSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProductSubscriptionsResponse> listProductSubscriptions(
            ListProductSubscriptionsRequest listProductSubscriptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProductSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProductSubscriptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProductSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProductSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProductSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProductSubscriptionsRequest, ListProductSubscriptionsResponse>()
                            .withOperationName("ListProductSubscriptions")
                            .withMarshaller(new ListProductSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProductSubscriptionsRequest));
            CompletableFuture<ListProductSubscriptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the user-based subscription products available from an identity provider.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProductSubscriptions(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListProductSubscriptionsPublisher publisher = client.listProductSubscriptionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListProductSubscriptionsPublisher publisher = client.listProductSubscriptionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProductSubscriptions(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProductSubscriptionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListProductSubscriptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListProductSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListProductSubscriptionsPublisher listProductSubscriptionsPaginator(
            ListProductSubscriptionsRequest listProductSubscriptionsRequest) {
        return new ListProductSubscriptionsPublisher(this, applyPaginatorUserAgent(listProductSubscriptionsRequest));
    }

    /**
     * <p>
     * Lists user associations for an identity provider.
     * </p>
     *
     * @param listUserAssociationsRequest
     * @return A Java Future containing the result of the ListUserAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListUserAssociations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListUserAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListUserAssociationsResponse> listUserAssociations(
            ListUserAssociationsRequest listUserAssociationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUserAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUserAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUserAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListUserAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUserAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUserAssociationsRequest, ListUserAssociationsResponse>()
                            .withOperationName("ListUserAssociations")
                            .withMarshaller(new ListUserAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUserAssociationsRequest));
            CompletableFuture<ListUserAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists user associations for an identity provider.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listUserAssociations(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListUserAssociationsPublisher publisher = client.listUserAssociationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.licensemanagerusersubscriptions.paginators.ListUserAssociationsPublisher publisher = client.listUserAssociationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUserAssociations(software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listUserAssociationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.ListUserAssociations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListUserAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListUserAssociationsPublisher listUserAssociationsPaginator(ListUserAssociationsRequest listUserAssociationsRequest) {
        return new ListUserAssociationsPublisher(this, applyPaginatorUserAgent(listUserAssociationsRequest));
    }

    /**
     * <p>
     * Registers an identity provider for user-based subscriptions.
     * </p>
     *
     * @param registerIdentityProviderRequest
     * @return A Java Future containing the result of the RegisterIdentityProvider operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.RegisterIdentityProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/RegisterIdentityProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterIdentityProviderResponse> registerIdentityProvider(
            RegisterIdentityProviderRequest registerIdentityProviderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerIdentityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterIdentityProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterIdentityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterIdentityProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterIdentityProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterIdentityProviderRequest, RegisterIdentityProviderResponse>()
                            .withOperationName("RegisterIdentityProvider")
                            .withMarshaller(new RegisterIdentityProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerIdentityProviderRequest));
            CompletableFuture<RegisterIdentityProviderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a product subscription for a user with the specified identity provider.
     * </p>
     * <note>
     * <p>
     * Your estimated bill for charges on the number of users and related costs will take 48 hours to appear for billing
     * periods that haven't closed (marked as <b>Pending</b> billing status) in Amazon Web Services Billing. For more
     * information, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/invoice.html">Viewing your
     * monthly charges</a> in the <i>Amazon Web Services Billing User Guide</i>.
     * </p>
     * </note>
     *
     * @param startProductSubscriptionRequest
     * @return A Java Future containing the result of the StartProductSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.StartProductSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/StartProductSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartProductSubscriptionResponse> startProductSubscription(
            StartProductSubscriptionRequest startProductSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startProductSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartProductSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartProductSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartProductSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartProductSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartProductSubscriptionRequest, StartProductSubscriptionResponse>()
                            .withOperationName("StartProductSubscription")
                            .withMarshaller(new StartProductSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startProductSubscriptionRequest));
            CompletableFuture<StartProductSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a product subscription for a user with the specified identity provider.
     * </p>
     *
     * @param stopProductSubscriptionRequest
     * @return A Java Future containing the result of the StopProductSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ConflictException The request couldn't be completed because it conflicted with the current state of
     *         the resource.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.StopProductSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/StopProductSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopProductSubscriptionResponse> stopProductSubscription(
            StopProductSubscriptionRequest stopProductSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopProductSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopProductSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopProductSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopProductSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopProductSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopProductSubscriptionRequest, StopProductSubscriptionResponse>()
                            .withOperationName("StopProductSubscription")
                            .withMarshaller(new StopProductSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopProductSubscriptionRequest));
            CompletableFuture<StopProductSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates additional product configuration settings for the registered identity provider.
     * </p>
     *
     * @param updateIdentityProviderSettingsRequest
     * @return A Java Future containing the result of the UpdateIdentityProviderSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerUserSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerUserSubscriptionsAsyncClient.UpdateIdentityProviderSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/UpdateIdentityProviderSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIdentityProviderSettingsResponse> updateIdentityProviderSettings(
            UpdateIdentityProviderSettingsRequest updateIdentityProviderSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateIdentityProviderSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIdentityProviderSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIdentityProviderSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateIdentityProviderSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIdentityProviderSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIdentityProviderSettingsRequest, UpdateIdentityProviderSettingsResponse>()
                            .withOperationName("UpdateIdentityProviderSettings")
                            .withMarshaller(new UpdateIdentityProviderSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateIdentityProviderSettingsRequest));
            CompletableFuture<UpdateIdentityProviderSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerUserSubscriptionsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LicenseManagerUserSubscriptionsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
