/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanagerusersubscriptions;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.internal.LicenseManagerUserSubscriptionsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.AccessDeniedException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.AssociateUserRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.AssociateUserResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ConflictException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DeregisterIdentityProviderRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DeregisterIdentityProviderResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DisassociateUserRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.DisassociateUserResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.InternalServerException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.LicenseManagerUserSubscriptionsException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListIdentityProvidersResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListInstancesResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListProductSubscriptionsResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ListUserAssociationsResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.RegisterIdentityProviderRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.RegisterIdentityProviderResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ResourceNotFoundException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StartProductSubscriptionRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StartProductSubscriptionResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StopProductSubscriptionRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.StopProductSubscriptionResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ThrottlingException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.UpdateIdentityProviderSettingsRequest;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.UpdateIdentityProviderSettingsResponse;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.model.ValidationException;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.AssociateUserRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.DeregisterIdentityProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.DisassociateUserRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListIdentityProvidersRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListProductSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.ListUserAssociationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.RegisterIdentityProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.StartProductSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.StopProductSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerusersubscriptions.transform.UpdateIdentityProviderSettingsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LicenseManagerUserSubscriptionsClient}.
 *
 * @see LicenseManagerUserSubscriptionsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerUserSubscriptionsClient implements LicenseManagerUserSubscriptionsClient {
    private static final Logger log = Logger.loggerFor(DefaultLicenseManagerUserSubscriptionsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLicenseManagerUserSubscriptionsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates the user to an EC2 instance to utilize user-based subscriptions.
     * </p>
     * <note>
     * <p>
     * Your estimated bill for charges on the number of users and related costs will take 48 hours to appear for billing
     * periods that haven't closed (marked as <b>Pending</b> billing status) in Amazon Web Services Billing. For more
     * information, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/invoice.html">Viewing your
     * monthly charges</a> in the <i>Amazon Web Services Billing User Guide</i>.
     * </p>
     * </note>
     *
     * @param associateUserRequest
     * @return Result of the AssociateUser operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.AssociateUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/AssociateUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateUserResponse associateUser(AssociateUserRequest associateUserRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateUser");

            return clientHandler.execute(new ClientExecutionParams<AssociateUserRequest, AssociateUserResponse>()
                    .withOperationName("AssociateUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(associateUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters the identity provider from providing user-based subscriptions.
     * </p>
     *
     * @param deregisterIdentityProviderRequest
     * @return Result of the DeregisterIdentityProvider operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.DeregisterIdentityProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/DeregisterIdentityProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterIdentityProviderResponse deregisterIdentityProvider(
            DeregisterIdentityProviderRequest deregisterIdentityProviderRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterIdentityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterIdentityProviderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterIdentityProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterIdentityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterIdentityProvider");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterIdentityProviderRequest, DeregisterIdentityProviderResponse>()
                            .withOperationName("DeregisterIdentityProvider").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deregisterIdentityProviderRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterIdentityProviderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates the user from an EC2 instance providing user-based subscriptions.
     * </p>
     *
     * @param disassociateUserRequest
     * @return Result of the DisassociateUser operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.DisassociateUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/DisassociateUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateUserResponse disassociateUser(DisassociateUserRequest disassociateUserRequest)
            throws ServiceQuotaExceededException, ConflictException, ValidationException, ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisassociateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateUser");

            return clientHandler.execute(new ClientExecutionParams<DisassociateUserRequest, DisassociateUserResponse>()
                    .withOperationName("DisassociateUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disassociateUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the identity providers for user-based subscriptions.
     * </p>
     *
     * @param listIdentityProvidersRequest
     * @return Result of the ListIdentityProviders operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.ListIdentityProviders
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListIdentityProviders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIdentityProvidersResponse listIdentityProviders(ListIdentityProvidersRequest listIdentityProvidersRequest)
            throws ServiceQuotaExceededException, ConflictException, ValidationException, ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIdentityProvidersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListIdentityProvidersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIdentityProvidersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentityProvidersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentityProviders");

            return clientHandler.execute(new ClientExecutionParams<ListIdentityProvidersRequest, ListIdentityProvidersResponse>()
                    .withOperationName("ListIdentityProviders").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listIdentityProvidersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIdentityProvidersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the EC2 instances providing user-based subscriptions.
     * </p>
     *
     * @param listInstancesRequest
     * @return Result of the ListInstances operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.ListInstances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstancesResponse listInstances(ListInstancesRequest listInstancesRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstancesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstances");

            return clientHandler.execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                    .withOperationName("ListInstances").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the user-based subscription products available from an identity provider.
     * </p>
     *
     * @param listProductSubscriptionsRequest
     * @return Result of the ListProductSubscriptions operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.ListProductSubscriptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListProductSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProductSubscriptionsResponse listProductSubscriptions(
            ListProductSubscriptionsRequest listProductSubscriptionsRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProductSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProductSubscriptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProductSubscriptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProductSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProductSubscriptions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProductSubscriptionsRequest, ListProductSubscriptionsResponse>()
                            .withOperationName("ListProductSubscriptions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProductSubscriptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProductSubscriptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists user associations for an identity provider.
     * </p>
     *
     * @param listUserAssociationsRequest
     * @return Result of the ListUserAssociations operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.ListUserAssociations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/ListUserAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListUserAssociationsResponse listUserAssociations(ListUserAssociationsRequest listUserAssociationsRequest)
            throws ServiceQuotaExceededException, ConflictException, ValidationException, ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUserAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListUserAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUserAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUserAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUserAssociations");

            return clientHandler.execute(new ClientExecutionParams<ListUserAssociationsRequest, ListUserAssociationsResponse>()
                    .withOperationName("ListUserAssociations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listUserAssociationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListUserAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers an identity provider for user-based subscriptions.
     * </p>
     *
     * @param registerIdentityProviderRequest
     * @return Result of the RegisterIdentityProvider operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.RegisterIdentityProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/RegisterIdentityProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterIdentityProviderResponse registerIdentityProvider(
            RegisterIdentityProviderRequest registerIdentityProviderRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterIdentityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterIdentityProviderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerIdentityProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerIdentityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterIdentityProvider");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterIdentityProviderRequest, RegisterIdentityProviderResponse>()
                            .withOperationName("RegisterIdentityProvider").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(registerIdentityProviderRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterIdentityProviderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a product subscription for a user with the specified identity provider.
     * </p>
     * <note>
     * <p>
     * Your estimated bill for charges on the number of users and related costs will take 48 hours to appear for billing
     * periods that haven't closed (marked as <b>Pending</b> billing status) in Amazon Web Services Billing. For more
     * information, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/invoice.html">Viewing your
     * monthly charges</a> in the <i>Amazon Web Services Billing User Guide</i>.
     * </p>
     * </note>
     *
     * @param startProductSubscriptionRequest
     * @return Result of the StartProductSubscription operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.StartProductSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/StartProductSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartProductSubscriptionResponse startProductSubscription(
            StartProductSubscriptionRequest startProductSubscriptionRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartProductSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartProductSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startProductSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startProductSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartProductSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<StartProductSubscriptionRequest, StartProductSubscriptionResponse>()
                            .withOperationName("StartProductSubscription").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startProductSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartProductSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a product subscription for a user with the specified identity provider.
     * </p>
     *
     * @param stopProductSubscriptionRequest
     * @return Result of the StopProductSubscription operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request failed because a service quota is exceeded.
     * @throws ConflictException
     *         The request couldn't be completed because it conflicted with the current state of the resource.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws ResourceNotFoundException
     *         The resource couldn't be found.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.StopProductSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/StopProductSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopProductSubscriptionResponse stopProductSubscription(StopProductSubscriptionRequest stopProductSubscriptionRequest)
            throws ServiceQuotaExceededException, ConflictException, ValidationException, ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopProductSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopProductSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopProductSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopProductSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopProductSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<StopProductSubscriptionRequest, StopProductSubscriptionResponse>()
                            .withOperationName("StopProductSubscription").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(stopProductSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopProductSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates additional product configuration settings for the registered identity provider.
     * </p>
     *
     * @param updateIdentityProviderSettingsRequest
     * @return Result of the UpdateIdentityProviderSettings operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws InternalServerException
     *         An exception occurred with the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LicenseManagerUserSubscriptionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LicenseManagerUserSubscriptionsClient.UpdateIdentityProviderSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-user-subscriptions-2018-05-10/UpdateIdentityProviderSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateIdentityProviderSettingsResponse updateIdentityProviderSettings(
            UpdateIdentityProviderSettingsRequest updateIdentityProviderSettingsRequest) throws ValidationException,
            ThrottlingException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            LicenseManagerUserSubscriptionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateIdentityProviderSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateIdentityProviderSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateIdentityProviderSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateIdentityProviderSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager User Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIdentityProviderSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateIdentityProviderSettingsRequest, UpdateIdentityProviderSettingsResponse>()
                            .withOperationName("UpdateIdentityProviderSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateIdentityProviderSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateIdentityProviderSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        LicenseManagerUserSubscriptionsServiceClientConfigurationBuilder serviceConfigBuilder = new LicenseManagerUserSubscriptionsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerUserSubscriptionsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build());
    }

    @Override
    public final LicenseManagerUserSubscriptionsServiceClientConfiguration serviceClientConfiguration() {
        return new LicenseManagerUserSubscriptionsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
