/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.lightsail.transform.DomainEntryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a domain recordset entry.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DomainEntry implements StructuredPojo, ToCopyableBuilder<DomainEntry.Builder, DomainEntry> {
    private final String id;

    private final String name;

    private final String target;

    private final Boolean isAlias;

    private final String type;

    private final Map<String, String> options;

    private DomainEntry(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.target = builder.target;
        this.isAlias = builder.isAlias;
        this.type = builder.type;
        this.options = builder.options;
    }

    /**
     * <p>
     * The ID of the domain recordset entry.
     * </p>
     * 
     * @return The ID of the domain recordset entry.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name of the domain.
     * </p>
     * 
     * @return The name of the domain.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The target AWS name server (e.g., <code>ns-111.awsdns-22.com.</code>).
     * </p>
     * <p>
     * For Lightsail load balancers, the value looks like
     * <code>ab1234c56789c6b86aba6fb203d443bc-123456789.us-east-2.elb.amazonaws.com</code>. Be sure to also set
     * <code>isAlias</code> to <code>true</code> when setting up an A record for a load balancer.
     * </p>
     * 
     * @return The target AWS name server (e.g., <code>ns-111.awsdns-22.com.</code>).</p>
     *         <p>
     *         For Lightsail load balancers, the value looks like
     *         <code>ab1234c56789c6b86aba6fb203d443bc-123456789.us-east-2.elb.amazonaws.com</code>. Be sure to also set
     *         <code>isAlias</code> to <code>true</code> when setting up an A record for a load balancer.
     */
    public String target() {
        return target;
    }

    /**
     * <p>
     * When <code>true</code>, specifies whether the domain entry is an alias used by the Lightsail load balancer. You
     * can include an alias (A type) record in your request, which points to a load balancer DNS name and routes traffic
     * to your load balancer
     * </p>
     * 
     * @return When <code>true</code>, specifies whether the domain entry is an alias used by the Lightsail load
     *         balancer. You can include an alias (A type) record in your request, which points to a load balancer DNS
     *         name and routes traffic to your load balancer
     */
    public Boolean isAlias() {
        return isAlias;
    }

    /**
     * <p>
     * The type of domain entry (e.g., <code>SOA</code> or <code>NS</code>).
     * </p>
     * 
     * @return The type of domain entry (e.g., <code>SOA</code> or <code>NS</code>).
     */
    public String type() {
        return type;
    }

    /**
     * <p>
     * (Deprecated) The options for the domain entry.
     * </p>
     * <note>
     * <p>
     * In releases prior to November 29, 2017, this parameter was not included in the API response. It is now
     * deprecated.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return (Deprecated) The options for the domain entry.</p> <note>
     *         <p>
     *         In releases prior to November 29, 2017, this parameter was not included in the API response. It is now
     *         deprecated.
     *         </p>
     */
    public Map<String, String> options() {
        return options;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(target());
        hashCode = 31 * hashCode + Objects.hashCode(isAlias());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(options());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DomainEntry)) {
            return false;
        }
        DomainEntry other = (DomainEntry) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(target(), other.target()) && Objects.equals(isAlias(), other.isAlias())
                && Objects.equals(type(), other.type()) && Objects.equals(options(), other.options());
    }

    @Override
    public String toString() {
        return ToString.builder("DomainEntry").add("Id", id()).add("Name", name()).add("Target", target())
                .add("IsAlias", isAlias()).add("Type", type()).add("Options", options()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "target":
            return Optional.ofNullable(clazz.cast(target()));
        case "isAlias":
            return Optional.ofNullable(clazz.cast(isAlias()));
        case "type":
            return Optional.ofNullable(clazz.cast(type()));
        case "options":
            return Optional.ofNullable(clazz.cast(options()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DomainEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DomainEntry> {
        /**
         * <p>
         * The ID of the domain recordset entry.
         * </p>
         * 
         * @param id
         *        The ID of the domain recordset entry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the domain.
         * </p>
         * 
         * @param name
         *        The name of the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The target AWS name server (e.g., <code>ns-111.awsdns-22.com.</code>).
         * </p>
         * <p>
         * For Lightsail load balancers, the value looks like
         * <code>ab1234c56789c6b86aba6fb203d443bc-123456789.us-east-2.elb.amazonaws.com</code>. Be sure to also set
         * <code>isAlias</code> to <code>true</code> when setting up an A record for a load balancer.
         * </p>
         * 
         * @param target
         *        The target AWS name server (e.g., <code>ns-111.awsdns-22.com.</code>).</p>
         *        <p>
         *        For Lightsail load balancers, the value looks like
         *        <code>ab1234c56789c6b86aba6fb203d443bc-123456789.us-east-2.elb.amazonaws.com</code>. Be sure to also
         *        set <code>isAlias</code> to <code>true</code> when setting up an A record for a load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder target(String target);

        /**
         * <p>
         * When <code>true</code>, specifies whether the domain entry is an alias used by the Lightsail load balancer.
         * You can include an alias (A type) record in your request, which points to a load balancer DNS name and routes
         * traffic to your load balancer
         * </p>
         * 
         * @param isAlias
         *        When <code>true</code>, specifies whether the domain entry is an alias used by the Lightsail load
         *        balancer. You can include an alias (A type) record in your request, which points to a load balancer
         *        DNS name and routes traffic to your load balancer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isAlias(Boolean isAlias);

        /**
         * <p>
         * The type of domain entry (e.g., <code>SOA</code> or <code>NS</code>).
         * </p>
         * 
         * @param type
         *        The type of domain entry (e.g., <code>SOA</code> or <code>NS</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * (Deprecated) The options for the domain entry.
         * </p>
         * <note>
         * <p>
         * In releases prior to November 29, 2017, this parameter was not included in the API response. It is now
         * deprecated.
         * </p>
         * </note>
         * 
         * @param options
         *        (Deprecated) The options for the domain entry.</p> <note>
         *        <p>
         *        In releases prior to November 29, 2017, this parameter was not included in the API response. It is now
         *        deprecated.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Map<String, String> options);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String target;

        private Boolean isAlias;

        private String type;

        private Map<String, String> options = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DomainEntry model) {
            id(model.id);
            name(model.name);
            target(model.target);
            isAlias(model.isAlias);
            type(model.type);
            options(model.options);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getTarget() {
            return target;
        }

        @Override
        public final Builder target(String target) {
            this.target = target;
            return this;
        }

        public final void setTarget(String target) {
            this.target = target;
        }

        public final Boolean getIsAlias() {
            return isAlias;
        }

        @Override
        public final Builder isAlias(Boolean isAlias) {
            this.isAlias = isAlias;
            return this;
        }

        public final void setIsAlias(Boolean isAlias) {
            this.isAlias = isAlias;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Map<String, String> getOptions() {
            return options;
        }

        @Override
        public final Builder options(Map<String, String> options) {
            this.options = DomainEntryOptionsCopier.copy(options);
            return this;
        }

        public final void setOptions(Map<String, String> options) {
            this.options = DomainEntryOptionsCopier.copy(options);
        }

        @Override
        public DomainEntry build() {
            return new DomainEntry(this);
        }
    }
}
