/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.lightsail.transform.DomainMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a domain where you are storing recordsets in Lightsail.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Domain implements StructuredPojo, ToCopyableBuilder<Domain.Builder, Domain> {
    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<DomainEntry> domainEntries;

    private Domain(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.domainEntries = builder.domainEntries;
    }

    /**
     * <p>
     * The name of the domain.
     * </p>
     * 
     * @return The name of the domain.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the domain recordset (e.g.,
     * <code>arn:aws:lightsail:global:123456789101:Domain/824cede0-abc7-4f84-8dbc-12345EXAMPLE</code>).
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the domain recordset (e.g.,
     *         <code>arn:aws:lightsail:global:123456789101:Domain/824cede0-abc7-4f84-8dbc-12345EXAMPLE</code>).
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about an instance or another
     * resource in Lightsail. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about an instance or
     *         another resource in Lightsail. This code enables our support team to look up your Lightsail information
     *         more easily.
     */
    public String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The date when the domain recordset was created.
     * </p>
     * 
     * @return The date when the domain recordset was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The AWS Region and Availability Zones where the domain recordset was created.
     * </p>
     * 
     * @return The AWS Region and Availability Zones where the domain recordset was created.
     */
    public ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeString}.
     * </p>
     * 
     * @return The resource type.
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeString}.
     * </p>
     * 
     * @return The resource type.
     * @see ResourceType
     */
    public String resourceTypeString() {
        return resourceType;
    }

    /**
     * <p>
     * An array of key-value pairs containing information about the domain entries.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of key-value pairs containing information about the domain entries.
     */
    public List<DomainEntry> domainEntries() {
        return domainEntries;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((arn() == null) ? 0 : arn().hashCode());
        hashCode = 31 * hashCode + ((supportCode() == null) ? 0 : supportCode().hashCode());
        hashCode = 31 * hashCode + ((createdAt() == null) ? 0 : createdAt().hashCode());
        hashCode = 31 * hashCode + ((location() == null) ? 0 : location().hashCode());
        hashCode = 31 * hashCode + ((resourceTypeString() == null) ? 0 : resourceTypeString().hashCode());
        hashCode = 31 * hashCode + ((domainEntries() == null) ? 0 : domainEntries().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Domain)) {
            return false;
        }
        Domain other = (Domain) obj;
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.arn() == null ^ this.arn() == null) {
            return false;
        }
        if (other.arn() != null && !other.arn().equals(this.arn())) {
            return false;
        }
        if (other.supportCode() == null ^ this.supportCode() == null) {
            return false;
        }
        if (other.supportCode() != null && !other.supportCode().equals(this.supportCode())) {
            return false;
        }
        if (other.createdAt() == null ^ this.createdAt() == null) {
            return false;
        }
        if (other.createdAt() != null && !other.createdAt().equals(this.createdAt())) {
            return false;
        }
        if (other.location() == null ^ this.location() == null) {
            return false;
        }
        if (other.location() != null && !other.location().equals(this.location())) {
            return false;
        }
        if (other.resourceTypeString() == null ^ this.resourceTypeString() == null) {
            return false;
        }
        if (other.resourceTypeString() != null && !other.resourceTypeString().equals(this.resourceTypeString())) {
            return false;
        }
        if (other.domainEntries() == null ^ this.domainEntries() == null) {
            return false;
        }
        if (other.domainEntries() != null && !other.domainEntries().equals(this.domainEntries())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (arn() != null) {
            sb.append("Arn: ").append(arn()).append(",");
        }
        if (supportCode() != null) {
            sb.append("SupportCode: ").append(supportCode()).append(",");
        }
        if (createdAt() != null) {
            sb.append("CreatedAt: ").append(createdAt()).append(",");
        }
        if (location() != null) {
            sb.append("Location: ").append(location()).append(",");
        }
        if (resourceTypeString() != null) {
            sb.append("ResourceType: ").append(resourceTypeString()).append(",");
        }
        if (domainEntries() != null) {
            sb.append("DomainEntries: ").append(domainEntries()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.of(clazz.cast(name()));
        case "arn":
            return Optional.of(clazz.cast(arn()));
        case "supportCode":
            return Optional.of(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.of(clazz.cast(createdAt()));
        case "location":
            return Optional.of(clazz.cast(location()));
        case "resourceType":
            return Optional.of(clazz.cast(resourceTypeString()));
        case "domainEntries":
            return Optional.of(clazz.cast(domainEntries()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DomainMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Domain> {
        /**
         * <p>
         * The name of the domain.
         * </p>
         * 
         * @param name
         *        The name of the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the domain recordset (e.g.,
         * <code>arn:aws:lightsail:global:123456789101:Domain/824cede0-abc7-4f84-8dbc-12345EXAMPLE</code>).
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the domain recordset (e.g.,
         *        <code>arn:aws:lightsail:global:123456789101:Domain/824cede0-abc7-4f84-8dbc-12345EXAMPLE</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about an instance or
         * another resource in Lightsail. This code enables our support team to look up your Lightsail information more
         * easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about an instance
         *        or another resource in Lightsail. This code enables our support team to look up your Lightsail
         *        information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The date when the domain recordset was created.
         * </p>
         * 
         * @param createdAt
         *        The date when the domain recordset was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The AWS Region and Availability Zones where the domain recordset was created.
         * </p>
         * 
         * @param location
         *        The AWS Region and Availability Zones where the domain recordset was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The resource type.
         * </p>
         * 
         * @param resourceType
         *        The resource type.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type.
         * </p>
         * 
         * @param resourceType
         *        The resource type.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * An array of key-value pairs containing information about the domain entries.
         * </p>
         * 
         * @param domainEntries
         *        An array of key-value pairs containing information about the domain entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainEntries(Collection<DomainEntry> domainEntries);

        /**
         * <p>
         * An array of key-value pairs containing information about the domain entries.
         * </p>
         * 
         * @param domainEntries
         *        An array of key-value pairs containing information about the domain entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainEntries(DomainEntry... domainEntries);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<DomainEntry> domainEntries;

        private BuilderImpl() {
        }

        private BuilderImpl(Domain model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            domainEntries(model.domainEntries);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<DomainEntry.Builder> getDomainEntries() {
            return domainEntries != null ? domainEntries.stream().map(DomainEntry::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder domainEntries(Collection<DomainEntry> domainEntries) {
            this.domainEntries = DomainEntryListCopier.copy(domainEntries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainEntries(DomainEntry... domainEntries) {
            domainEntries(Arrays.asList(domainEntries));
            return this;
        }

        public final void setDomainEntries(Collection<DomainEntry.BuilderImpl> domainEntries) {
            this.domainEntries = DomainEntryListCopier.copyFromBuilder(domainEntries);
        }

        @Override
        public Domain build() {
            return new Domain(this);
        }
    }
}
