/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes information about the health of the instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceHealthSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceHealthSummary.Builder, InstanceHealthSummary> {
    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceHealthSummary::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<String> INSTANCE_HEALTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceHealthSummary::instanceHealthAsString)).setter(setter(Builder::instanceHealth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceHealth").build()).build();

    private static final SdkField<String> INSTANCE_HEALTH_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceHealthSummary::instanceHealthReasonAsString)).setter(setter(Builder::instanceHealthReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceHealthReason").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_NAME_FIELD,
            INSTANCE_HEALTH_FIELD, INSTANCE_HEALTH_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceName;

    private final String instanceHealth;

    private final String instanceHealthReason;

    private InstanceHealthSummary(BuilderImpl builder) {
        this.instanceName = builder.instanceName;
        this.instanceHealth = builder.instanceHealth;
        this.instanceHealthReason = builder.instanceHealthReason;
    }

    /**
     * <p>
     * The name of the Lightsail instance for which you are requesting health check data.
     * </p>
     * 
     * @return The name of the Lightsail instance for which you are requesting health check data.
     */
    public String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * Describes the overall instance health. Valid values are below.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceHealth}
     * will return {@link InstanceHealthState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceHealthAsString}.
     * </p>
     * 
     * @return Describes the overall instance health. Valid values are below.
     * @see InstanceHealthState
     */
    public InstanceHealthState instanceHealth() {
        return InstanceHealthState.fromValue(instanceHealth);
    }

    /**
     * <p>
     * Describes the overall instance health. Valid values are below.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceHealth}
     * will return {@link InstanceHealthState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceHealthAsString}.
     * </p>
     * 
     * @return Describes the overall instance health. Valid values are below.
     * @see InstanceHealthState
     */
    public String instanceHealthAsString() {
        return instanceHealth;
    }

    /**
     * <p>
     * More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>, then an
     * <code>instanceHealthReason</code> value is not provided.
     * </p>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being registered with
     * the load balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the target instance
     * the minimum number of health checks required to determine its health status.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected HTTP code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to the target
     * instance timed out, the target instance response was malformed, or the target instance failed the health check
     * for an unknown reason.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the target
     * instance is in an Availability Zone that is not enabled for its load balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail load
     * balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of being
     * deregistered and the deregistration delay period has not expired.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceHealthReason} will return {@link InstanceHealthReason#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #instanceHealthReasonAsString}.
     * </p>
     * 
     * @return More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>,
     *         then an <code>instanceHealthReason</code> value is not provided.</p>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b>
     *         <code>instanceHealthReason</code> </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being
     *         registered with the load balancer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the target
     *         instance the minimum number of health checks required to determine its health status.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b>
     *         <code>instanceHealthReason</code> </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected HTTP
     *         code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to
     *         the target instance timed out, the target instance response was malformed, or the target instance failed
     *         the health check for an unknown reason.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b> <code>instanceHealthReason</code>
     *         </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the
     *         target instance is in an Availability Zone that is not enabled for its load balancer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail load
     *         balancer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b>
     *         <code>instanceHealthReason</code> </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of being
     *         deregistered and the deregistration delay period has not expired.
     *         </p>
     *         </li>
     * @see InstanceHealthReason
     */
    public InstanceHealthReason instanceHealthReason() {
        return InstanceHealthReason.fromValue(instanceHealthReason);
    }

    /**
     * <p>
     * More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>, then an
     * <code>instanceHealthReason</code> value is not provided.
     * </p>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being registered with
     * the load balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the target instance
     * the minimum number of health checks required to determine its health status.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected HTTP code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to the target
     * instance timed out, the target instance response was malformed, or the target instance failed the health check
     * for an unknown reason.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the target
     * instance is in an Availability Zone that is not enabled for its load balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail load
     * balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b> <code>instanceHealthReason</code> </b>
     * value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of being
     * deregistered and the deregistration delay period has not expired.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceHealthReason} will return {@link InstanceHealthReason#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #instanceHealthReasonAsString}.
     * </p>
     * 
     * @return More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>,
     *         then an <code>instanceHealthReason</code> value is not provided.</p>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b>
     *         <code>instanceHealthReason</code> </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being
     *         registered with the load balancer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the target
     *         instance the minimum number of health checks required to determine its health status.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b>
     *         <code>instanceHealthReason</code> </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected HTTP
     *         code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to
     *         the target instance timed out, the target instance response was malformed, or the target instance failed
     *         the health check for an unknown reason.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b> <code>instanceHealthReason</code>
     *         </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the
     *         target instance is in an Availability Zone that is not enabled for its load balancer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail load
     *         balancer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b>
     *         <code>instanceHealthReason</code> </b> value can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of being
     *         deregistered and the deregistration delay period has not expired.
     *         </p>
     *         </li>
     * @see InstanceHealthReason
     */
    public String instanceHealthReasonAsString() {
        return instanceHealthReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceHealthAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceHealthReasonAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceHealthSummary)) {
            return false;
        }
        InstanceHealthSummary other = (InstanceHealthSummary) obj;
        return Objects.equals(instanceName(), other.instanceName())
                && Objects.equals(instanceHealthAsString(), other.instanceHealthAsString())
                && Objects.equals(instanceHealthReasonAsString(), other.instanceHealthReasonAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceHealthSummary").add("InstanceName", instanceName())
                .add("InstanceHealth", instanceHealthAsString()).add("InstanceHealthReason", instanceHealthReasonAsString())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "instanceHealth":
            return Optional.ofNullable(clazz.cast(instanceHealthAsString()));
        case "instanceHealthReason":
            return Optional.ofNullable(clazz.cast(instanceHealthReasonAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceHealthSummary, T> g) {
        return obj -> g.apply((InstanceHealthSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceHealthSummary> {
        /**
         * <p>
         * The name of the Lightsail instance for which you are requesting health check data.
         * </p>
         * 
         * @param instanceName
         *        The name of the Lightsail instance for which you are requesting health check data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * Describes the overall instance health. Valid values are below.
         * </p>
         * 
         * @param instanceHealth
         *        Describes the overall instance health. Valid values are below.
         * @see InstanceHealthState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceHealthState
         */
        Builder instanceHealth(String instanceHealth);

        /**
         * <p>
         * Describes the overall instance health. Valid values are below.
         * </p>
         * 
         * @param instanceHealth
         *        Describes the overall instance health. Valid values are below.
         * @see InstanceHealthState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceHealthState
         */
        Builder instanceHealth(InstanceHealthState instanceHealth);

        /**
         * <p>
         * More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>, then
         * an <code>instanceHealthReason</code> value is not provided.
         * </p>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being registered
         * with the load balancer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the target
         * instance the minimum number of health checks required to determine its health status.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected HTTP code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to the
         * target instance timed out, the target instance response was malformed, or the target instance failed the
         * health check for an unknown reason.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the target
         * instance is in an Availability Zone that is not enabled for its load balancer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail load
         * balancer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of being
         * deregistered and the deregistration delay period has not expired.
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceHealthReason
         *        More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>
         *        , then an <code>instanceHealthReason</code> value is not provided.</p>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being
         *        registered with the load balancer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the
         *        target instance the minimum number of health checks required to determine its health status.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected
         *        HTTP code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to
         *        the target instance timed out, the target instance response was malformed, or the target instance
         *        failed the health check for an unknown reason.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the
         *        target instance is in an Availability Zone that is not enabled for its load balancer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail
         *        load balancer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of
         *        being deregistered and the deregistration delay period has not expired.
         *        </p>
         *        </li>
         * @see InstanceHealthReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceHealthReason
         */
        Builder instanceHealthReason(String instanceHealthReason);

        /**
         * <p>
         * More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>, then
         * an <code>instanceHealthReason</code> value is not provided.
         * </p>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being registered
         * with the load balancer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the target
         * instance the minimum number of health checks required to determine its health status.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected HTTP code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to the
         * target instance timed out, the target instance response was malformed, or the target instance failed the
         * health check for an unknown reason.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the target
         * instance is in an Availability Zone that is not enabled for its load balancer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail load
         * balancer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b> <code>instanceHealthReason</code>
         * </b> value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of being
         * deregistered and the deregistration delay period has not expired.
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceHealthReason
         *        More information about the instance health. If the <code>instanceHealth</code> is <code>healthy</code>
         *        , then an <code>instanceHealthReason</code> value is not provided.</p>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>initial</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Lb.RegistrationInProgress</code> </b> - The target instance is in the process of being
         *        registered with the load balancer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Lb.InitialHealthChecking</code> </b> - The Lightsail load balancer is still sending the
         *        target instance the minimum number of health checks required to determine its health status.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>unhealthy</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Instance.ResponseCodeMismatch</code> </b> - The health checks did not return an expected
         *        HTTP code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.Timeout</code> </b> - The health check requests timed out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.FailedHealthChecks</code> </b> - The health checks failed because the connection to
         *        the target instance timed out, the target instance response was malformed, or the target instance
         *        failed the health check for an unknown reason.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Lb.InternalError</code> </b> - The health checks failed due to an internal error.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>unused</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Instance.NotRegistered</code> </b> - The target instance is not registered with the target
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.NotInUse</code> </b> - The target group is not used by any load balancer, or the
         *        target instance is in an Availability Zone that is not enabled for its load balancer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.IpUnusable</code> </b> - The target IP address is reserved for use by a Lightsail
         *        load balancer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Instance.InvalidState</code> </b> - The target is in the stopped or terminated state.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <b> <code>instanceHealth</code> </b> is <code>draining</code>, the <b>
         *        <code>instanceHealthReason</code> </b> value can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Instance.DeregistrationInProgress</code> </b> - The target instance is in the process of
         *        being deregistered and the deregistration delay period has not expired.
         *        </p>
         *        </li>
         * @see InstanceHealthReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceHealthReason
         */
        Builder instanceHealthReason(InstanceHealthReason instanceHealthReason);
    }

    static final class BuilderImpl implements Builder {
        private String instanceName;

        private String instanceHealth;

        private String instanceHealthReason;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceHealthSummary model) {
            instanceName(model.instanceName);
            instanceHealth(model.instanceHealth);
            instanceHealthReason(model.instanceHealthReason);
        }

        public final String getInstanceName() {
            return instanceName;
        }

        @Override
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        public final String getInstanceHealthAsString() {
            return instanceHealth;
        }

        @Override
        public final Builder instanceHealth(String instanceHealth) {
            this.instanceHealth = instanceHealth;
            return this;
        }

        @Override
        public final Builder instanceHealth(InstanceHealthState instanceHealth) {
            this.instanceHealth(instanceHealth == null ? null : instanceHealth.toString());
            return this;
        }

        public final void setInstanceHealth(String instanceHealth) {
            this.instanceHealth = instanceHealth;
        }

        public final String getInstanceHealthReasonAsString() {
            return instanceHealthReason;
        }

        @Override
        public final Builder instanceHealthReason(String instanceHealthReason) {
            this.instanceHealthReason = instanceHealthReason;
            return this;
        }

        @Override
        public final Builder instanceHealthReason(InstanceHealthReason instanceHealthReason) {
            this.instanceHealthReason(instanceHealthReason == null ? null : instanceHealthReason.toString());
            return this;
        }

        public final void setInstanceHealthReason(String instanceHealthReason) {
            this.instanceHealthReason = instanceHealthReason;
        }

        @Override
        public InstanceHealthSummary build() {
            return new InstanceHealthSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
