/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a system disk or a block storage disk.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Disk implements SdkPojo, Serializable, ToCopyableBuilder<Disk.Builder, Disk> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).getter(getter(Disk::arn))
            .setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Disk::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).getter(getter(Disk::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(Disk::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AddOn>> ADD_ONS_FIELD = SdkField
            .<List<AddOn>> builder(MarshallingType.LIST)
            .getter(getter(Disk::addOns))
            .setter(setter(Builder::addOns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("addOns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AddOn> builder(MarshallingType.SDK_POJO)
                                            .constructor(AddOn::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Disk::sizeInGb)).setter(setter(Builder::sizeInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInGb").build()).build();

    private static final SdkField<Boolean> IS_SYSTEM_DISK_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Disk::isSystemDisk)).setter(setter(Builder::isSystemDisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isSystemDisk").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Disk::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("iops").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("path").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> ATTACHED_TO_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::attachedTo)).setter(setter(Builder::attachedTo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attachedTo").build()).build();

    private static final SdkField<Boolean> IS_ATTACHED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Disk::isAttached)).setter(setter(Builder::isAttached))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isAttached").build()).build();

    private static final SdkField<String> ATTACHMENT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Disk::attachmentState)).setter(setter(Builder::attachmentState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attachmentState").build()).build();

    private static final SdkField<Integer> GB_IN_USE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Disk::gbInUse)).setter(setter(Builder::gbInUse))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gbInUse").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD, ADD_ONS_FIELD,
            SIZE_IN_GB_FIELD, IS_SYSTEM_DISK_FIELD, IOPS_FIELD, PATH_FIELD, STATE_FIELD, ATTACHED_TO_FIELD, IS_ATTACHED_FIELD,
            ATTACHMENT_STATE_FIELD, GB_IN_USE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final List<AddOn> addOns;

    private final Integer sizeInGb;

    private final Boolean isSystemDisk;

    private final Integer iops;

    private final String path;

    private final String state;

    private final String attachedTo;

    private final Boolean isAttached;

    private final String attachmentState;

    private final Integer gbInUse;

    private Disk(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.addOns = builder.addOns;
        this.sizeInGb = builder.sizeInGb;
        this.isSystemDisk = builder.isSystemDisk;
        this.iops = builder.iops;
        this.path = builder.path;
        this.state = builder.state;
        this.attachedTo = builder.attachedTo;
        this.isAttached = builder.isAttached;
        this.attachmentState = builder.attachmentState;
        this.gbInUse = builder.gbInUse;
    }

    /**
     * <p>
     * The unique name of the disk.
     * </p>
     * 
     * @return The unique name of the disk.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the disk.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the disk.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about an instance or another
     * resource in Lightsail. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about an instance or
     *         another resource in Lightsail. This code enables our support team to look up your Lightsail information
     *         more easily.
     */
    public String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The date when the disk was created.
     * </p>
     * 
     * @return The date when the disk was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The AWS Region and Availability Zone where the disk is located.
     * </p>
     * 
     * @return The AWS Region and Availability Zone where the disk is located.
     */
    public ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The Lightsail resource type (e.g., <code>Disk</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type (e.g., <code>Disk</code>).
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The Lightsail resource type (e.g., <code>Disk</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type (e.g., <code>Disk</code>).
     * @see ResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
     *         Guide</a>.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * Returns true if the AddOns property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAddOns() {
        return addOns != null && !(addOns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects representing the add-ons enabled on the disk.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAddOns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects representing the add-ons enabled on the disk.
     */
    public List<AddOn> addOns() {
        return addOns;
    }

    /**
     * <p>
     * The size of the disk in GB.
     * </p>
     * 
     * @return The size of the disk in GB.
     */
    public Integer sizeInGb() {
        return sizeInGb;
    }

    /**
     * <p>
     * A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
     * </p>
     * 
     * @return A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
     */
    public Boolean isSystemDisk() {
        return isSystemDisk;
    }

    /**
     * <p>
     * The input/output operations per second (IOPS) of the disk.
     * </p>
     * 
     * @return The input/output operations per second (IOPS) of the disk.
     */
    public Integer iops() {
        return iops;
    }

    /**
     * <p>
     * The disk path.
     * </p>
     * 
     * @return The disk path.
     */
    public String path() {
        return path;
    }

    /**
     * <p>
     * Describes the status of the disk.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link DiskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Describes the status of the disk.
     * @see DiskState
     */
    public DiskState state() {
        return DiskState.fromValue(state);
    }

    /**
     * <p>
     * Describes the status of the disk.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link DiskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Describes the status of the disk.
     * @see DiskState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The resources to which the disk is attached.
     * </p>
     * 
     * @return The resources to which the disk is attached.
     */
    public String attachedTo() {
        return attachedTo;
    }

    /**
     * <p>
     * A Boolean value indicating whether the disk is attached.
     * </p>
     * 
     * @return A Boolean value indicating whether the disk is attached.
     */
    public Boolean isAttached() {
        return isAttached;
    }

    /**
     * <p>
     * (Deprecated) The attachment state of the disk.
     * </p>
     * <note>
     * <p>
     * In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks in the API
     * response. It is now deprecated, but still included in the response. Use <code>isAttached</code> instead.
     * </p>
     * </note>
     * 
     * @return (Deprecated) The attachment state of the disk.</p> <note>
     *         <p>
     *         In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks in
     *         the API response. It is now deprecated, but still included in the response. Use <code>isAttached</code>
     *         instead.
     *         </p>
     */
    public String attachmentState() {
        return attachmentState;
    }

    /**
     * <p>
     * (Deprecated) The number of GB in use by the disk.
     * </p>
     * <note>
     * <p>
     * In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
     * deprecated.
     * </p>
     * </note>
     * 
     * @return (Deprecated) The number of GB in use by the disk.</p> <note>
     *         <p>
     *         In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
     *         deprecated.
     *         </p>
     */
    public Integer gbInUse() {
        return gbInUse;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(addOns());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInGb());
        hashCode = 31 * hashCode + Objects.hashCode(isSystemDisk());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attachedTo());
        hashCode = 31 * hashCode + Objects.hashCode(isAttached());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentState());
        hashCode = 31 * hashCode + Objects.hashCode(gbInUse());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Disk)) {
            return false;
        }
        Disk other = (Disk) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && Objects.equals(tags(), other.tags())
                && Objects.equals(addOns(), other.addOns()) && Objects.equals(sizeInGb(), other.sizeInGb())
                && Objects.equals(isSystemDisk(), other.isSystemDisk()) && Objects.equals(iops(), other.iops())
                && Objects.equals(path(), other.path()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(attachedTo(), other.attachedTo()) && Objects.equals(isAttached(), other.isAttached())
                && Objects.equals(attachmentState(), other.attachmentState()) && Objects.equals(gbInUse(), other.gbInUse());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Disk").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", tags()).add("AddOns", addOns()).add("SizeInGb", sizeInGb()).add("IsSystemDisk", isSystemDisk())
                .add("Iops", iops()).add("Path", path()).add("State", stateAsString()).add("AttachedTo", attachedTo())
                .add("IsAttached", isAttached()).add("AttachmentState", attachmentState()).add("GbInUse", gbInUse()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "addOns":
            return Optional.ofNullable(clazz.cast(addOns()));
        case "sizeInGb":
            return Optional.ofNullable(clazz.cast(sizeInGb()));
        case "isSystemDisk":
            return Optional.ofNullable(clazz.cast(isSystemDisk()));
        case "iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "path":
            return Optional.ofNullable(clazz.cast(path()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "attachedTo":
            return Optional.ofNullable(clazz.cast(attachedTo()));
        case "isAttached":
            return Optional.ofNullable(clazz.cast(isAttached()));
        case "attachmentState":
            return Optional.ofNullable(clazz.cast(attachmentState()));
        case "gbInUse":
            return Optional.ofNullable(clazz.cast(gbInUse()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Disk, T> g) {
        return obj -> g.apply((Disk) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Disk> {
        /**
         * <p>
         * The unique name of the disk.
         * </p>
         * 
         * @param name
         *        The unique name of the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the disk.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about an instance or
         * another resource in Lightsail. This code enables our support team to look up your Lightsail information more
         * easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about an instance
         *        or another resource in Lightsail. This code enables our support team to look up your Lightsail
         *        information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The date when the disk was created.
         * </p>
         * 
         * @param createdAt
         *        The date when the disk was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The AWS Region and Availability Zone where the disk is located.
         * </p>
         * 
         * @param location
         *        The AWS Region and Availability Zone where the disk is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The AWS Region and Availability Zone where the disk is located.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The Lightsail resource type (e.g., <code>Disk</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type (e.g., <code>Disk</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Lightsail resource type (e.g., <code>Disk</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type (e.g., <code>Disk</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * An array of objects representing the add-ons enabled on the disk.
         * </p>
         * 
         * @param addOns
         *        An array of objects representing the add-ons enabled on the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addOns(Collection<AddOn> addOns);

        /**
         * <p>
         * An array of objects representing the add-ons enabled on the disk.
         * </p>
         * 
         * @param addOns
         *        An array of objects representing the add-ons enabled on the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addOns(AddOn... addOns);

        /**
         * <p>
         * An array of objects representing the add-ons enabled on the disk.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AddOn>.Builder} avoiding the need to create
         * one manually via {@link List<AddOn>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AddOn>.Builder#build()} is called immediately and its result
         * is passed to {@link #addOns(List<AddOn>)}.
         * 
         * @param addOns
         *        a consumer that will call methods on {@link List<AddOn>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #addOns(List<AddOn>)
         */
        Builder addOns(Consumer<AddOn.Builder>... addOns);

        /**
         * <p>
         * The size of the disk in GB.
         * </p>
         * 
         * @param sizeInGb
         *        The size of the disk in GB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInGb(Integer sizeInGb);

        /**
         * <p>
         * A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
         * </p>
         * 
         * @param isSystemDisk
         *        A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isSystemDisk(Boolean isSystemDisk);

        /**
         * <p>
         * The input/output operations per second (IOPS) of the disk.
         * </p>
         * 
         * @param iops
         *        The input/output operations per second (IOPS) of the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * The disk path.
         * </p>
         * 
         * @param path
         *        The disk path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Describes the status of the disk.
         * </p>
         * 
         * @param state
         *        Describes the status of the disk.
         * @see DiskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiskState
         */
        Builder state(String state);

        /**
         * <p>
         * Describes the status of the disk.
         * </p>
         * 
         * @param state
         *        Describes the status of the disk.
         * @see DiskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiskState
         */
        Builder state(DiskState state);

        /**
         * <p>
         * The resources to which the disk is attached.
         * </p>
         * 
         * @param attachedTo
         *        The resources to which the disk is attached.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachedTo(String attachedTo);

        /**
         * <p>
         * A Boolean value indicating whether the disk is attached.
         * </p>
         * 
         * @param isAttached
         *        A Boolean value indicating whether the disk is attached.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isAttached(Boolean isAttached);

        /**
         * <p>
         * (Deprecated) The attachment state of the disk.
         * </p>
         * <note>
         * <p>
         * In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks in the
         * API response. It is now deprecated, but still included in the response. Use <code>isAttached</code> instead.
         * </p>
         * </note>
         * 
         * @param attachmentState
         *        (Deprecated) The attachment state of the disk.</p> <note>
         *        <p>
         *        In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks
         *        in the API response. It is now deprecated, but still included in the response. Use
         *        <code>isAttached</code> instead.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentState(String attachmentState);

        /**
         * <p>
         * (Deprecated) The number of GB in use by the disk.
         * </p>
         * <note>
         * <p>
         * In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
         * deprecated.
         * </p>
         * </note>
         * 
         * @param gbInUse
         *        (Deprecated) The number of GB in use by the disk.</p> <note>
         *        <p>
         *        In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
         *        deprecated.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gbInUse(Integer gbInUse);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<AddOn> addOns = DefaultSdkAutoConstructList.getInstance();

        private Integer sizeInGb;

        private Boolean isSystemDisk;

        private Integer iops;

        private String path;

        private String state;

        private String attachedTo;

        private Boolean isAttached;

        private String attachmentState;

        private Integer gbInUse;

        private BuilderImpl() {
        }

        private BuilderImpl(Disk model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            addOns(model.addOns);
            sizeInGb(model.sizeInGb);
            isSystemDisk(model.isSystemDisk);
            iops(model.iops);
            path(model.path);
            state(model.state);
            attachedTo(model.attachedTo);
            isAttached(model.isAttached);
            attachmentState(model.attachmentState);
            gbInUse(model.gbInUse);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceTypeAsString() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final Collection<AddOn.Builder> getAddOns() {
            return addOns != null ? addOns.stream().map(AddOn::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder addOns(Collection<AddOn> addOns) {
            this.addOns = AddOnListCopier.copy(addOns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder addOns(AddOn... addOns) {
            addOns(Arrays.asList(addOns));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder addOns(Consumer<AddOn.Builder>... addOns) {
            addOns(Stream.of(addOns).map(c -> AddOn.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAddOns(Collection<AddOn.BuilderImpl> addOns) {
            this.addOns = AddOnListCopier.copyFromBuilder(addOns);
        }

        public final Integer getSizeInGb() {
            return sizeInGb;
        }

        @Override
        public final Builder sizeInGb(Integer sizeInGb) {
            this.sizeInGb = sizeInGb;
            return this;
        }

        public final void setSizeInGb(Integer sizeInGb) {
            this.sizeInGb = sizeInGb;
        }

        public final Boolean getIsSystemDisk() {
            return isSystemDisk;
        }

        @Override
        public final Builder isSystemDisk(Boolean isSystemDisk) {
            this.isSystemDisk = isSystemDisk;
            return this;
        }

        public final void setIsSystemDisk(Boolean isSystemDisk) {
            this.isSystemDisk = isSystemDisk;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(DiskState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getAttachedTo() {
            return attachedTo;
        }

        @Override
        public final Builder attachedTo(String attachedTo) {
            this.attachedTo = attachedTo;
            return this;
        }

        public final void setAttachedTo(String attachedTo) {
            this.attachedTo = attachedTo;
        }

        public final Boolean getIsAttached() {
            return isAttached;
        }

        @Override
        public final Builder isAttached(Boolean isAttached) {
            this.isAttached = isAttached;
            return this;
        }

        public final void setIsAttached(Boolean isAttached) {
            this.isAttached = isAttached;
        }

        public final String getAttachmentState() {
            return attachmentState;
        }

        @Override
        public final Builder attachmentState(String attachmentState) {
            this.attachmentState = attachmentState;
            return this;
        }

        public final void setAttachmentState(String attachmentState) {
            this.attachmentState = attachmentState;
        }

        public final Integer getGbInUse() {
            return gbInUse;
        }

        @Override
        public final Builder gbInUse(Integer gbInUse) {
            this.gbInUse = gbInUse;
            return this;
        }

        public final void setGbInUse(Integer gbInUse) {
            this.gbInUse = gbInUse;
        }

        @Override
        public Disk build() {
            return new Disk(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
