/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the SSH key pair.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeyPair implements SdkPojo, Serializable, ToCopyableBuilder<KeyPair.Builder, KeyPair> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeyPair::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeyPair::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeyPair::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(KeyPair::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).getter(getter(KeyPair::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeyPair::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(KeyPair::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FINGERPRINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeyPair::fingerprint)).setter(setter(Builder::fingerprint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fingerprint").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD, FINGERPRINT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final String fingerprint;

    private KeyPair(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.fingerprint = builder.fingerprint;
    }

    /**
     * <p>
     * The friendly name of the SSH key pair.
     * </p>
     * 
     * @return The friendly name of the SSH key pair.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the key pair (e.g.,
     * <code>arn:aws:lightsail:us-east-2:123456789101:KeyPair/05859e3d-331d-48ba-9034-12345EXAMPLE</code>).
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the key pair (e.g.,
     *         <code>arn:aws:lightsail:us-east-2:123456789101:KeyPair/05859e3d-331d-48ba-9034-12345EXAMPLE</code>).
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about an instance or another
     * resource in Lightsail. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about an instance or
     *         another resource in Lightsail. This code enables our support team to look up your Lightsail information
     *         more easily.
     */
    public String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The timestamp when the key pair was created (e.g., <code>1479816991.349</code>).
     * </p>
     * 
     * @return The timestamp when the key pair was created (e.g., <code>1479816991.349</code>).
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The region name and Availability Zone where the key pair was created.
     * </p>
     * 
     * @return The region name and Availability Zone where the key pair was created.
     */
    public ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The resource type (usually <code>KeyPair</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type (usually <code>KeyPair</code>).
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type (usually <code>KeyPair</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type (usually <code>KeyPair</code>).
     * @see ResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
     *         Guide</a>.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The RSA fingerprint of the key pair.
     * </p>
     * 
     * @return The RSA fingerprint of the key pair.
     */
    public String fingerprint() {
        return fingerprint;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(fingerprint());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeyPair)) {
            return false;
        }
        KeyPair other = (KeyPair) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && Objects.equals(tags(), other.tags())
                && Objects.equals(fingerprint(), other.fingerprint());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("KeyPair").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", tags()).add("Fingerprint", fingerprint()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "fingerprint":
            return Optional.ofNullable(clazz.cast(fingerprint()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KeyPair, T> g) {
        return obj -> g.apply((KeyPair) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeyPair> {
        /**
         * <p>
         * The friendly name of the SSH key pair.
         * </p>
         * 
         * @param name
         *        The friendly name of the SSH key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the key pair (e.g.,
         * <code>arn:aws:lightsail:us-east-2:123456789101:KeyPair/05859e3d-331d-48ba-9034-12345EXAMPLE</code>).
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the key pair (e.g.,
         *        <code>arn:aws:lightsail:us-east-2:123456789101:KeyPair/05859e3d-331d-48ba-9034-12345EXAMPLE</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about an instance or
         * another resource in Lightsail. This code enables our support team to look up your Lightsail information more
         * easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about an instance
         *        or another resource in Lightsail. This code enables our support team to look up your Lightsail
         *        information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The timestamp when the key pair was created (e.g., <code>1479816991.349</code>).
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the key pair was created (e.g., <code>1479816991.349</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The region name and Availability Zone where the key pair was created.
         * </p>
         * 
         * @param location
         *        The region name and Availability Zone where the key pair was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The region name and Availability Zone where the key pair was created.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The resource type (usually <code>KeyPair</code>).
         * </p>
         * 
         * @param resourceType
         *        The resource type (usually <code>KeyPair</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type (usually <code>KeyPair</code>).
         * </p>
         * 
         * @param resourceType
         *        The resource type (usually <code>KeyPair</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The RSA fingerprint of the key pair.
         * </p>
         * 
         * @param fingerprint
         *        The RSA fingerprint of the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fingerprint(String fingerprint);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String fingerprint;

        private BuilderImpl() {
        }

        private BuilderImpl(KeyPair model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            fingerprint(model.fingerprint);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getFingerprint() {
            return fingerprint;
        }

        @Override
        public final Builder fingerprint(String fingerprint) {
            this.fingerprint = fingerprint;
            return this;
        }

        public final void setFingerprint(String fingerprint) {
            this.fingerprint = fingerprint;
        }

        @Override
        public KeyPair build() {
            return new KeyPair(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
