/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the origin resource of an Amazon Lightsail content delivery network (CDN) distribution.
 * </p>
 * <p>
 * An origin can be a Lightsail instance or load balancer. A distribution pulls content from an origin, caches it, and
 * serves it to viewers via a worldwide network of edge servers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Origin implements SdkPojo, Serializable, ToCopyableBuilder<Origin.Builder, Origin> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Origin::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(Origin::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<String> REGION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("regionName").getter(getter(Origin::regionNameAsString)).setter(setter(Builder::regionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionName").build()).build();

    private static final SdkField<String> PROTOCOL_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocolPolicy").getter(getter(Origin::protocolPolicyAsString)).setter(setter(Builder::protocolPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocolPolicy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            RESOURCE_TYPE_FIELD, REGION_NAME_FIELD, PROTOCOL_POLICY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String resourceType;

    private final String regionName;

    private final String protocolPolicy;

    private Origin(BuilderImpl builder) {
        this.name = builder.name;
        this.resourceType = builder.resourceType;
        this.regionName = builder.regionName;
        this.protocolPolicy = builder.protocolPolicy;
    }

    /**
     * <p>
     * The name of the origin resource.
     * </p>
     * 
     * @return The name of the origin resource.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The resource type of the origin resource (e.g., <i>Instance</i>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type of the origin resource (e.g., <i>Instance</i>).
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type of the origin resource (e.g., <i>Instance</i>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type of the origin resource (e.g., <i>Instance</i>).
     * @see ResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The AWS Region name of the origin resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #regionName} will
     * return {@link RegionName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionNameAsString}.
     * </p>
     * 
     * @return The AWS Region name of the origin resource.
     * @see RegionName
     */
    public RegionName regionName() {
        return RegionName.fromValue(regionName);
    }

    /**
     * <p>
     * The AWS Region name of the origin resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #regionName} will
     * return {@link RegionName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionNameAsString}.
     * </p>
     * 
     * @return The AWS Region name of the origin resource.
     * @see RegionName
     */
    public String regionNameAsString() {
        return regionName;
    }

    /**
     * <p>
     * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to pull
     * content.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocolPolicy}
     * will return {@link OriginProtocolPolicyEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #protocolPolicyAsString}.
     * </p>
     * 
     * @return The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin
     *         to pull content.
     * @see OriginProtocolPolicyEnum
     */
    public OriginProtocolPolicyEnum protocolPolicy() {
        return OriginProtocolPolicyEnum.fromValue(protocolPolicy);
    }

    /**
     * <p>
     * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to pull
     * content.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocolPolicy}
     * will return {@link OriginProtocolPolicyEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #protocolPolicyAsString}.
     * </p>
     * 
     * @return The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin
     *         to pull content.
     * @see OriginProtocolPolicyEnum
     */
    public String protocolPolicyAsString() {
        return protocolPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(regionNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(protocolPolicyAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Origin)) {
            return false;
        }
        Origin other = (Origin) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(regionNameAsString(), other.regionNameAsString())
                && Objects.equals(protocolPolicyAsString(), other.protocolPolicyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Origin").add("Name", name()).add("ResourceType", resourceTypeAsString())
                .add("RegionName", regionNameAsString()).add("ProtocolPolicy", protocolPolicyAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "regionName":
            return Optional.ofNullable(clazz.cast(regionNameAsString()));
        case "protocolPolicy":
            return Optional.ofNullable(clazz.cast(protocolPolicyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Origin, T> g) {
        return obj -> g.apply((Origin) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Origin> {
        /**
         * <p>
         * The name of the origin resource.
         * </p>
         * 
         * @param name
         *        The name of the origin resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The resource type of the origin resource (e.g., <i>Instance</i>).
         * </p>
         * 
         * @param resourceType
         *        The resource type of the origin resource (e.g., <i>Instance</i>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type of the origin resource (e.g., <i>Instance</i>).
         * </p>
         * 
         * @param resourceType
         *        The resource type of the origin resource (e.g., <i>Instance</i>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The AWS Region name of the origin resource.
         * </p>
         * 
         * @param regionName
         *        The AWS Region name of the origin resource.
         * @see RegionName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionName
         */
        Builder regionName(String regionName);

        /**
         * <p>
         * The AWS Region name of the origin resource.
         * </p>
         * 
         * @param regionName
         *        The AWS Region name of the origin resource.
         * @see RegionName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionName
         */
        Builder regionName(RegionName regionName);

        /**
         * <p>
         * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to
         * pull content.
         * </p>
         * 
         * @param protocolPolicy
         *        The protocol that your Amazon Lightsail distribution uses when establishing a connection with your
         *        origin to pull content.
         * @see OriginProtocolPolicyEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginProtocolPolicyEnum
         */
        Builder protocolPolicy(String protocolPolicy);

        /**
         * <p>
         * The protocol that your Amazon Lightsail distribution uses when establishing a connection with your origin to
         * pull content.
         * </p>
         * 
         * @param protocolPolicy
         *        The protocol that your Amazon Lightsail distribution uses when establishing a connection with your
         *        origin to pull content.
         * @see OriginProtocolPolicyEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginProtocolPolicyEnum
         */
        Builder protocolPolicy(OriginProtocolPolicyEnum protocolPolicy);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String resourceType;

        private String regionName;

        private String protocolPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(Origin model) {
            name(model.name);
            resourceType(model.resourceType);
            regionName(model.regionName);
            protocolPolicy(model.protocolPolicy);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final String getRegionName() {
            return regionName;
        }

        @Override
        public final Builder regionName(String regionName) {
            this.regionName = regionName;
            return this;
        }

        @Override
        public final Builder regionName(RegionName regionName) {
            this.regionName(regionName == null ? null : regionName.toString());
            return this;
        }

        public final void setRegionName(String regionName) {
            this.regionName = regionName;
        }

        public final String getProtocolPolicy() {
            return protocolPolicy;
        }

        @Override
        public final Builder protocolPolicy(String protocolPolicy) {
            this.protocolPolicy = protocolPolicy;
            return this;
        }

        @Override
        public final Builder protocolPolicy(OriginProtocolPolicyEnum protocolPolicy) {
            this.protocolPolicy(protocolPolicy == null ? null : protocolPolicy.toString());
            return this;
        }

        public final void setProtocolPolicy(String protocolPolicy) {
            this.protocolPolicy = protocolPolicy;
        }

        @Override
        public Origin build() {
            return new Origin(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
