/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a container deployment configuration of an Amazon Lightsail container service.
 * </p>
 * <p>
 * A deployment specifies the settings, such as the ports and launch command, of containers that are deployed to your
 * container service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerServiceDeployment implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerServiceDeployment.Builder, ContainerServiceDeployment> {
    private static final SdkField<Integer> VERSION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("version").getter(getter(ContainerServiceDeployment::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(ContainerServiceDeployment::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<Map<String, Container>> CONTAINERS_FIELD = SdkField
            .<Map<String, Container>> builder(MarshallingType.MAP)
            .memberName("containers")
            .getter(getter(ContainerServiceDeployment::containers))
            .setter(setter(Builder::containers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containers").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Container> builder(MarshallingType.SDK_POJO)
                                            .constructor(Container::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ContainerServiceEndpoint> PUBLIC_ENDPOINT_FIELD = SdkField
            .<ContainerServiceEndpoint> builder(MarshallingType.SDK_POJO).memberName("publicEndpoint")
            .getter(getter(ContainerServiceDeployment::publicEndpoint)).setter(setter(Builder::publicEndpoint))
            .constructor(ContainerServiceEndpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicEndpoint").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(ContainerServiceDeployment::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD, STATE_FIELD,
            CONTAINERS_FIELD, PUBLIC_ENDPOINT_FIELD, CREATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer version;

    private final String state;

    private final Map<String, Container> containers;

    private final ContainerServiceEndpoint publicEndpoint;

    private final Instant createdAt;

    private ContainerServiceDeployment(BuilderImpl builder) {
        this.version = builder.version;
        this.state = builder.state;
        this.containers = builder.containers;
        this.publicEndpoint = builder.publicEndpoint;
        this.createdAt = builder.createdAt;
    }

    /**
     * <p>
     * The version number of the deployment.
     * </p>
     * 
     * @return The version number of the deployment.
     */
    public Integer version() {
        return version;
    }

    /**
     * <p>
     * The state of the deployment.
     * </p>
     * <p>
     * A deployment can be in one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Activating</code> - The deployment is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Active</code> - The deployment was successfully created, and it's currently running on the container
     * service. The container service can have only one deployment in an active state at a time.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Inactive</code> - The deployment was previously successfully created, but it is not currently running on
     * the container service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the log events
     * for the containers in the deployment to try to determine the reason for the failure.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ContainerServiceDeploymentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the deployment.</p>
     *         <p>
     *         A deployment can be in one of the following states:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Activating</code> - The deployment is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Active</code> - The deployment was successfully created, and it's currently running on the
     *         container service. The container service can have only one deployment in an active state at a time.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Inactive</code> - The deployment was previously successfully created, but it is not currently
     *         running on the container service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the log
     *         events for the containers in the deployment to try to determine the reason for the failure.
     *         </p>
     *         </li>
     * @see ContainerServiceDeploymentState
     */
    public ContainerServiceDeploymentState state() {
        return ContainerServiceDeploymentState.fromValue(state);
    }

    /**
     * <p>
     * The state of the deployment.
     * </p>
     * <p>
     * A deployment can be in one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Activating</code> - The deployment is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Active</code> - The deployment was successfully created, and it's currently running on the container
     * service. The container service can have only one deployment in an active state at a time.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Inactive</code> - The deployment was previously successfully created, but it is not currently running on
     * the container service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the log events
     * for the containers in the deployment to try to determine the reason for the failure.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ContainerServiceDeploymentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the deployment.</p>
     *         <p>
     *         A deployment can be in one of the following states:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Activating</code> - The deployment is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Active</code> - The deployment was successfully created, and it's currently running on the
     *         container service. The container service can have only one deployment in an active state at a time.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Inactive</code> - The deployment was previously successfully created, but it is not currently
     *         running on the container service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the log
     *         events for the containers in the deployment to try to determine the reason for the failure.
     *         </p>
     *         </li>
     * @see ContainerServiceDeploymentState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * Returns true if the Containers property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasContainers() {
        return containers != null && !(containers instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An object that describes the configuration for the containers of the deployment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasContainers()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An object that describes the configuration for the containers of the deployment.
     */
    public Map<String, Container> containers() {
        return containers;
    }

    /**
     * <p>
     * An object that describes the endpoint of the deployment.
     * </p>
     * 
     * @return An object that describes the endpoint of the deployment.
     */
    public ContainerServiceEndpoint publicEndpoint() {
        return publicEndpoint;
    }

    /**
     * <p>
     * The timestamp when the deployment was created.
     * </p>
     * 
     * @return The timestamp when the deployment was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainers() ? containers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(publicEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerServiceDeployment)) {
            return false;
        }
        ContainerServiceDeployment other = (ContainerServiceDeployment) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(stateAsString(), other.stateAsString())
                && hasContainers() == other.hasContainers() && Objects.equals(containers(), other.containers())
                && Objects.equals(publicEndpoint(), other.publicEndpoint()) && Objects.equals(createdAt(), other.createdAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContainerServiceDeployment").add("Version", version()).add("State", stateAsString())
                .add("Containers", hasContainers() ? containers() : null).add("PublicEndpoint", publicEndpoint())
                .add("CreatedAt", createdAt()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "containers":
            return Optional.ofNullable(clazz.cast(containers()));
        case "publicEndpoint":
            return Optional.ofNullable(clazz.cast(publicEndpoint()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerServiceDeployment, T> g) {
        return obj -> g.apply((ContainerServiceDeployment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerServiceDeployment> {
        /**
         * <p>
         * The version number of the deployment.
         * </p>
         * 
         * @param version
         *        The version number of the deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Integer version);

        /**
         * <p>
         * The state of the deployment.
         * </p>
         * <p>
         * A deployment can be in one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Activating</code> - The deployment is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Active</code> - The deployment was successfully created, and it's currently running on the container
         * service. The container service can have only one deployment in an active state at a time.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Inactive</code> - The deployment was previously successfully created, but it is not currently running
         * on the container service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the log
         * events for the containers in the deployment to try to determine the reason for the failure.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the deployment.</p>
         *        <p>
         *        A deployment can be in one of the following states:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Activating</code> - The deployment is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Active</code> - The deployment was successfully created, and it's currently running on the
         *        container service. The container service can have only one deployment in an active state at a time.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Inactive</code> - The deployment was previously successfully created, but it is not currently
         *        running on the container service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the
         *        log events for the containers in the deployment to try to determine the reason for the failure.
         *        </p>
         *        </li>
         * @see ContainerServiceDeploymentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerServiceDeploymentState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the deployment.
         * </p>
         * <p>
         * A deployment can be in one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Activating</code> - The deployment is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Active</code> - The deployment was successfully created, and it's currently running on the container
         * service. The container service can have only one deployment in an active state at a time.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Inactive</code> - The deployment was previously successfully created, but it is not currently running
         * on the container service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the log
         * events for the containers in the deployment to try to determine the reason for the failure.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the deployment.</p>
         *        <p>
         *        A deployment can be in one of the following states:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Activating</code> - The deployment is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Active</code> - The deployment was successfully created, and it's currently running on the
         *        container service. The container service can have only one deployment in an active state at a time.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Inactive</code> - The deployment was previously successfully created, but it is not currently
         *        running on the container service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The deployment failed. Use the <code>GetContainerLog</code> action to view the
         *        log events for the containers in the deployment to try to determine the reason for the failure.
         *        </p>
         *        </li>
         * @see ContainerServiceDeploymentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerServiceDeploymentState
         */
        Builder state(ContainerServiceDeploymentState state);

        /**
         * <p>
         * An object that describes the configuration for the containers of the deployment.
         * </p>
         * 
         * @param containers
         *        An object that describes the configuration for the containers of the deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containers(Map<String, Container> containers);

        /**
         * <p>
         * An object that describes the endpoint of the deployment.
         * </p>
         * 
         * @param publicEndpoint
         *        An object that describes the endpoint of the deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicEndpoint(ContainerServiceEndpoint publicEndpoint);

        /**
         * <p>
         * An object that describes the endpoint of the deployment.
         * </p>
         * This is a convenience that creates an instance of the {@link ContainerServiceEndpoint.Builder} avoiding the
         * need to create one manually via {@link ContainerServiceEndpoint#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContainerServiceEndpoint.Builder#build()} is called immediately
         * and its result is passed to {@link #publicEndpoint(ContainerServiceEndpoint)}.
         * 
         * @param publicEndpoint
         *        a consumer that will call methods on {@link ContainerServiceEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicEndpoint(ContainerServiceEndpoint)
         */
        default Builder publicEndpoint(Consumer<ContainerServiceEndpoint.Builder> publicEndpoint) {
            return publicEndpoint(ContainerServiceEndpoint.builder().applyMutation(publicEndpoint).build());
        }

        /**
         * <p>
         * The timestamp when the deployment was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the deployment was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);
    }

    static final class BuilderImpl implements Builder {
        private Integer version;

        private String state;

        private Map<String, Container> containers = DefaultSdkAutoConstructMap.getInstance();

        private ContainerServiceEndpoint publicEndpoint;

        private Instant createdAt;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerServiceDeployment model) {
            version(model.version);
            state(model.state);
            containers(model.containers);
            publicEndpoint(model.publicEndpoint);
            createdAt(model.createdAt);
        }

        public final Integer getVersion() {
            return version;
        }

        @Override
        public final Builder version(Integer version) {
            this.version = version;
            return this;
        }

        public final void setVersion(Integer version) {
            this.version = version;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ContainerServiceDeploymentState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final Map<String, Container.Builder> getContainers() {
            if (containers instanceof SdkAutoConstructMap) {
                return null;
            }
            return containers != null ? CollectionUtils.mapValues(containers, Container::toBuilder) : null;
        }

        @Override
        public final Builder containers(Map<String, Container> containers) {
            this.containers = ContainerMapCopier.copy(containers);
            return this;
        }

        public final void setContainers(Map<String, Container.BuilderImpl> containers) {
            this.containers = ContainerMapCopier.copyFromBuilder(containers);
        }

        public final ContainerServiceEndpoint.Builder getPublicEndpoint() {
            return publicEndpoint != null ? publicEndpoint.toBuilder() : null;
        }

        @Override
        public final Builder publicEndpoint(ContainerServiceEndpoint publicEndpoint) {
            this.publicEndpoint = publicEndpoint;
            return this;
        }

        public final void setPublicEndpoint(ContainerServiceEndpoint.BuilderImpl publicEndpoint) {
            this.publicEndpoint = publicEndpoint != null ? publicEndpoint.build() : null;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public ContainerServiceDeployment build() {
            return new ContainerServiceDeployment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
