/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetDistributionMetricDataRequest extends LightsailRequest implements
        ToCopyableBuilder<GetDistributionMetricDataRequest.Builder, GetDistributionMetricDataRequest> {
    private static final SdkField<String> DISTRIBUTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("distributionName").getter(getter(GetDistributionMetricDataRequest::distributionName))
            .setter(setter(Builder::distributionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("distributionName").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricName").getter(getter(GetDistributionMetricDataRequest::metricNameAsString))
            .setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricName").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(GetDistributionMetricDataRequest::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(GetDistributionMetricDataRequest::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("period").getter(getter(GetDistributionMetricDataRequest::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("period").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("unit")
            .getter(getter(GetDistributionMetricDataRequest::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unit").build()).build();

    private static final SdkField<List<String>> STATISTICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("statistics")
            .getter(getter(GetDistributionMetricDataRequest::statisticsAsStrings))
            .setter(setter(Builder::statisticsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statistics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DISTRIBUTION_NAME_FIELD,
            METRIC_NAME_FIELD, START_TIME_FIELD, END_TIME_FIELD, PERIOD_FIELD, UNIT_FIELD, STATISTICS_FIELD));

    private final String distributionName;

    private final String metricName;

    private final Instant startTime;

    private final Instant endTime;

    private final Integer period;

    private final String unit;

    private final List<String> statistics;

    private GetDistributionMetricDataRequest(BuilderImpl builder) {
        super(builder);
        this.distributionName = builder.distributionName;
        this.metricName = builder.metricName;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.period = builder.period;
        this.unit = builder.unit;
        this.statistics = builder.statistics;
    }

    /**
     * <p>
     * The name of the distribution for which to get metric data.
     * </p>
     * <p>
     * Use the <code>GetDistributions</code> action to get a list of distribution names that you can specify.
     * </p>
     * 
     * @return The name of the distribution for which to get metric data.</p>
     *         <p>
     *         Use the <code>GetDistributions</code> action to get a list of distribution names that you can specify.
     */
    public String distributionName() {
        return distributionName;
    }

    /**
     * <p>
     * The metric for which you want to return information.
     * </p>
     * <p>
     * Valid distribution metric names are listed below, along with the most useful <code>statistics</code> to include
     * in your request, and the published <code>unit</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail distribution, for
     * all HTTP methods, and for both HTTP and HTTPS requests.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>None</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and OPTIONS
     * requests.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>None</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
     * distribution, using POST and PUT requests.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>None</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP status
     * code was 4xx or 5xx.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP status
     * cod was 4xx. In these cases, the client or client viewer may have made an error. For example, a status code of
     * 404 (Not Found) means that the client requested an object that could not be found.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP status
     * code was 5xx. In these cases, the origin server did not satisfy the requests. For example, a status code of 503
     * (Service Unavailable) means that the origin server is currently unavailable.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link DistributionMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #metricNameAsString}.
     * </p>
     * 
     * @return The metric for which you want to return information.</p>
     *         <p>
     *         Valid distribution metric names are listed below, along with the most useful <code>statistics</code> to
     *         include in your request, and the published <code>unit</code> value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail
     *         distribution, for all HTTP methods, and for both HTTP and HTTPS requests.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>None</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and
     *         OPTIONS requests.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>None</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
     *         distribution, using POST and PUT requests.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>None</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's
     *         HTTP status code was 4xx or 5xx.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
     *         status cod was 4xx. In these cases, the client or client viewer may have made an error. For example, a
     *         status code of 404 (Not Found) means that the client requested an object that could not be found.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
     *         status code was 5xx. In these cases, the origin server did not satisfy the requests. For example, a
     *         status code of 503 (Service Unavailable) means that the origin server is currently unavailable.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     * @see DistributionMetricName
     */
    public DistributionMetricName metricName() {
        return DistributionMetricName.fromValue(metricName);
    }

    /**
     * <p>
     * The metric for which you want to return information.
     * </p>
     * <p>
     * Valid distribution metric names are listed below, along with the most useful <code>statistics</code> to include
     * in your request, and the published <code>unit</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail distribution, for
     * all HTTP methods, and for both HTTP and HTTPS requests.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>None</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and OPTIONS
     * requests.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>None</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
     * distribution, using POST and PUT requests.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>None</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP status
     * code was 4xx or 5xx.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP status
     * cod was 4xx. In these cases, the client or client viewer may have made an error. For example, a status code of
     * 404 (Not Found) means that the client requested an object that could not be found.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP status
     * code was 5xx. In these cases, the origin server did not satisfy the requests. For example, a status code of 503
     * (Service Unavailable) means that the origin server is currently unavailable.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link DistributionMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #metricNameAsString}.
     * </p>
     * 
     * @return The metric for which you want to return information.</p>
     *         <p>
     *         Valid distribution metric names are listed below, along with the most useful <code>statistics</code> to
     *         include in your request, and the published <code>unit</code> value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail
     *         distribution, for all HTTP methods, and for both HTTP and HTTPS requests.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>None</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and
     *         OPTIONS requests.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>None</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
     *         distribution, using POST and PUT requests.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>None</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's
     *         HTTP status code was 4xx or 5xx.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
     *         status cod was 4xx. In these cases, the client or client viewer may have made an error. For example, a
     *         status code of 404 (Not Found) means that the client requested an object that could not be found.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
     *         status code was 5xx. In these cases, the origin server did not satisfy the requests. For example, a
     *         status code of 503 (Service Unavailable) means that the origin server is currently unavailable.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     * @see DistributionMetricName
     */
    public String metricNameAsString() {
        return metricName;
    }

    /**
     * <p>
     * The start of the time interval for which to get metric data.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specified in Coordinated Universal Time (UTC).
     * </p>
     * </li>
     * <li>
     * <p>
     * Specified in the Unix time format.
     * </p>
     * <p>
     * For example, if you wish to use a start time of October 1, 2018, at 8 PM UTC, specify <code>1538424000</code> as
     * the start time.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can convert a human-friendly time to Unix time format using a converter like <a
     * href="https://www.epochconverter.com/">Epoch converter</a>.
     * </p>
     * 
     * @return The start of the time interval for which to get metric data.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specified in Coordinated Universal Time (UTC).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specified in the Unix time format.
     *         </p>
     *         <p>
     *         For example, if you wish to use a start time of October 1, 2018, at 8 PM UTC, specify
     *         <code>1538424000</code> as the start time.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can convert a human-friendly time to Unix time format using a converter like <a
     *         href="https://www.epochconverter.com/">Epoch converter</a>.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end of the time interval for which to get metric data.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specified in Coordinated Universal Time (UTC).
     * </p>
     * </li>
     * <li>
     * <p>
     * Specified in the Unix time format.
     * </p>
     * <p>
     * For example, if you wish to use an end time of October 1, 2018, at 9 PM UTC, specify <code>1538427600</code> as
     * the end time.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can convert a human-friendly time to Unix time format using a converter like <a
     * href="https://www.epochconverter.com/">Epoch converter</a>.
     * </p>
     * 
     * @return The end of the time interval for which to get metric data.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specified in Coordinated Universal Time (UTC).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specified in the Unix time format.
     *         </p>
     *         <p>
     *         For example, if you wish to use an end time of October 1, 2018, at 9 PM UTC, specify
     *         <code>1538427600</code> as the end time.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can convert a human-friendly time to Unix time format using a converter like <a
     *         href="https://www.epochconverter.com/">Epoch converter</a>.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The granularity, in seconds, for the metric data points that will be returned.
     * </p>
     * 
     * @return The granularity, in seconds, for the metric data points that will be returned.
     */
    public Integer period() {
        return period;
    }

    /**
     * <p>
     * The unit for the metric data request.
     * </p>
     * <p>
     * Valid units depend on the metric data being requested. For the valid units with each available metric, see the
     * <code>metricName</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for the metric data request.</p>
     *         <p>
     *         Valid units depend on the metric data being requested. For the valid units with each available metric,
     *         see the <code>metricName</code> parameter.
     * @see MetricUnit
     */
    public MetricUnit unit() {
        return MetricUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit for the metric data request.
     * </p>
     * <p>
     * Valid units depend on the metric data being requested. For the valid units with each available metric, see the
     * <code>metricName</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for the metric data request.</p>
     *         <p>
     *         Valid units depend on the metric data being requested. For the valid units with each available metric,
     *         see the <code>metricName</code> parameter.
     * @see MetricUnit
     */
    public String unitAsString() {
        return unit;
    }

    /**
     * <p>
     * The statistic for the metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this statistic
     * with the Minimum and Maximum values, you can determine the full scope of a metric and how close the average use
     * is to the Minimum and Maximum values. This comparison helps you to know when to increase or decrease your
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStatistics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The statistic for the metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - All values submitted for the matching metric added together. You can use this
     *         statistic to determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
     *         statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close
     *         the average use is to the Minimum and Maximum values. This comparison helps you to know when to increase
     *         or decrease your resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     */
    public List<MetricStatistic> statistics() {
        return MetricStatisticListCopier.copyStringToEnum(statistics);
    }

    /**
     * Returns true if the Statistics property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasStatistics() {
        return statistics != null && !(statistics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The statistic for the metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this statistic
     * with the Minimum and Maximum values, you can determine the full scope of a metric and how close the average use
     * is to the Minimum and Maximum values. This comparison helps you to know when to increase or decrease your
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStatistics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The statistic for the metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - All values submitted for the matching metric added together. You can use this
     *         statistic to determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
     *         statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close
     *         the average use is to the Minimum and Maximum values. This comparison helps you to know when to increase
     *         or decrease your resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     */
    public List<String> statisticsAsStrings() {
        return statistics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(distributionName());
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatistics() ? statisticsAsStrings() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetDistributionMetricDataRequest)) {
            return false;
        }
        GetDistributionMetricDataRequest other = (GetDistributionMetricDataRequest) obj;
        return Objects.equals(distributionName(), other.distributionName())
                && Objects.equals(metricNameAsString(), other.metricNameAsString())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(period(), other.period()) && Objects.equals(unitAsString(), other.unitAsString())
                && hasStatistics() == other.hasStatistics() && Objects.equals(statisticsAsStrings(), other.statisticsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetDistributionMetricDataRequest").add("DistributionName", distributionName())
                .add("MetricName", metricNameAsString()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Period", period()).add("Unit", unitAsString())
                .add("Statistics", hasStatistics() ? statisticsAsStrings() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "distributionName":
            return Optional.ofNullable(clazz.cast(distributionName()));
        case "metricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "period":
            return Optional.ofNullable(clazz.cast(period()));
        case "unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        case "statistics":
            return Optional.ofNullable(clazz.cast(statisticsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetDistributionMetricDataRequest, T> g) {
        return obj -> g.apply((GetDistributionMetricDataRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LightsailRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetDistributionMetricDataRequest> {
        /**
         * <p>
         * The name of the distribution for which to get metric data.
         * </p>
         * <p>
         * Use the <code>GetDistributions</code> action to get a list of distribution names that you can specify.
         * </p>
         * 
         * @param distributionName
         *        The name of the distribution for which to get metric data.</p>
         *        <p>
         *        Use the <code>GetDistributions</code> action to get a list of distribution names that you can specify.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distributionName(String distributionName);

        /**
         * <p>
         * The metric for which you want to return information.
         * </p>
         * <p>
         * Valid distribution metric names are listed below, along with the most useful <code>statistics</code> to
         * include in your request, and the published <code>unit</code> value.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail distribution,
         * for all HTTP methods, and for both HTTP and HTTPS requests.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>None</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and OPTIONS
         * requests.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>None</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
         * distribution, using POST and PUT requests.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>None</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
         * status code was 4xx or 5xx.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
         * status cod was 4xx. In these cases, the client or client viewer may have made an error. For example, a status
         * code of 404 (Not Found) means that the client requested an object that could not be found.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
         * status code was 5xx. In these cases, the origin server did not satisfy the requests. For example, a status
         * code of 503 (Service Unavailable) means that the origin server is currently unavailable.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The metric for which you want to return information.</p>
         *        <p>
         *        Valid distribution metric names are listed below, along with the most useful <code>statistics</code>
         *        to include in your request, and the published <code>unit</code> value.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail
         *        distribution, for all HTTP methods, and for both HTTP and HTTPS requests.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>None</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and
         *        OPTIONS requests.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>None</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
         *        distribution, using POST and PUT requests.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>None</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's
         *        HTTP status code was 4xx or 5xx.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's
         *        HTTP status cod was 4xx. In these cases, the client or client viewer may have made an error. For
         *        example, a status code of 404 (Not Found) means that the client requested an object that could not be
         *        found.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's
         *        HTTP status code was 5xx. In these cases, the origin server did not satisfy the requests. For example,
         *        a status code of 503 (Service Unavailable) means that the origin server is currently unavailable.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         * @see DistributionMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistributionMetricName
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The metric for which you want to return information.
         * </p>
         * <p>
         * Valid distribution metric names are listed below, along with the most useful <code>statistics</code> to
         * include in your request, and the published <code>unit</code> value.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail distribution,
         * for all HTTP methods, and for both HTTP and HTTPS requests.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>None</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and OPTIONS
         * requests.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>None</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
         * distribution, using POST and PUT requests.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>None</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
         * status code was 4xx or 5xx.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
         * status cod was 4xx. In these cases, the client or client viewer may have made an error. For example, a status
         * code of 404 (Not Found) means that the client requested an object that could not be found.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's HTTP
         * status code was 5xx. In these cases, the origin server did not satisfy the requests. For example, a status
         * code of 503 (Service Unavailable) means that the origin server is currently unavailable.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The metric for which you want to return information.</p>
         *        <p>
         *        Valid distribution metric names are listed below, along with the most useful <code>statistics</code>
         *        to include in your request, and the published <code>unit</code> value.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Requests</code> </b> - The total number of viewer requests received by your Lightsail
         *        distribution, for all HTTP methods, and for both HTTP and HTTPS requests.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>None</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>BytesDownloaded</code> </b> - The number of bytes downloaded by viewers for GET, HEAD, and
         *        OPTIONS requests.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>None</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>BytesUploaded </code> </b> - The number of bytes uploaded to your origin by your Lightsail
         *        distribution, using POST and PUT requests.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>None</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>TotalErrorRate</code> </b> - The percentage of all viewer requests for which the response's
         *        HTTP status code was 4xx or 5xx.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>4xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's
         *        HTTP status cod was 4xx. In these cases, the client or client viewer may have made an error. For
         *        example, a status code of 404 (Not Found) means that the client requested an object that could not be
         *        found.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>5xxErrorRate</code> </b> - The percentage of all viewer requests for which the response's
         *        HTTP status code was 5xx. In these cases, the origin server did not satisfy the requests. For example,
         *        a status code of 503 (Service Unavailable) means that the origin server is currently unavailable.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         * @see DistributionMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistributionMetricName
         */
        Builder metricName(DistributionMetricName metricName);

        /**
         * <p>
         * The start of the time interval for which to get metric data.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specified in Coordinated Universal Time (UTC).
         * </p>
         * </li>
         * <li>
         * <p>
         * Specified in the Unix time format.
         * </p>
         * <p>
         * For example, if you wish to use a start time of October 1, 2018, at 8 PM UTC, specify <code>1538424000</code>
         * as the start time.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can convert a human-friendly time to Unix time format using a converter like <a
         * href="https://www.epochconverter.com/">Epoch converter</a>.
         * </p>
         * 
         * @param startTime
         *        The start of the time interval for which to get metric data.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specified in Coordinated Universal Time (UTC).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specified in the Unix time format.
         *        </p>
         *        <p>
         *        For example, if you wish to use a start time of October 1, 2018, at 8 PM UTC, specify
         *        <code>1538424000</code> as the start time.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can convert a human-friendly time to Unix time format using a converter like <a
         *        href="https://www.epochconverter.com/">Epoch converter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end of the time interval for which to get metric data.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specified in Coordinated Universal Time (UTC).
         * </p>
         * </li>
         * <li>
         * <p>
         * Specified in the Unix time format.
         * </p>
         * <p>
         * For example, if you wish to use an end time of October 1, 2018, at 9 PM UTC, specify <code>1538427600</code>
         * as the end time.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can convert a human-friendly time to Unix time format using a converter like <a
         * href="https://www.epochconverter.com/">Epoch converter</a>.
         * </p>
         * 
         * @param endTime
         *        The end of the time interval for which to get metric data.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specified in Coordinated Universal Time (UTC).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specified in the Unix time format.
         *        </p>
         *        <p>
         *        For example, if you wish to use an end time of October 1, 2018, at 9 PM UTC, specify
         *        <code>1538427600</code> as the end time.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can convert a human-friendly time to Unix time format using a converter like <a
         *        href="https://www.epochconverter.com/">Epoch converter</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The granularity, in seconds, for the metric data points that will be returned.
         * </p>
         * 
         * @param period
         *        The granularity, in seconds, for the metric data points that will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The unit for the metric data request.
         * </p>
         * <p>
         * Valid units depend on the metric data being requested. For the valid units with each available metric, see
         * the <code>metricName</code> parameter.
         * </p>
         * 
         * @param unit
         *        The unit for the metric data request.</p>
         *        <p>
         *        Valid units depend on the metric data being requested. For the valid units with each available metric,
         *        see the <code>metricName</code> parameter.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit for the metric data request.
         * </p>
         * <p>
         * Valid units depend on the metric data being requested. For the valid units with each available metric, see
         * the <code>metricName</code> parameter.
         * </p>
         * 
         * @param unit
         *        The unit for the metric data request.</p>
         *        <p>
         *        Valid units depend on the metric data being requested. For the valid units with each available metric,
         *        see the <code>metricName</code> parameter.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(MetricUnit unit);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(Collection<String> statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(String... statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(Collection<MetricStatistic> statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(MetricStatistic... statistics);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LightsailRequest.BuilderImpl implements Builder {
        private String distributionName;

        private String metricName;

        private Instant startTime;

        private Instant endTime;

        private Integer period;

        private String unit;

        private List<String> statistics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetDistributionMetricDataRequest model) {
            super(model);
            distributionName(model.distributionName);
            metricName(model.metricName);
            startTime(model.startTime);
            endTime(model.endTime);
            period(model.period);
            unit(model.unit);
            statisticsWithStrings(model.statistics);
        }

        public final String getDistributionName() {
            return distributionName;
        }

        @Override
        public final Builder distributionName(String distributionName) {
            this.distributionName = distributionName;
            return this;
        }

        public final void setDistributionName(String distributionName) {
            this.distributionName = distributionName;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public final Builder metricName(DistributionMetricName metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final Integer getPeriod() {
            return period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(MetricUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        public final Collection<String> getStatistics() {
            if (statistics instanceof SdkAutoConstructList) {
                return null;
            }
            return statistics;
        }

        @Override
        public final Builder statisticsWithStrings(Collection<String> statistics) {
            this.statistics = MetricStatisticListCopier.copy(statistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statisticsWithStrings(String... statistics) {
            statisticsWithStrings(Arrays.asList(statistics));
            return this;
        }

        @Override
        public final Builder statistics(Collection<MetricStatistic> statistics) {
            this.statistics = MetricStatisticListCopier.copyEnumToString(statistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statistics(MetricStatistic... statistics) {
            statistics(Arrays.asList(statistics));
            return this;
        }

        public final void setStatistics(Collection<String> statistics) {
            this.statistics = MetricStatisticListCopier.copy(statistics);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetDistributionMetricDataRequest build() {
            return new GetDistributionMetricDataRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
