/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a database.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RelationalDatabase implements SdkPojo, Serializable,
        ToCopyableBuilder<RelationalDatabase.Builder, RelationalDatabase> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(RelationalDatabase::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(RelationalDatabase::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("supportCode").getter(getter(RelationalDatabase::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(RelationalDatabase::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).memberName("location")
            .getter(getter(RelationalDatabase::location)).setter(setter(Builder::location))
            .constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(RelationalDatabase::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(RelationalDatabase::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RELATIONAL_DATABASE_BLUEPRINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("relationalDatabaseBlueprintId")
            .getter(getter(RelationalDatabase::relationalDatabaseBlueprintId))
            .setter(setter(Builder::relationalDatabaseBlueprintId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relationalDatabaseBlueprintId")
                    .build()).build();

    private static final SdkField<String> RELATIONAL_DATABASE_BUNDLE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("relationalDatabaseBundleId")
            .getter(getter(RelationalDatabase::relationalDatabaseBundleId))
            .setter(setter(Builder::relationalDatabaseBundleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relationalDatabaseBundleId").build())
            .build();

    private static final SdkField<String> MASTER_DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("masterDatabaseName").getter(getter(RelationalDatabase::masterDatabaseName))
            .setter(setter(Builder::masterDatabaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("masterDatabaseName").build())
            .build();

    private static final SdkField<RelationalDatabaseHardware> HARDWARE_FIELD = SdkField
            .<RelationalDatabaseHardware> builder(MarshallingType.SDK_POJO).memberName("hardware")
            .getter(getter(RelationalDatabase::hardware)).setter(setter(Builder::hardware))
            .constructor(RelationalDatabaseHardware::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hardware").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(RelationalDatabase::state)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> SECONDARY_AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("secondaryAvailabilityZone").getter(getter(RelationalDatabase::secondaryAvailabilityZone))
            .setter(setter(Builder::secondaryAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondaryAvailabilityZone").build())
            .build();

    private static final SdkField<Boolean> BACKUP_RETENTION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("backupRetentionEnabled").getter(getter(RelationalDatabase::backupRetentionEnabled))
            .setter(setter(Builder::backupRetentionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("backupRetentionEnabled").build())
            .build();

    private static final SdkField<PendingModifiedRelationalDatabaseValues> PENDING_MODIFIED_VALUES_FIELD = SdkField
            .<PendingModifiedRelationalDatabaseValues> builder(MarshallingType.SDK_POJO).memberName("pendingModifiedValues")
            .getter(getter(RelationalDatabase::pendingModifiedValues)).setter(setter(Builder::pendingModifiedValues))
            .constructor(PendingModifiedRelationalDatabaseValues::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pendingModifiedValues").build())
            .build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("engine")
            .getter(getter(RelationalDatabase::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("engineVersion").getter(getter(RelationalDatabase::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("engineVersion").build()).build();

    private static final SdkField<Instant> LATEST_RESTORABLE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("latestRestorableTime").getter(getter(RelationalDatabase::latestRestorableTime))
            .setter(setter(Builder::latestRestorableTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("latestRestorableTime").build())
            .build();

    private static final SdkField<String> MASTER_USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("masterUsername").getter(getter(RelationalDatabase::masterUsername))
            .setter(setter(Builder::masterUsername))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("masterUsername").build()).build();

    private static final SdkField<String> PARAMETER_APPLY_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("parameterApplyStatus").getter(getter(RelationalDatabase::parameterApplyStatus))
            .setter(setter(Builder::parameterApplyStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parameterApplyStatus").build())
            .build();

    private static final SdkField<String> PREFERRED_BACKUP_WINDOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("preferredBackupWindow").getter(getter(RelationalDatabase::preferredBackupWindow))
            .setter(setter(Builder::preferredBackupWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("preferredBackupWindow").build())
            .build();

    private static final SdkField<String> PREFERRED_MAINTENANCE_WINDOW_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("preferredMaintenanceWindow")
            .getter(getter(RelationalDatabase::preferredMaintenanceWindow))
            .setter(setter(Builder::preferredMaintenanceWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("preferredMaintenanceWindow").build())
            .build();

    private static final SdkField<Boolean> PUBLICLY_ACCESSIBLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("publiclyAccessible").getter(getter(RelationalDatabase::publiclyAccessible))
            .setter(setter(Builder::publiclyAccessible))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publiclyAccessible").build())
            .build();

    private static final SdkField<RelationalDatabaseEndpoint> MASTER_ENDPOINT_FIELD = SdkField
            .<RelationalDatabaseEndpoint> builder(MarshallingType.SDK_POJO).memberName("masterEndpoint")
            .getter(getter(RelationalDatabase::masterEndpoint)).setter(setter(Builder::masterEndpoint))
            .constructor(RelationalDatabaseEndpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("masterEndpoint").build()).build();

    private static final SdkField<List<PendingMaintenanceAction>> PENDING_MAINTENANCE_ACTIONS_FIELD = SdkField
            .<List<PendingMaintenanceAction>> builder(MarshallingType.LIST)
            .memberName("pendingMaintenanceActions")
            .getter(getter(RelationalDatabase::pendingMaintenanceActions))
            .setter(setter(Builder::pendingMaintenanceActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pendingMaintenanceActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PendingMaintenanceAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(PendingMaintenanceAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CA_CERTIFICATE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("caCertificateIdentifier").getter(getter(RelationalDatabase::caCertificateIdentifier))
            .setter(setter(Builder::caCertificateIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("caCertificateIdentifier").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD,
            RELATIONAL_DATABASE_BLUEPRINT_ID_FIELD, RELATIONAL_DATABASE_BUNDLE_ID_FIELD, MASTER_DATABASE_NAME_FIELD,
            HARDWARE_FIELD, STATE_FIELD, SECONDARY_AVAILABILITY_ZONE_FIELD, BACKUP_RETENTION_ENABLED_FIELD,
            PENDING_MODIFIED_VALUES_FIELD, ENGINE_FIELD, ENGINE_VERSION_FIELD, LATEST_RESTORABLE_TIME_FIELD,
            MASTER_USERNAME_FIELD, PARAMETER_APPLY_STATUS_FIELD, PREFERRED_BACKUP_WINDOW_FIELD,
            PREFERRED_MAINTENANCE_WINDOW_FIELD, PUBLICLY_ACCESSIBLE_FIELD, MASTER_ENDPOINT_FIELD,
            PENDING_MAINTENANCE_ACTIONS_FIELD, CA_CERTIFICATE_IDENTIFIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final String relationalDatabaseBlueprintId;

    private final String relationalDatabaseBundleId;

    private final String masterDatabaseName;

    private final RelationalDatabaseHardware hardware;

    private final String state;

    private final String secondaryAvailabilityZone;

    private final Boolean backupRetentionEnabled;

    private final PendingModifiedRelationalDatabaseValues pendingModifiedValues;

    private final String engine;

    private final String engineVersion;

    private final Instant latestRestorableTime;

    private final String masterUsername;

    private final String parameterApplyStatus;

    private final String preferredBackupWindow;

    private final String preferredMaintenanceWindow;

    private final Boolean publiclyAccessible;

    private final RelationalDatabaseEndpoint masterEndpoint;

    private final List<PendingMaintenanceAction> pendingMaintenanceActions;

    private final String caCertificateIdentifier;

    private RelationalDatabase(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.relationalDatabaseBlueprintId = builder.relationalDatabaseBlueprintId;
        this.relationalDatabaseBundleId = builder.relationalDatabaseBundleId;
        this.masterDatabaseName = builder.masterDatabaseName;
        this.hardware = builder.hardware;
        this.state = builder.state;
        this.secondaryAvailabilityZone = builder.secondaryAvailabilityZone;
        this.backupRetentionEnabled = builder.backupRetentionEnabled;
        this.pendingModifiedValues = builder.pendingModifiedValues;
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.latestRestorableTime = builder.latestRestorableTime;
        this.masterUsername = builder.masterUsername;
        this.parameterApplyStatus = builder.parameterApplyStatus;
        this.preferredBackupWindow = builder.preferredBackupWindow;
        this.preferredMaintenanceWindow = builder.preferredMaintenanceWindow;
        this.publiclyAccessible = builder.publiclyAccessible;
        this.masterEndpoint = builder.masterEndpoint;
        this.pendingMaintenanceActions = builder.pendingMaintenanceActions;
        this.caCertificateIdentifier = builder.caCertificateIdentifier;
    }

    /**
     * <p>
     * The unique name of the database resource in Lightsail.
     * </p>
     * 
     * @return The unique name of the database resource in Lightsail.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the database.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the database.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code for the database. Include this code in your email to support when you have questions about a
     * database in Lightsail. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code for the database. Include this code in your email to support when you have questions
     *         about a database in Lightsail. This code enables our support team to look up your Lightsail information
     *         more easily.
     */
    public final String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The timestamp when the database was created. Formatted in Unix time.
     * </p>
     * 
     * @return The timestamp when the database was created. Formatted in Unix time.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Region name and Availability Zone where the database is located.
     * </p>
     * 
     * @return The Region name and Availability Zone where the database is located.
     */
    public final ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
     *         Guide</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The blueprint ID for the database. A blueprint describes the major engine version of a database.
     * </p>
     * 
     * @return The blueprint ID for the database. A blueprint describes the major engine version of a database.
     */
    public final String relationalDatabaseBlueprintId() {
        return relationalDatabaseBlueprintId;
    }

    /**
     * <p>
     * The bundle ID for the database. A bundle describes the performance specifications for your database.
     * </p>
     * 
     * @return The bundle ID for the database. A bundle describes the performance specifications for your database.
     */
    public final String relationalDatabaseBundleId() {
        return relationalDatabaseBundleId;
    }

    /**
     * <p>
     * The name of the master database created when the Lightsail database resource is created.
     * </p>
     * 
     * @return The name of the master database created when the Lightsail database resource is created.
     */
    public final String masterDatabaseName() {
        return masterDatabaseName;
    }

    /**
     * <p>
     * Describes the hardware of the database.
     * </p>
     * 
     * @return Describes the hardware of the database.
     */
    public final RelationalDatabaseHardware hardware() {
        return hardware;
    }

    /**
     * <p>
     * Describes the current state of the database.
     * </p>
     * 
     * @return Describes the current state of the database.
     */
    public final String state() {
        return state;
    }

    /**
     * <p>
     * Describes the secondary Availability Zone of a high availability database.
     * </p>
     * <p>
     * The secondary database is used for failover support of a high availability database.
     * </p>
     * 
     * @return Describes the secondary Availability Zone of a high availability database.</p>
     *         <p>
     *         The secondary database is used for failover support of a high availability database.
     */
    public final String secondaryAvailabilityZone() {
        return secondaryAvailabilityZone;
    }

    /**
     * <p>
     * A Boolean value indicating whether automated backup retention is enabled for the database.
     * </p>
     * 
     * @return A Boolean value indicating whether automated backup retention is enabled for the database.
     */
    public final Boolean backupRetentionEnabled() {
        return backupRetentionEnabled;
    }

    /**
     * <p>
     * Describes pending database value modifications.
     * </p>
     * 
     * @return Describes pending database value modifications.
     */
    public final PendingModifiedRelationalDatabaseValues pendingModifiedValues() {
        return pendingModifiedValues;
    }

    /**
     * <p>
     * The database software (for example, <code>MySQL</code>).
     * </p>
     * 
     * @return The database software (for example, <code>MySQL</code>).
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The database engine version (for example, <code>5.7.23</code>).
     * </p>
     * 
     * @return The database engine version (for example, <code>5.7.23</code>).
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The latest point in time to which the database can be restored. Formatted in Unix time.
     * </p>
     * 
     * @return The latest point in time to which the database can be restored. Formatted in Unix time.
     */
    public final Instant latestRestorableTime() {
        return latestRestorableTime;
    }

    /**
     * <p>
     * The master user name of the database.
     * </p>
     * 
     * @return The master user name of the database.
     */
    public final String masterUsername() {
        return masterUsername;
    }

    /**
     * <p>
     * The status of parameter updates for the database.
     * </p>
     * 
     * @return The status of parameter updates for the database.
     */
    public final String parameterApplyStatus() {
        return parameterApplyStatus;
    }

    /**
     * <p>
     * The daily time range during which automated backups are created for the database (for example,
     * <code>16:00-16:30</code>).
     * </p>
     * 
     * @return The daily time range during which automated backups are created for the database (for example,
     *         <code>16:00-16:30</code>).
     */
    public final String preferredBackupWindow() {
        return preferredBackupWindow;
    }

    /**
     * <p>
     * The weekly time range during which system maintenance can occur on the database.
     * </p>
     * <p>
     * In the format <code>ddd:hh24:mi-ddd:hh24:mi</code>. For example, <code>Tue:17:00-Tue:17:30</code>.
     * </p>
     * 
     * @return The weekly time range during which system maintenance can occur on the database.</p>
     *         <p>
     *         In the format <code>ddd:hh24:mi-ddd:hh24:mi</code>. For example, <code>Tue:17:00-Tue:17:30</code>.
     */
    public final String preferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }

    /**
     * <p>
     * A Boolean value indicating whether the database is publicly accessible.
     * </p>
     * 
     * @return A Boolean value indicating whether the database is publicly accessible.
     */
    public final Boolean publiclyAccessible() {
        return publiclyAccessible;
    }

    /**
     * <p>
     * The master endpoint for the database.
     * </p>
     * 
     * @return The master endpoint for the database.
     */
    public final RelationalDatabaseEndpoint masterEndpoint() {
        return masterEndpoint;
    }

    /**
     * Returns true if the PendingMaintenanceActions property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasPendingMaintenanceActions() {
        return pendingMaintenanceActions != null && !(pendingMaintenanceActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the pending maintenance actions for the database.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPendingMaintenanceActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Describes the pending maintenance actions for the database.
     */
    public final List<PendingMaintenanceAction> pendingMaintenanceActions() {
        return pendingMaintenanceActions;
    }

    /**
     * <p>
     * The certificate associated with the database.
     * </p>
     * 
     * @return The certificate associated with the database.
     */
    public final String caCertificateIdentifier() {
        return caCertificateIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(relationalDatabaseBlueprintId());
        hashCode = 31 * hashCode + Objects.hashCode(relationalDatabaseBundleId());
        hashCode = 31 * hashCode + Objects.hashCode(masterDatabaseName());
        hashCode = 31 * hashCode + Objects.hashCode(hardware());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(secondaryAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(backupRetentionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(pendingModifiedValues());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(latestRestorableTime());
        hashCode = 31 * hashCode + Objects.hashCode(masterUsername());
        hashCode = 31 * hashCode + Objects.hashCode(parameterApplyStatus());
        hashCode = 31 * hashCode + Objects.hashCode(preferredBackupWindow());
        hashCode = 31 * hashCode + Objects.hashCode(preferredMaintenanceWindow());
        hashCode = 31 * hashCode + Objects.hashCode(publiclyAccessible());
        hashCode = 31 * hashCode + Objects.hashCode(masterEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(hasPendingMaintenanceActions() ? pendingMaintenanceActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(caCertificateIdentifier());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RelationalDatabase)) {
            return false;
        }
        RelationalDatabase other = (RelationalDatabase) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(relationalDatabaseBlueprintId(), other.relationalDatabaseBlueprintId())
                && Objects.equals(relationalDatabaseBundleId(), other.relationalDatabaseBundleId())
                && Objects.equals(masterDatabaseName(), other.masterDatabaseName())
                && Objects.equals(hardware(), other.hardware()) && Objects.equals(state(), other.state())
                && Objects.equals(secondaryAvailabilityZone(), other.secondaryAvailabilityZone())
                && Objects.equals(backupRetentionEnabled(), other.backupRetentionEnabled())
                && Objects.equals(pendingModifiedValues(), other.pendingModifiedValues())
                && Objects.equals(engine(), other.engine()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(latestRestorableTime(), other.latestRestorableTime())
                && Objects.equals(masterUsername(), other.masterUsername())
                && Objects.equals(parameterApplyStatus(), other.parameterApplyStatus())
                && Objects.equals(preferredBackupWindow(), other.preferredBackupWindow())
                && Objects.equals(preferredMaintenanceWindow(), other.preferredMaintenanceWindow())
                && Objects.equals(publiclyAccessible(), other.publiclyAccessible())
                && Objects.equals(masterEndpoint(), other.masterEndpoint())
                && hasPendingMaintenanceActions() == other.hasPendingMaintenanceActions()
                && Objects.equals(pendingMaintenanceActions(), other.pendingMaintenanceActions())
                && Objects.equals(caCertificateIdentifier(), other.caCertificateIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RelationalDatabase").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("RelationalDatabaseBlueprintId", relationalDatabaseBlueprintId())
                .add("RelationalDatabaseBundleId", relationalDatabaseBundleId()).add("MasterDatabaseName", masterDatabaseName())
                .add("Hardware", hardware()).add("State", state()).add("SecondaryAvailabilityZone", secondaryAvailabilityZone())
                .add("BackupRetentionEnabled", backupRetentionEnabled()).add("PendingModifiedValues", pendingModifiedValues())
                .add("Engine", engine()).add("EngineVersion", engineVersion())
                .add("LatestRestorableTime", latestRestorableTime()).add("MasterUsername", masterUsername())
                .add("ParameterApplyStatus", parameterApplyStatus()).add("PreferredBackupWindow", preferredBackupWindow())
                .add("PreferredMaintenanceWindow", preferredMaintenanceWindow()).add("PubliclyAccessible", publiclyAccessible())
                .add("MasterEndpoint", masterEndpoint())
                .add("PendingMaintenanceActions", hasPendingMaintenanceActions() ? pendingMaintenanceActions() : null)
                .add("CaCertificateIdentifier", caCertificateIdentifier()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "relationalDatabaseBlueprintId":
            return Optional.ofNullable(clazz.cast(relationalDatabaseBlueprintId()));
        case "relationalDatabaseBundleId":
            return Optional.ofNullable(clazz.cast(relationalDatabaseBundleId()));
        case "masterDatabaseName":
            return Optional.ofNullable(clazz.cast(masterDatabaseName()));
        case "hardware":
            return Optional.ofNullable(clazz.cast(hardware()));
        case "state":
            return Optional.ofNullable(clazz.cast(state()));
        case "secondaryAvailabilityZone":
            return Optional.ofNullable(clazz.cast(secondaryAvailabilityZone()));
        case "backupRetentionEnabled":
            return Optional.ofNullable(clazz.cast(backupRetentionEnabled()));
        case "pendingModifiedValues":
            return Optional.ofNullable(clazz.cast(pendingModifiedValues()));
        case "engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "engineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "latestRestorableTime":
            return Optional.ofNullable(clazz.cast(latestRestorableTime()));
        case "masterUsername":
            return Optional.ofNullable(clazz.cast(masterUsername()));
        case "parameterApplyStatus":
            return Optional.ofNullable(clazz.cast(parameterApplyStatus()));
        case "preferredBackupWindow":
            return Optional.ofNullable(clazz.cast(preferredBackupWindow()));
        case "preferredMaintenanceWindow":
            return Optional.ofNullable(clazz.cast(preferredMaintenanceWindow()));
        case "publiclyAccessible":
            return Optional.ofNullable(clazz.cast(publiclyAccessible()));
        case "masterEndpoint":
            return Optional.ofNullable(clazz.cast(masterEndpoint()));
        case "pendingMaintenanceActions":
            return Optional.ofNullable(clazz.cast(pendingMaintenanceActions()));
        case "caCertificateIdentifier":
            return Optional.ofNullable(clazz.cast(caCertificateIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RelationalDatabase, T> g) {
        return obj -> g.apply((RelationalDatabase) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RelationalDatabase> {
        /**
         * <p>
         * The unique name of the database resource in Lightsail.
         * </p>
         * 
         * @param name
         *        The unique name of the database resource in Lightsail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the database.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code for the database. Include this code in your email to support when you have questions about a
         * database in Lightsail. This code enables our support team to look up your Lightsail information more easily.
         * </p>
         * 
         * @param supportCode
         *        The support code for the database. Include this code in your email to support when you have questions
         *        about a database in Lightsail. This code enables our support team to look up your Lightsail
         *        information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The timestamp when the database was created. Formatted in Unix time.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the database was created. Formatted in Unix time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Region name and Availability Zone where the database is located.
         * </p>
         * 
         * @param location
         *        The Region name and Availability Zone where the database is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The Region name and Availability Zone where the database is located.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type for the database (for example, <code>RelationalDatabase</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The blueprint ID for the database. A blueprint describes the major engine version of a database.
         * </p>
         * 
         * @param relationalDatabaseBlueprintId
         *        The blueprint ID for the database. A blueprint describes the major engine version of a database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationalDatabaseBlueprintId(String relationalDatabaseBlueprintId);

        /**
         * <p>
         * The bundle ID for the database. A bundle describes the performance specifications for your database.
         * </p>
         * 
         * @param relationalDatabaseBundleId
         *        The bundle ID for the database. A bundle describes the performance specifications for your database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationalDatabaseBundleId(String relationalDatabaseBundleId);

        /**
         * <p>
         * The name of the master database created when the Lightsail database resource is created.
         * </p>
         * 
         * @param masterDatabaseName
         *        The name of the master database created when the Lightsail database resource is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterDatabaseName(String masterDatabaseName);

        /**
         * <p>
         * Describes the hardware of the database.
         * </p>
         * 
         * @param hardware
         *        Describes the hardware of the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hardware(RelationalDatabaseHardware hardware);

        /**
         * <p>
         * Describes the hardware of the database.
         * </p>
         * This is a convenience that creates an instance of the {@link RelationalDatabaseHardware.Builder} avoiding the
         * need to create one manually via {@link RelationalDatabaseHardware#builder()}.
         *
         * When the {@link Consumer} completes, {@link RelationalDatabaseHardware.Builder#build()} is called immediately
         * and its result is passed to {@link #hardware(RelationalDatabaseHardware)}.
         * 
         * @param hardware
         *        a consumer that will call methods on {@link RelationalDatabaseHardware.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hardware(RelationalDatabaseHardware)
         */
        default Builder hardware(Consumer<RelationalDatabaseHardware.Builder> hardware) {
            return hardware(RelationalDatabaseHardware.builder().applyMutation(hardware).build());
        }

        /**
         * <p>
         * Describes the current state of the database.
         * </p>
         * 
         * @param state
         *        Describes the current state of the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * Describes the secondary Availability Zone of a high availability database.
         * </p>
         * <p>
         * The secondary database is used for failover support of a high availability database.
         * </p>
         * 
         * @param secondaryAvailabilityZone
         *        Describes the secondary Availability Zone of a high availability database.</p>
         *        <p>
         *        The secondary database is used for failover support of a high availability database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryAvailabilityZone(String secondaryAvailabilityZone);

        /**
         * <p>
         * A Boolean value indicating whether automated backup retention is enabled for the database.
         * </p>
         * 
         * @param backupRetentionEnabled
         *        A Boolean value indicating whether automated backup retention is enabled for the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupRetentionEnabled(Boolean backupRetentionEnabled);

        /**
         * <p>
         * Describes pending database value modifications.
         * </p>
         * 
         * @param pendingModifiedValues
         *        Describes pending database value modifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingModifiedValues(PendingModifiedRelationalDatabaseValues pendingModifiedValues);

        /**
         * <p>
         * Describes pending database value modifications.
         * </p>
         * This is a convenience that creates an instance of the {@link PendingModifiedRelationalDatabaseValues.Builder}
         * avoiding the need to create one manually via {@link PendingModifiedRelationalDatabaseValues#builder()}.
         *
         * When the {@link Consumer} completes, {@link PendingModifiedRelationalDatabaseValues.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #pendingModifiedValues(PendingModifiedRelationalDatabaseValues)}.
         * 
         * @param pendingModifiedValues
         *        a consumer that will call methods on {@link PendingModifiedRelationalDatabaseValues.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pendingModifiedValues(PendingModifiedRelationalDatabaseValues)
         */
        default Builder pendingModifiedValues(Consumer<PendingModifiedRelationalDatabaseValues.Builder> pendingModifiedValues) {
            return pendingModifiedValues(PendingModifiedRelationalDatabaseValues.builder().applyMutation(pendingModifiedValues)
                    .build());
        }

        /**
         * <p>
         * The database software (for example, <code>MySQL</code>).
         * </p>
         * 
         * @param engine
         *        The database software (for example, <code>MySQL</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The database engine version (for example, <code>5.7.23</code>).
         * </p>
         * 
         * @param engineVersion
         *        The database engine version (for example, <code>5.7.23</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The latest point in time to which the database can be restored. Formatted in Unix time.
         * </p>
         * 
         * @param latestRestorableTime
         *        The latest point in time to which the database can be restored. Formatted in Unix time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestRestorableTime(Instant latestRestorableTime);

        /**
         * <p>
         * The master user name of the database.
         * </p>
         * 
         * @param masterUsername
         *        The master user name of the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUsername(String masterUsername);

        /**
         * <p>
         * The status of parameter updates for the database.
         * </p>
         * 
         * @param parameterApplyStatus
         *        The status of parameter updates for the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterApplyStatus(String parameterApplyStatus);

        /**
         * <p>
         * The daily time range during which automated backups are created for the database (for example,
         * <code>16:00-16:30</code>).
         * </p>
         * 
         * @param preferredBackupWindow
         *        The daily time range during which automated backups are created for the database (for example,
         *        <code>16:00-16:30</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredBackupWindow(String preferredBackupWindow);

        /**
         * <p>
         * The weekly time range during which system maintenance can occur on the database.
         * </p>
         * <p>
         * In the format <code>ddd:hh24:mi-ddd:hh24:mi</code>. For example, <code>Tue:17:00-Tue:17:30</code>.
         * </p>
         * 
         * @param preferredMaintenanceWindow
         *        The weekly time range during which system maintenance can occur on the database.</p>
         *        <p>
         *        In the format <code>ddd:hh24:mi-ddd:hh24:mi</code>. For example, <code>Tue:17:00-Tue:17:30</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredMaintenanceWindow(String preferredMaintenanceWindow);

        /**
         * <p>
         * A Boolean value indicating whether the database is publicly accessible.
         * </p>
         * 
         * @param publiclyAccessible
         *        A Boolean value indicating whether the database is publicly accessible.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publiclyAccessible(Boolean publiclyAccessible);

        /**
         * <p>
         * The master endpoint for the database.
         * </p>
         * 
         * @param masterEndpoint
         *        The master endpoint for the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterEndpoint(RelationalDatabaseEndpoint masterEndpoint);

        /**
         * <p>
         * The master endpoint for the database.
         * </p>
         * This is a convenience that creates an instance of the {@link RelationalDatabaseEndpoint.Builder} avoiding the
         * need to create one manually via {@link RelationalDatabaseEndpoint#builder()}.
         *
         * When the {@link Consumer} completes, {@link RelationalDatabaseEndpoint.Builder#build()} is called immediately
         * and its result is passed to {@link #masterEndpoint(RelationalDatabaseEndpoint)}.
         * 
         * @param masterEndpoint
         *        a consumer that will call methods on {@link RelationalDatabaseEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #masterEndpoint(RelationalDatabaseEndpoint)
         */
        default Builder masterEndpoint(Consumer<RelationalDatabaseEndpoint.Builder> masterEndpoint) {
            return masterEndpoint(RelationalDatabaseEndpoint.builder().applyMutation(masterEndpoint).build());
        }

        /**
         * <p>
         * Describes the pending maintenance actions for the database.
         * </p>
         * 
         * @param pendingMaintenanceActions
         *        Describes the pending maintenance actions for the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingMaintenanceActions(Collection<PendingMaintenanceAction> pendingMaintenanceActions);

        /**
         * <p>
         * Describes the pending maintenance actions for the database.
         * </p>
         * 
         * @param pendingMaintenanceActions
         *        Describes the pending maintenance actions for the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingMaintenanceActions(PendingMaintenanceAction... pendingMaintenanceActions);

        /**
         * <p>
         * Describes the pending maintenance actions for the database.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PendingMaintenanceAction>.Builder} avoiding
         * the need to create one manually via {@link List<PendingMaintenanceAction>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PendingMaintenanceAction>.Builder#build()} is called
         * immediately and its result is passed to {@link #pendingMaintenanceActions(List<PendingMaintenanceAction>)}.
         * 
         * @param pendingMaintenanceActions
         *        a consumer that will call methods on {@link List<PendingMaintenanceAction>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pendingMaintenanceActions(List<PendingMaintenanceAction>)
         */
        Builder pendingMaintenanceActions(Consumer<PendingMaintenanceAction.Builder>... pendingMaintenanceActions);

        /**
         * <p>
         * The certificate associated with the database.
         * </p>
         * 
         * @param caCertificateIdentifier
         *        The certificate associated with the database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder caCertificateIdentifier(String caCertificateIdentifier);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String relationalDatabaseBlueprintId;

        private String relationalDatabaseBundleId;

        private String masterDatabaseName;

        private RelationalDatabaseHardware hardware;

        private String state;

        private String secondaryAvailabilityZone;

        private Boolean backupRetentionEnabled;

        private PendingModifiedRelationalDatabaseValues pendingModifiedValues;

        private String engine;

        private String engineVersion;

        private Instant latestRestorableTime;

        private String masterUsername;

        private String parameterApplyStatus;

        private String preferredBackupWindow;

        private String preferredMaintenanceWindow;

        private Boolean publiclyAccessible;

        private RelationalDatabaseEndpoint masterEndpoint;

        private List<PendingMaintenanceAction> pendingMaintenanceActions = DefaultSdkAutoConstructList.getInstance();

        private String caCertificateIdentifier;

        private BuilderImpl() {
        }

        private BuilderImpl(RelationalDatabase model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            relationalDatabaseBlueprintId(model.relationalDatabaseBlueprintId);
            relationalDatabaseBundleId(model.relationalDatabaseBundleId);
            masterDatabaseName(model.masterDatabaseName);
            hardware(model.hardware);
            state(model.state);
            secondaryAvailabilityZone(model.secondaryAvailabilityZone);
            backupRetentionEnabled(model.backupRetentionEnabled);
            pendingModifiedValues(model.pendingModifiedValues);
            engine(model.engine);
            engineVersion(model.engineVersion);
            latestRestorableTime(model.latestRestorableTime);
            masterUsername(model.masterUsername);
            parameterApplyStatus(model.parameterApplyStatus);
            preferredBackupWindow(model.preferredBackupWindow);
            preferredMaintenanceWindow(model.preferredMaintenanceWindow);
            publiclyAccessible(model.publiclyAccessible);
            masterEndpoint(model.masterEndpoint);
            pendingMaintenanceActions(model.pendingMaintenanceActions);
            caCertificateIdentifier(model.caCertificateIdentifier);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getRelationalDatabaseBlueprintId() {
            return relationalDatabaseBlueprintId;
        }

        @Override
        public final Builder relationalDatabaseBlueprintId(String relationalDatabaseBlueprintId) {
            this.relationalDatabaseBlueprintId = relationalDatabaseBlueprintId;
            return this;
        }

        public final void setRelationalDatabaseBlueprintId(String relationalDatabaseBlueprintId) {
            this.relationalDatabaseBlueprintId = relationalDatabaseBlueprintId;
        }

        public final String getRelationalDatabaseBundleId() {
            return relationalDatabaseBundleId;
        }

        @Override
        public final Builder relationalDatabaseBundleId(String relationalDatabaseBundleId) {
            this.relationalDatabaseBundleId = relationalDatabaseBundleId;
            return this;
        }

        public final void setRelationalDatabaseBundleId(String relationalDatabaseBundleId) {
            this.relationalDatabaseBundleId = relationalDatabaseBundleId;
        }

        public final String getMasterDatabaseName() {
            return masterDatabaseName;
        }

        @Override
        public final Builder masterDatabaseName(String masterDatabaseName) {
            this.masterDatabaseName = masterDatabaseName;
            return this;
        }

        public final void setMasterDatabaseName(String masterDatabaseName) {
            this.masterDatabaseName = masterDatabaseName;
        }

        public final RelationalDatabaseHardware.Builder getHardware() {
            return hardware != null ? hardware.toBuilder() : null;
        }

        @Override
        public final Builder hardware(RelationalDatabaseHardware hardware) {
            this.hardware = hardware;
            return this;
        }

        public final void setHardware(RelationalDatabaseHardware.BuilderImpl hardware) {
            this.hardware = hardware != null ? hardware.build() : null;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getSecondaryAvailabilityZone() {
            return secondaryAvailabilityZone;
        }

        @Override
        public final Builder secondaryAvailabilityZone(String secondaryAvailabilityZone) {
            this.secondaryAvailabilityZone = secondaryAvailabilityZone;
            return this;
        }

        public final void setSecondaryAvailabilityZone(String secondaryAvailabilityZone) {
            this.secondaryAvailabilityZone = secondaryAvailabilityZone;
        }

        public final Boolean getBackupRetentionEnabled() {
            return backupRetentionEnabled;
        }

        @Override
        public final Builder backupRetentionEnabled(Boolean backupRetentionEnabled) {
            this.backupRetentionEnabled = backupRetentionEnabled;
            return this;
        }

        public final void setBackupRetentionEnabled(Boolean backupRetentionEnabled) {
            this.backupRetentionEnabled = backupRetentionEnabled;
        }

        public final PendingModifiedRelationalDatabaseValues.Builder getPendingModifiedValues() {
            return pendingModifiedValues != null ? pendingModifiedValues.toBuilder() : null;
        }

        @Override
        public final Builder pendingModifiedValues(PendingModifiedRelationalDatabaseValues pendingModifiedValues) {
            this.pendingModifiedValues = pendingModifiedValues;
            return this;
        }

        public final void setPendingModifiedValues(PendingModifiedRelationalDatabaseValues.BuilderImpl pendingModifiedValues) {
            this.pendingModifiedValues = pendingModifiedValues != null ? pendingModifiedValues.build() : null;
        }

        public final String getEngine() {
            return engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final Instant getLatestRestorableTime() {
            return latestRestorableTime;
        }

        @Override
        public final Builder latestRestorableTime(Instant latestRestorableTime) {
            this.latestRestorableTime = latestRestorableTime;
            return this;
        }

        public final void setLatestRestorableTime(Instant latestRestorableTime) {
            this.latestRestorableTime = latestRestorableTime;
        }

        public final String getMasterUsername() {
            return masterUsername;
        }

        @Override
        public final Builder masterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
            return this;
        }

        public final void setMasterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
        }

        public final String getParameterApplyStatus() {
            return parameterApplyStatus;
        }

        @Override
        public final Builder parameterApplyStatus(String parameterApplyStatus) {
            this.parameterApplyStatus = parameterApplyStatus;
            return this;
        }

        public final void setParameterApplyStatus(String parameterApplyStatus) {
            this.parameterApplyStatus = parameterApplyStatus;
        }

        public final String getPreferredBackupWindow() {
            return preferredBackupWindow;
        }

        @Override
        public final Builder preferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
            return this;
        }

        public final void setPreferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
        }

        public final String getPreferredMaintenanceWindow() {
            return preferredMaintenanceWindow;
        }

        @Override
        public final Builder preferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
            return this;
        }

        public final void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        }

        public final Boolean getPubliclyAccessible() {
            return publiclyAccessible;
        }

        @Override
        public final Builder publiclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
            return this;
        }

        public final void setPubliclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
        }

        public final RelationalDatabaseEndpoint.Builder getMasterEndpoint() {
            return masterEndpoint != null ? masterEndpoint.toBuilder() : null;
        }

        @Override
        public final Builder masterEndpoint(RelationalDatabaseEndpoint masterEndpoint) {
            this.masterEndpoint = masterEndpoint;
            return this;
        }

        public final void setMasterEndpoint(RelationalDatabaseEndpoint.BuilderImpl masterEndpoint) {
            this.masterEndpoint = masterEndpoint != null ? masterEndpoint.build() : null;
        }

        public final List<PendingMaintenanceAction.Builder> getPendingMaintenanceActions() {
            List<PendingMaintenanceAction.Builder> result = PendingMaintenanceActionListCopier
                    .copyToBuilder(this.pendingMaintenanceActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder pendingMaintenanceActions(Collection<PendingMaintenanceAction> pendingMaintenanceActions) {
            this.pendingMaintenanceActions = PendingMaintenanceActionListCopier.copy(pendingMaintenanceActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder pendingMaintenanceActions(PendingMaintenanceAction... pendingMaintenanceActions) {
            pendingMaintenanceActions(Arrays.asList(pendingMaintenanceActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder pendingMaintenanceActions(Consumer<PendingMaintenanceAction.Builder>... pendingMaintenanceActions) {
            pendingMaintenanceActions(Stream.of(pendingMaintenanceActions)
                    .map(c -> PendingMaintenanceAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPendingMaintenanceActions(Collection<PendingMaintenanceAction.BuilderImpl> pendingMaintenanceActions) {
            this.pendingMaintenanceActions = PendingMaintenanceActionListCopier.copyFromBuilder(pendingMaintenanceActions);
        }

        public final String getCaCertificateIdentifier() {
            return caCertificateIdentifier;
        }

        @Override
        public final Builder caCertificateIdentifier(String caCertificateIdentifier) {
            this.caCertificateIdentifier = caCertificateIdentifier;
            return this;
        }

        public final void setCaCertificateIdentifier(String caCertificateIdentifier) {
            this.caCertificateIdentifier = caCertificateIdentifier;
        }

        @Override
        public RelationalDatabase build() {
            return new RelationalDatabase(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
