/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon Lightsail container service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerService implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerService.Builder, ContainerService> {
    private static final SdkField<String> CONTAINER_SERVICE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerServiceName").getter(getter(ContainerService::containerServiceName))
            .setter(setter(Builder::containerServiceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerServiceName").build())
            .build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ContainerService::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(ContainerService::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).memberName("location")
            .getter(getter(ContainerService::location)).setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(ContainerService::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(ContainerService::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POWER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("power")
            .getter(getter(ContainerService::powerAsString)).setter(setter(Builder::power))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("power").build()).build();

    private static final SdkField<String> POWER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("powerId").getter(getter(ContainerService::powerId)).setter(setter(Builder::powerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("powerId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(ContainerService::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<ContainerServiceStateDetail> STATE_DETAIL_FIELD = SdkField
            .<ContainerServiceStateDetail> builder(MarshallingType.SDK_POJO).memberName("stateDetail")
            .getter(getter(ContainerService::stateDetail)).setter(setter(Builder::stateDetail))
            .constructor(ContainerServiceStateDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateDetail").build()).build();

    private static final SdkField<Integer> SCALE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("scale")
            .getter(getter(ContainerService::scale)).setter(setter(Builder::scale))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scale").build()).build();

    private static final SdkField<ContainerServiceDeployment> CURRENT_DEPLOYMENT_FIELD = SdkField
            .<ContainerServiceDeployment> builder(MarshallingType.SDK_POJO).memberName("currentDeployment")
            .getter(getter(ContainerService::currentDeployment)).setter(setter(Builder::currentDeployment))
            .constructor(ContainerServiceDeployment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentDeployment").build()).build();

    private static final SdkField<ContainerServiceDeployment> NEXT_DEPLOYMENT_FIELD = SdkField
            .<ContainerServiceDeployment> builder(MarshallingType.SDK_POJO).memberName("nextDeployment")
            .getter(getter(ContainerService::nextDeployment)).setter(setter(Builder::nextDeployment))
            .constructor(ContainerServiceDeployment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nextDeployment").build()).build();

    private static final SdkField<Boolean> IS_DISABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isDisabled").getter(getter(ContainerService::isDisabled)).setter(setter(Builder::isDisabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isDisabled").build()).build();

    private static final SdkField<String> PRINCIPAL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("principalArn").getter(getter(ContainerService::principalArn)).setter(setter(Builder::principalArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("principalArn").build()).build();

    private static final SdkField<String> PRIVATE_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("privateDomainName").getter(getter(ContainerService::privateDomainName))
            .setter(setter(Builder::privateDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privateDomainName").build()).build();

    private static final SdkField<Map<String, List<String>>> PUBLIC_DOMAIN_NAMES_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("publicDomainNames")
            .getter(getter(ContainerService::publicDomainNames))
            .setter(setter(Builder::publicDomainNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicDomainNames").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("url")
            .getter(getter(ContainerService::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_SERVICE_NAME_FIELD,
            ARN_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD, POWER_FIELD, POWER_ID_FIELD,
            STATE_FIELD, STATE_DETAIL_FIELD, SCALE_FIELD, CURRENT_DEPLOYMENT_FIELD, NEXT_DEPLOYMENT_FIELD, IS_DISABLED_FIELD,
            PRINCIPAL_ARN_FIELD, PRIVATE_DOMAIN_NAME_FIELD, PUBLIC_DOMAIN_NAMES_FIELD, URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerServiceName;

    private final String arn;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final String power;

    private final String powerId;

    private final String state;

    private final ContainerServiceStateDetail stateDetail;

    private final Integer scale;

    private final ContainerServiceDeployment currentDeployment;

    private final ContainerServiceDeployment nextDeployment;

    private final Boolean isDisabled;

    private final String principalArn;

    private final String privateDomainName;

    private final Map<String, List<String>> publicDomainNames;

    private final String url;

    private ContainerService(BuilderImpl builder) {
        this.containerServiceName = builder.containerServiceName;
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.power = builder.power;
        this.powerId = builder.powerId;
        this.state = builder.state;
        this.stateDetail = builder.stateDetail;
        this.scale = builder.scale;
        this.currentDeployment = builder.currentDeployment;
        this.nextDeployment = builder.nextDeployment;
        this.isDisabled = builder.isDisabled;
        this.principalArn = builder.principalArn;
        this.privateDomainName = builder.privateDomainName;
        this.publicDomainNames = builder.publicDomainNames;
        this.url = builder.url;
    }

    /**
     * <p>
     * The name of the container service.
     * </p>
     * 
     * @return The name of the container service.
     */
    public final String containerServiceName() {
        return containerServiceName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container service.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container service.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The timestamp when the container service was created.
     * </p>
     * 
     * @return The timestamp when the container service was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * An object that describes the location of the container service, such as the AWS Region and Availability Zone.
     * </p>
     * 
     * @return An object that describes the location of the container service, such as the AWS Region and Availability
     *         Zone.
     */
    public final ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail Developer
     * Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
     *         Developer Guide</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The power specification of the container service.
     * </p>
     * <p>
     * The power specifies the amount of RAM, the number of vCPUs, and the base price of the container service.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #power} will return
     * {@link ContainerServicePowerName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #powerAsString}.
     * </p>
     * 
     * @return The power specification of the container service.</p>
     *         <p>
     *         The power specifies the amount of RAM, the number of vCPUs, and the base price of the container service.
     * @see ContainerServicePowerName
     */
    public final ContainerServicePowerName power() {
        return ContainerServicePowerName.fromValue(power);
    }

    /**
     * <p>
     * The power specification of the container service.
     * </p>
     * <p>
     * The power specifies the amount of RAM, the number of vCPUs, and the base price of the container service.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #power} will return
     * {@link ContainerServicePowerName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #powerAsString}.
     * </p>
     * 
     * @return The power specification of the container service.</p>
     *         <p>
     *         The power specifies the amount of RAM, the number of vCPUs, and the base price of the container service.
     * @see ContainerServicePowerName
     */
    public final String powerAsString() {
        return power;
    }

    /**
     * <p>
     * The ID of the power of the container service.
     * </p>
     * 
     * @return The ID of the power of the container service.
     */
    public final String powerId() {
        return powerId;
    }

    /**
     * <p>
     * The current state of the container service.
     * </p>
     * <p>
     * The following container service states are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> - The container service is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READY</code> - The container service is running but it does not have an active container deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEPLOYING</code> - The container service is launching a container deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code> - The container service is running and it has an active container deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The container service is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> - The container service is disabled, and its active deployment and containers, if any, are
     * shut down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ContainerServiceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the container service.</p>
     *         <p>
     *         The following container service states are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - The container service is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READY</code> - The container service is running but it does not have an active container
     *         deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEPLOYING</code> - The container service is launching a container deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code> - The container service is running and it has an active container deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The container service is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> - The container service is disabled, and its active deployment and containers, if
     *         any, are shut down.
     *         </p>
     *         </li>
     * @see ContainerServiceState
     */
    public final ContainerServiceState state() {
        return ContainerServiceState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the container service.
     * </p>
     * <p>
     * The following container service states are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> - The container service is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READY</code> - The container service is running but it does not have an active container deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEPLOYING</code> - The container service is launching a container deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code> - The container service is running and it has an active container deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The container service is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> - The container service is disabled, and its active deployment and containers, if any, are
     * shut down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ContainerServiceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the container service.</p>
     *         <p>
     *         The following container service states are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - The container service is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READY</code> - The container service is running but it does not have an active container
     *         deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEPLOYING</code> - The container service is launching a container deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code> - The container service is running and it has an active container deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The container service is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> - The container service is disabled, and its active deployment and containers, if
     *         any, are shut down.
     *         </p>
     *         </li>
     * @see ContainerServiceState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * An object that describes the current state of the container service.
     * </p>
     * <note>
     * <p>
     * The state detail is populated only when a container service is in a <code>PENDING</code>, <code>DEPLOYING</code>,
     * or <code>UPDATING</code> state.
     * </p>
     * </note>
     * 
     * @return An object that describes the current state of the container service.</p> <note>
     *         <p>
     *         The state detail is populated only when a container service is in a <code>PENDING</code>,
     *         <code>DEPLOYING</code>, or <code>UPDATING</code> state.
     *         </p>
     */
    public final ContainerServiceStateDetail stateDetail() {
        return stateDetail;
    }

    /**
     * <p>
     * The scale specification of the container service.
     * </p>
     * <p>
     * The scale specifies the allocated compute nodes of the container service.
     * </p>
     * 
     * @return The scale specification of the container service.</p>
     *         <p>
     *         The scale specifies the allocated compute nodes of the container service.
     */
    public final Integer scale() {
        return scale;
    }

    /**
     * <p>
     * An object that describes the current container deployment of the container service.
     * </p>
     * 
     * @return An object that describes the current container deployment of the container service.
     */
    public final ContainerServiceDeployment currentDeployment() {
        return currentDeployment;
    }

    /**
     * <p>
     * An object that describes the next deployment of the container service.
     * </p>
     * <p>
     * This value is <code>null</code> when there is no deployment in a <code>pending</code> state.
     * </p>
     * 
     * @return An object that describes the next deployment of the container service.</p>
     *         <p>
     *         This value is <code>null</code> when there is no deployment in a <code>pending</code> state.
     */
    public final ContainerServiceDeployment nextDeployment() {
        return nextDeployment;
    }

    /**
     * <p>
     * A Boolean value indicating whether the container service is disabled.
     * </p>
     * 
     * @return A Boolean value indicating whether the container service is disabled.
     */
    public final Boolean isDisabled() {
        return isDisabled;
    }

    /**
     * <p>
     * The principal ARN of the container service.
     * </p>
     * <p>
     * The principal ARN can be used to create a trust relationship between your standard AWS account and your Lightsail
     * container service. This allows you to give your service permission to access resources in your standard AWS
     * account.
     * </p>
     * 
     * @return The principal ARN of the container service.</p>
     *         <p>
     *         The principal ARN can be used to create a trust relationship between your standard AWS account and your
     *         Lightsail container service. This allows you to give your service permission to access resources in your
     *         standard AWS account.
     */
    public final String principalArn() {
        return principalArn;
    }

    /**
     * <p>
     * The private domain name of the container service.
     * </p>
     * <p>
     * The private domain name is accessible only by other resources within the default virtual private cloud (VPC) of
     * your Lightsail account.
     * </p>
     * 
     * @return The private domain name of the container service.</p>
     *         <p>
     *         The private domain name is accessible only by other resources within the default virtual private cloud
     *         (VPC) of your Lightsail account.
     */
    public final String privateDomainName() {
        return privateDomainName;
    }

    /**
     * Returns true if the PublicDomainNames property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasPublicDomainNames() {
        return publicDomainNames != null && !(publicDomainNames instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The public domain name of the container service, such as <code>example.com</code> and
     * <code>www.example.com</code>.
     * </p>
     * <p>
     * You can specify up to four public domain names for a container service. The domain names that you specify are
     * used when you create a deployment with a container configured as the public endpoint of your container service.
     * </p>
     * <p>
     * If you don't specify public domain names, then you can use the default domain of the container service.
     * </p>
     * <important>
     * <p>
     * You must create and validate an SSL/TLS certificate before you can use public domain names with your container
     * service. Use the <code>CreateCertificate</code> action to create a certificate for the public domain names you
     * want to use with your container service.
     * </p>
     * </important>
     * <p>
     * See <code>CreateContainerService</code> or <code>UpdateContainerService</code> for information about how to
     * specify public domain names for your Lightsail container service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPublicDomainNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The public domain name of the container service, such as <code>example.com</code> and
     *         <code>www.example.com</code>.</p>
     *         <p>
     *         You can specify up to four public domain names for a container service. The domain names that you specify
     *         are used when you create a deployment with a container configured as the public endpoint of your
     *         container service.
     *         </p>
     *         <p>
     *         If you don't specify public domain names, then you can use the default domain of the container service.
     *         </p>
     *         <important>
     *         <p>
     *         You must create and validate an SSL/TLS certificate before you can use public domain names with your
     *         container service. Use the <code>CreateCertificate</code> action to create a certificate for the public
     *         domain names you want to use with your container service.
     *         </p>
     *         </important>
     *         <p>
     *         See <code>CreateContainerService</code> or <code>UpdateContainerService</code> for information about how
     *         to specify public domain names for your Lightsail container service.
     */
    public final Map<String, List<String>> publicDomainNames() {
        return publicDomainNames;
    }

    /**
     * <p>
     * The publicly accessible URL of the container service.
     * </p>
     * <p>
     * If no public endpoint is specified in the <code>currentDeployment</code>, this URL returns a 404 response.
     * </p>
     * 
     * @return The publicly accessible URL of the container service.</p>
     *         <p>
     *         If no public endpoint is specified in the <code>currentDeployment</code>, this URL returns a 404
     *         response.
     */
    public final String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerServiceName());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(powerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(powerId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateDetail());
        hashCode = 31 * hashCode + Objects.hashCode(scale());
        hashCode = 31 * hashCode + Objects.hashCode(currentDeployment());
        hashCode = 31 * hashCode + Objects.hashCode(nextDeployment());
        hashCode = 31 * hashCode + Objects.hashCode(isDisabled());
        hashCode = 31 * hashCode + Objects.hashCode(principalArn());
        hashCode = 31 * hashCode + Objects.hashCode(privateDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(hasPublicDomainNames() ? publicDomainNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerService)) {
            return false;
        }
        ContainerService other = (ContainerService) obj;
        return Objects.equals(containerServiceName(), other.containerServiceName()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(powerAsString(), other.powerAsString())
                && Objects.equals(powerId(), other.powerId()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(stateDetail(), other.stateDetail()) && Objects.equals(scale(), other.scale())
                && Objects.equals(currentDeployment(), other.currentDeployment())
                && Objects.equals(nextDeployment(), other.nextDeployment()) && Objects.equals(isDisabled(), other.isDisabled())
                && Objects.equals(principalArn(), other.principalArn())
                && Objects.equals(privateDomainName(), other.privateDomainName())
                && hasPublicDomainNames() == other.hasPublicDomainNames()
                && Objects.equals(publicDomainNames(), other.publicDomainNames()) && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerService").add("ContainerServiceName", containerServiceName()).add("Arn", arn())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("Power", powerAsString()).add("PowerId", powerId())
                .add("State", stateAsString()).add("StateDetail", stateDetail()).add("Scale", scale())
                .add("CurrentDeployment", currentDeployment()).add("NextDeployment", nextDeployment())
                .add("IsDisabled", isDisabled()).add("PrincipalArn", principalArn())
                .add("PrivateDomainName", privateDomainName())
                .add("PublicDomainNames", hasPublicDomainNames() ? publicDomainNames() : null).add("Url", url()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerServiceName":
            return Optional.ofNullable(clazz.cast(containerServiceName()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "power":
            return Optional.ofNullable(clazz.cast(powerAsString()));
        case "powerId":
            return Optional.ofNullable(clazz.cast(powerId()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "stateDetail":
            return Optional.ofNullable(clazz.cast(stateDetail()));
        case "scale":
            return Optional.ofNullable(clazz.cast(scale()));
        case "currentDeployment":
            return Optional.ofNullable(clazz.cast(currentDeployment()));
        case "nextDeployment":
            return Optional.ofNullable(clazz.cast(nextDeployment()));
        case "isDisabled":
            return Optional.ofNullable(clazz.cast(isDisabled()));
        case "principalArn":
            return Optional.ofNullable(clazz.cast(principalArn()));
        case "privateDomainName":
            return Optional.ofNullable(clazz.cast(privateDomainName()));
        case "publicDomainNames":
            return Optional.ofNullable(clazz.cast(publicDomainNames()));
        case "url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerService, T> g) {
        return obj -> g.apply((ContainerService) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerService> {
        /**
         * <p>
         * The name of the container service.
         * </p>
         * 
         * @param containerServiceName
         *        The name of the container service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerServiceName(String containerServiceName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container service.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the container service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The timestamp when the container service was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the container service was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * An object that describes the location of the container service, such as the AWS Region and Availability Zone.
         * </p>
         * 
         * @param location
         *        An object that describes the location of the container service, such as the AWS Region and
         *        Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * An object that describes the location of the container service, such as the AWS Region and Availability Zone.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type of the container service (i.e., <code>ContainerService</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon
         *        Lightsail Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon
         *        Lightsail Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The power specification of the container service.
         * </p>
         * <p>
         * The power specifies the amount of RAM, the number of vCPUs, and the base price of the container service.
         * </p>
         * 
         * @param power
         *        The power specification of the container service.</p>
         *        <p>
         *        The power specifies the amount of RAM, the number of vCPUs, and the base price of the container
         *        service.
         * @see ContainerServicePowerName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerServicePowerName
         */
        Builder power(String power);

        /**
         * <p>
         * The power specification of the container service.
         * </p>
         * <p>
         * The power specifies the amount of RAM, the number of vCPUs, and the base price of the container service.
         * </p>
         * 
         * @param power
         *        The power specification of the container service.</p>
         *        <p>
         *        The power specifies the amount of RAM, the number of vCPUs, and the base price of the container
         *        service.
         * @see ContainerServicePowerName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerServicePowerName
         */
        Builder power(ContainerServicePowerName power);

        /**
         * <p>
         * The ID of the power of the container service.
         * </p>
         * 
         * @param powerId
         *        The ID of the power of the container service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder powerId(String powerId);

        /**
         * <p>
         * The current state of the container service.
         * </p>
         * <p>
         * The following container service states are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> - The container service is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READY</code> - The container service is running but it does not have an active container deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEPLOYING</code> - The container service is launching a container deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code> - The container service is running and it has an active container deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The container service is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> - The container service is disabled, and its active deployment and containers, if any,
         * are shut down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the container service.</p>
         *        <p>
         *        The following container service states are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - The container service is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READY</code> - The container service is running but it does not have an active container
         *        deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEPLOYING</code> - The container service is launching a container deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code> - The container service is running and it has an active container deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The container service is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> - The container service is disabled, and its active deployment and containers,
         *        if any, are shut down.
         *        </p>
         *        </li>
         * @see ContainerServiceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerServiceState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the container service.
         * </p>
         * <p>
         * The following container service states are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> - The container service is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READY</code> - The container service is running but it does not have an active container deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEPLOYING</code> - The container service is launching a container deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code> - The container service is running and it has an active container deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The container service is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> - The container service is disabled, and its active deployment and containers, if any,
         * are shut down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the container service.</p>
         *        <p>
         *        The following container service states are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - The container service is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READY</code> - The container service is running but it does not have an active container
         *        deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEPLOYING</code> - The container service is launching a container deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code> - The container service is running and it has an active container deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The container service capacity or its custom domains are being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The container service is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> - The container service is disabled, and its active deployment and containers,
         *        if any, are shut down.
         *        </p>
         *        </li>
         * @see ContainerServiceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerServiceState
         */
        Builder state(ContainerServiceState state);

        /**
         * <p>
         * An object that describes the current state of the container service.
         * </p>
         * <note>
         * <p>
         * The state detail is populated only when a container service is in a <code>PENDING</code>,
         * <code>DEPLOYING</code>, or <code>UPDATING</code> state.
         * </p>
         * </note>
         * 
         * @param stateDetail
         *        An object that describes the current state of the container service.</p> <note>
         *        <p>
         *        The state detail is populated only when a container service is in a <code>PENDING</code>,
         *        <code>DEPLOYING</code>, or <code>UPDATING</code> state.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateDetail(ContainerServiceStateDetail stateDetail);

        /**
         * <p>
         * An object that describes the current state of the container service.
         * </p>
         * <note>
         * <p>
         * The state detail is populated only when a container service is in a <code>PENDING</code>,
         * <code>DEPLOYING</code>, or <code>UPDATING</code> state.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link ContainerServiceStateDetail.Builder}
         * avoiding the need to create one manually via {@link ContainerServiceStateDetail#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContainerServiceStateDetail.Builder#build()} is called
         * immediately and its result is passed to {@link #stateDetail(ContainerServiceStateDetail)}.
         * 
         * @param stateDetail
         *        a consumer that will call methods on {@link ContainerServiceStateDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stateDetail(ContainerServiceStateDetail)
         */
        default Builder stateDetail(Consumer<ContainerServiceStateDetail.Builder> stateDetail) {
            return stateDetail(ContainerServiceStateDetail.builder().applyMutation(stateDetail).build());
        }

        /**
         * <p>
         * The scale specification of the container service.
         * </p>
         * <p>
         * The scale specifies the allocated compute nodes of the container service.
         * </p>
         * 
         * @param scale
         *        The scale specification of the container service.</p>
         *        <p>
         *        The scale specifies the allocated compute nodes of the container service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scale(Integer scale);

        /**
         * <p>
         * An object that describes the current container deployment of the container service.
         * </p>
         * 
         * @param currentDeployment
         *        An object that describes the current container deployment of the container service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentDeployment(ContainerServiceDeployment currentDeployment);

        /**
         * <p>
         * An object that describes the current container deployment of the container service.
         * </p>
         * This is a convenience that creates an instance of the {@link ContainerServiceDeployment.Builder} avoiding the
         * need to create one manually via {@link ContainerServiceDeployment#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContainerServiceDeployment.Builder#build()} is called immediately
         * and its result is passed to {@link #currentDeployment(ContainerServiceDeployment)}.
         * 
         * @param currentDeployment
         *        a consumer that will call methods on {@link ContainerServiceDeployment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentDeployment(ContainerServiceDeployment)
         */
        default Builder currentDeployment(Consumer<ContainerServiceDeployment.Builder> currentDeployment) {
            return currentDeployment(ContainerServiceDeployment.builder().applyMutation(currentDeployment).build());
        }

        /**
         * <p>
         * An object that describes the next deployment of the container service.
         * </p>
         * <p>
         * This value is <code>null</code> when there is no deployment in a <code>pending</code> state.
         * </p>
         * 
         * @param nextDeployment
         *        An object that describes the next deployment of the container service.</p>
         *        <p>
         *        This value is <code>null</code> when there is no deployment in a <code>pending</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextDeployment(ContainerServiceDeployment nextDeployment);

        /**
         * <p>
         * An object that describes the next deployment of the container service.
         * </p>
         * <p>
         * This value is <code>null</code> when there is no deployment in a <code>pending</code> state.
         * </p>
         * This is a convenience that creates an instance of the {@link ContainerServiceDeployment.Builder} avoiding the
         * need to create one manually via {@link ContainerServiceDeployment#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContainerServiceDeployment.Builder#build()} is called immediately
         * and its result is passed to {@link #nextDeployment(ContainerServiceDeployment)}.
         * 
         * @param nextDeployment
         *        a consumer that will call methods on {@link ContainerServiceDeployment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nextDeployment(ContainerServiceDeployment)
         */
        default Builder nextDeployment(Consumer<ContainerServiceDeployment.Builder> nextDeployment) {
            return nextDeployment(ContainerServiceDeployment.builder().applyMutation(nextDeployment).build());
        }

        /**
         * <p>
         * A Boolean value indicating whether the container service is disabled.
         * </p>
         * 
         * @param isDisabled
         *        A Boolean value indicating whether the container service is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDisabled(Boolean isDisabled);

        /**
         * <p>
         * The principal ARN of the container service.
         * </p>
         * <p>
         * The principal ARN can be used to create a trust relationship between your standard AWS account and your
         * Lightsail container service. This allows you to give your service permission to access resources in your
         * standard AWS account.
         * </p>
         * 
         * @param principalArn
         *        The principal ARN of the container service.</p>
         *        <p>
         *        The principal ARN can be used to create a trust relationship between your standard AWS account and
         *        your Lightsail container service. This allows you to give your service permission to access resources
         *        in your standard AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principalArn(String principalArn);

        /**
         * <p>
         * The private domain name of the container service.
         * </p>
         * <p>
         * The private domain name is accessible only by other resources within the default virtual private cloud (VPC)
         * of your Lightsail account.
         * </p>
         * 
         * @param privateDomainName
         *        The private domain name of the container service.</p>
         *        <p>
         *        The private domain name is accessible only by other resources within the default virtual private cloud
         *        (VPC) of your Lightsail account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateDomainName(String privateDomainName);

        /**
         * <p>
         * The public domain name of the container service, such as <code>example.com</code> and
         * <code>www.example.com</code>.
         * </p>
         * <p>
         * You can specify up to four public domain names for a container service. The domain names that you specify are
         * used when you create a deployment with a container configured as the public endpoint of your container
         * service.
         * </p>
         * <p>
         * If you don't specify public domain names, then you can use the default domain of the container service.
         * </p>
         * <important>
         * <p>
         * You must create and validate an SSL/TLS certificate before you can use public domain names with your
         * container service. Use the <code>CreateCertificate</code> action to create a certificate for the public
         * domain names you want to use with your container service.
         * </p>
         * </important>
         * <p>
         * See <code>CreateContainerService</code> or <code>UpdateContainerService</code> for information about how to
         * specify public domain names for your Lightsail container service.
         * </p>
         * 
         * @param publicDomainNames
         *        The public domain name of the container service, such as <code>example.com</code> and
         *        <code>www.example.com</code>.</p>
         *        <p>
         *        You can specify up to four public domain names for a container service. The domain names that you
         *        specify are used when you create a deployment with a container configured as the public endpoint of
         *        your container service.
         *        </p>
         *        <p>
         *        If you don't specify public domain names, then you can use the default domain of the container
         *        service.
         *        </p>
         *        <important>
         *        <p>
         *        You must create and validate an SSL/TLS certificate before you can use public domain names with your
         *        container service. Use the <code>CreateCertificate</code> action to create a certificate for the
         *        public domain names you want to use with your container service.
         *        </p>
         *        </important>
         *        <p>
         *        See <code>CreateContainerService</code> or <code>UpdateContainerService</code> for information about
         *        how to specify public domain names for your Lightsail container service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicDomainNames(Map<String, ? extends Collection<String>> publicDomainNames);

        /**
         * <p>
         * The publicly accessible URL of the container service.
         * </p>
         * <p>
         * If no public endpoint is specified in the <code>currentDeployment</code>, this URL returns a 404 response.
         * </p>
         * 
         * @param url
         *        The publicly accessible URL of the container service.</p>
         *        <p>
         *        If no public endpoint is specified in the <code>currentDeployment</code>, this URL returns a 404
         *        response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String containerServiceName;

        private String arn;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String power;

        private String powerId;

        private String state;

        private ContainerServiceStateDetail stateDetail;

        private Integer scale;

        private ContainerServiceDeployment currentDeployment;

        private ContainerServiceDeployment nextDeployment;

        private Boolean isDisabled;

        private String principalArn;

        private String privateDomainName;

        private Map<String, List<String>> publicDomainNames = DefaultSdkAutoConstructMap.getInstance();

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerService model) {
            containerServiceName(model.containerServiceName);
            arn(model.arn);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            power(model.power);
            powerId(model.powerId);
            state(model.state);
            stateDetail(model.stateDetail);
            scale(model.scale);
            currentDeployment(model.currentDeployment);
            nextDeployment(model.nextDeployment);
            isDisabled(model.isDisabled);
            principalArn(model.principalArn);
            privateDomainName(model.privateDomainName);
            publicDomainNames(model.publicDomainNames);
            url(model.url);
        }

        public final String getContainerServiceName() {
            return containerServiceName;
        }

        @Override
        public final Builder containerServiceName(String containerServiceName) {
            this.containerServiceName = containerServiceName;
            return this;
        }

        public final void setContainerServiceName(String containerServiceName) {
            this.containerServiceName = containerServiceName;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getPower() {
            return power;
        }

        @Override
        public final Builder power(String power) {
            this.power = power;
            return this;
        }

        @Override
        public final Builder power(ContainerServicePowerName power) {
            this.power(power == null ? null : power.toString());
            return this;
        }

        public final void setPower(String power) {
            this.power = power;
        }

        public final String getPowerId() {
            return powerId;
        }

        @Override
        public final Builder powerId(String powerId) {
            this.powerId = powerId;
            return this;
        }

        public final void setPowerId(String powerId) {
            this.powerId = powerId;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ContainerServiceState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final ContainerServiceStateDetail.Builder getStateDetail() {
            return stateDetail != null ? stateDetail.toBuilder() : null;
        }

        @Override
        public final Builder stateDetail(ContainerServiceStateDetail stateDetail) {
            this.stateDetail = stateDetail;
            return this;
        }

        public final void setStateDetail(ContainerServiceStateDetail.BuilderImpl stateDetail) {
            this.stateDetail = stateDetail != null ? stateDetail.build() : null;
        }

        public final Integer getScale() {
            return scale;
        }

        @Override
        public final Builder scale(Integer scale) {
            this.scale = scale;
            return this;
        }

        public final void setScale(Integer scale) {
            this.scale = scale;
        }

        public final ContainerServiceDeployment.Builder getCurrentDeployment() {
            return currentDeployment != null ? currentDeployment.toBuilder() : null;
        }

        @Override
        public final Builder currentDeployment(ContainerServiceDeployment currentDeployment) {
            this.currentDeployment = currentDeployment;
            return this;
        }

        public final void setCurrentDeployment(ContainerServiceDeployment.BuilderImpl currentDeployment) {
            this.currentDeployment = currentDeployment != null ? currentDeployment.build() : null;
        }

        public final ContainerServiceDeployment.Builder getNextDeployment() {
            return nextDeployment != null ? nextDeployment.toBuilder() : null;
        }

        @Override
        public final Builder nextDeployment(ContainerServiceDeployment nextDeployment) {
            this.nextDeployment = nextDeployment;
            return this;
        }

        public final void setNextDeployment(ContainerServiceDeployment.BuilderImpl nextDeployment) {
            this.nextDeployment = nextDeployment != null ? nextDeployment.build() : null;
        }

        public final Boolean getIsDisabled() {
            return isDisabled;
        }

        @Override
        public final Builder isDisabled(Boolean isDisabled) {
            this.isDisabled = isDisabled;
            return this;
        }

        public final void setIsDisabled(Boolean isDisabled) {
            this.isDisabled = isDisabled;
        }

        public final String getPrincipalArn() {
            return principalArn;
        }

        @Override
        public final Builder principalArn(String principalArn) {
            this.principalArn = principalArn;
            return this;
        }

        public final void setPrincipalArn(String principalArn) {
            this.principalArn = principalArn;
        }

        public final String getPrivateDomainName() {
            return privateDomainName;
        }

        @Override
        public final Builder privateDomainName(String privateDomainName) {
            this.privateDomainName = privateDomainName;
            return this;
        }

        public final void setPrivateDomainName(String privateDomainName) {
            this.privateDomainName = privateDomainName;
        }

        public final Map<String, ? extends Collection<String>> getPublicDomainNames() {
            if (publicDomainNames instanceof SdkAutoConstructMap) {
                return null;
            }
            return publicDomainNames;
        }

        @Override
        public final Builder publicDomainNames(Map<String, ? extends Collection<String>> publicDomainNames) {
            this.publicDomainNames = ContainerServicePublicDomainsCopier.copy(publicDomainNames);
            return this;
        }

        public final void setPublicDomainNames(Map<String, ? extends Collection<String>> publicDomainNames) {
            this.publicDomainNames = ContainerServicePublicDomainsCopier.copy(publicDomainNames);
        }

        public final String getUrl() {
            return url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public ContainerService build() {
            return new ContainerService(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
