/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the per-path cache behavior of an Amazon Lightsail content delivery network (CDN) distribution.
 * </p>
 * <p>
 * A per-path cache behavior is used to override, or add an exception to, the default cache behavior of a distribution.
 * For example, if the <code>cacheBehavior</code> is set to <code>cache</code>, then a per-path cache behavior can be
 * used to specify a directory, file, or file type that your distribution will cache. Alternately, if the distribution's
 * <code>cacheBehavior</code> is <code>dont-cache</code>, then a per-path cache behavior can be used to specify a
 * directory, file, or file type that your distribution will not cache.
 * </p>
 * <p>
 * if the cacheBehavior's behavior is set to 'cache', then
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CacheBehaviorPerPath implements SdkPojo, Serializable,
        ToCopyableBuilder<CacheBehaviorPerPath.Builder, CacheBehaviorPerPath> {
    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("path")
            .getter(getter(CacheBehaviorPerPath::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("path").build()).build();

    private static final SdkField<String> BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("behavior").getter(getter(CacheBehaviorPerPath::behaviorAsString)).setter(setter(Builder::behavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("behavior").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATH_FIELD, BEHAVIOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final String path;

    private final String behavior;

    private CacheBehaviorPerPath(BuilderImpl builder) {
        this.path = builder.path;
        this.behavior = builder.behavior;
    }

    /**
     * <p>
     * The path to a directory or file to cached, or not cache. Use an asterisk symbol to specify wildcard directories (
     * <code>path/to/assets/*</code>), and file types (<code>*.html, *jpg, *js</code>). Directories and file paths are
     * case-sensitive.
     * </p>
     * <p>
     * Examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify the following to cache all files in the document root of an Apache web server running on a Lightsail
     * instance.
     * </p>
     * <p>
     * <code>var/www/html/</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify the following file to cache only the index page in the document root of an Apache web server.
     * </p>
     * <p>
     * <code>var/www/html/index.html</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify the following to cache only the .html files in the document root of an Apache web server.
     * </p>
     * <p>
     * <code>var/www/html/*.html</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify the following to cache only the .jpg, .png, and .gif files in the images sub-directory of the document
     * root of an Apache web server.
     * </p>
     * <p>
     * <code>var/www/html/images/*.jpg</code>
     * </p>
     * <p>
     * <code>var/www/html/images/*.png</code>
     * </p>
     * <p>
     * <code>var/www/html/images/*.gif</code>
     * </p>
     * <p>
     * Specify the following to cache all files in the images sub-directory of the document root of an Apache web
     * server.
     * </p>
     * <p>
     * <code>var/www/html/images/</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The path to a directory or file to cached, or not cache. Use an asterisk symbol to specify wildcard
     *         directories (<code>path/to/assets/*</code>), and file types (<code>*.html, *jpg, *js</code>). Directories
     *         and file paths are case-sensitive.</p>
     *         <p>
     *         Examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify the following to cache all files in the document root of an Apache web server running on a
     *         Lightsail instance.
     *         </p>
     *         <p>
     *         <code>var/www/html/</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify the following file to cache only the index page in the document root of an Apache web server.
     *         </p>
     *         <p>
     *         <code>var/www/html/index.html</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify the following to cache only the .html files in the document root of an Apache web server.
     *         </p>
     *         <p>
     *         <code>var/www/html/*.html</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify the following to cache only the .jpg, .png, and .gif files in the images sub-directory of the
     *         document root of an Apache web server.
     *         </p>
     *         <p>
     *         <code>var/www/html/images/*.jpg</code>
     *         </p>
     *         <p>
     *         <code>var/www/html/images/*.png</code>
     *         </p>
     *         <p>
     *         <code>var/www/html/images/*.gif</code>
     *         </p>
     *         <p>
     *         Specify the following to cache all files in the images sub-directory of the document root of an Apache
     *         web server.
     *         </p>
     *         <p>
     *         <code>var/www/html/images/</code>
     *         </p>
     *         </li>
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * The cache behavior for the specified path.
     * </p>
     * <p>
     * You can specify one of the following per-path cache behaviors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>cache</code> </b> - This behavior caches the specified path.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #behavior} will
     * return {@link BehaviorEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #behaviorAsString}.
     * </p>
     * 
     * @return The cache behavior for the specified path.</p>
     *         <p>
     *         You can specify one of the following per-path cache behaviors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>cache</code> </b> - This behavior caches the specified path.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
     *         </p>
     *         </li>
     * @see BehaviorEnum
     */
    public final BehaviorEnum behavior() {
        return BehaviorEnum.fromValue(behavior);
    }

    /**
     * <p>
     * The cache behavior for the specified path.
     * </p>
     * <p>
     * You can specify one of the following per-path cache behaviors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>cache</code> </b> - This behavior caches the specified path.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #behavior} will
     * return {@link BehaviorEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #behaviorAsString}.
     * </p>
     * 
     * @return The cache behavior for the specified path.</p>
     *         <p>
     *         You can specify one of the following per-path cache behaviors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>cache</code> </b> - This behavior caches the specified path.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
     *         </p>
     *         </li>
     * @see BehaviorEnum
     */
    public final String behaviorAsString() {
        return behavior;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(behaviorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CacheBehaviorPerPath)) {
            return false;
        }
        CacheBehaviorPerPath other = (CacheBehaviorPerPath) obj;
        return Objects.equals(path(), other.path()) && Objects.equals(behaviorAsString(), other.behaviorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CacheBehaviorPerPath").add("Path", path()).add("Behavior", behaviorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "path":
            return Optional.ofNullable(clazz.cast(path()));
        case "behavior":
            return Optional.ofNullable(clazz.cast(behaviorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CacheBehaviorPerPath, T> g) {
        return obj -> g.apply((CacheBehaviorPerPath) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CacheBehaviorPerPath> {
        /**
         * <p>
         * The path to a directory or file to cached, or not cache. Use an asterisk symbol to specify wildcard
         * directories (<code>path/to/assets/*</code>), and file types (<code>*.html, *jpg, *js</code>). Directories and
         * file paths are case-sensitive.
         * </p>
         * <p>
         * Examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify the following to cache all files in the document root of an Apache web server running on a Lightsail
         * instance.
         * </p>
         * <p>
         * <code>var/www/html/</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify the following file to cache only the index page in the document root of an Apache web server.
         * </p>
         * <p>
         * <code>var/www/html/index.html</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify the following to cache only the .html files in the document root of an Apache web server.
         * </p>
         * <p>
         * <code>var/www/html/*.html</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify the following to cache only the .jpg, .png, and .gif files in the images sub-directory of the
         * document root of an Apache web server.
         * </p>
         * <p>
         * <code>var/www/html/images/*.jpg</code>
         * </p>
         * <p>
         * <code>var/www/html/images/*.png</code>
         * </p>
         * <p>
         * <code>var/www/html/images/*.gif</code>
         * </p>
         * <p>
         * Specify the following to cache all files in the images sub-directory of the document root of an Apache web
         * server.
         * </p>
         * <p>
         * <code>var/www/html/images/</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param path
         *        The path to a directory or file to cached, or not cache. Use an asterisk symbol to specify wildcard
         *        directories (<code>path/to/assets/*</code>), and file types (<code>*.html, *jpg, *js</code>).
         *        Directories and file paths are case-sensitive.</p>
         *        <p>
         *        Examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify the following to cache all files in the document root of an Apache web server running on a
         *        Lightsail instance.
         *        </p>
         *        <p>
         *        <code>var/www/html/</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify the following file to cache only the index page in the document root of an Apache web server.
         *        </p>
         *        <p>
         *        <code>var/www/html/index.html</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify the following to cache only the .html files in the document root of an Apache web server.
         *        </p>
         *        <p>
         *        <code>var/www/html/*.html</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify the following to cache only the .jpg, .png, and .gif files in the images sub-directory of the
         *        document root of an Apache web server.
         *        </p>
         *        <p>
         *        <code>var/www/html/images/*.jpg</code>
         *        </p>
         *        <p>
         *        <code>var/www/html/images/*.png</code>
         *        </p>
         *        <p>
         *        <code>var/www/html/images/*.gif</code>
         *        </p>
         *        <p>
         *        Specify the following to cache all files in the images sub-directory of the document root of an Apache
         *        web server.
         *        </p>
         *        <p>
         *        <code>var/www/html/images/</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * The cache behavior for the specified path.
         * </p>
         * <p>
         * You can specify one of the following per-path cache behaviors:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>cache</code> </b> - This behavior caches the specified path.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
         * </p>
         * </li>
         * </ul>
         * 
         * @param behavior
         *        The cache behavior for the specified path.</p>
         *        <p>
         *        You can specify one of the following per-path cache behaviors:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>cache</code> </b> - This behavior caches the specified path.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
         *        </p>
         *        </li>
         * @see BehaviorEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BehaviorEnum
         */
        Builder behavior(String behavior);

        /**
         * <p>
         * The cache behavior for the specified path.
         * </p>
         * <p>
         * You can specify one of the following per-path cache behaviors:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>cache</code> </b> - This behavior caches the specified path.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
         * </p>
         * </li>
         * </ul>
         * 
         * @param behavior
         *        The cache behavior for the specified path.</p>
         *        <p>
         *        You can specify one of the following per-path cache behaviors:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>cache</code> </b> - This behavior caches the specified path.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>dont-cache</code> </b> - This behavior doesn't cache the specified path.
         *        </p>
         *        </li>
         * @see BehaviorEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BehaviorEnum
         */
        Builder behavior(BehaviorEnum behavior);
    }

    static final class BuilderImpl implements Builder {
        private String path;

        private String behavior;

        private BuilderImpl() {
        }

        private BuilderImpl(CacheBehaviorPerPath model) {
            path(model.path);
            behavior(model.behavior);
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final String getBehavior() {
            return behavior;
        }

        @Override
        public final Builder behavior(String behavior) {
            this.behavior = behavior;
            return this;
        }

        @Override
        public final Builder behavior(BehaviorEnum behavior) {
            this.behavior(behavior == null ? null : behavior.toString());
            return this;
        }

        public final void setBehavior(String behavior) {
            this.behavior = behavior;
        }

        @Override
        public CacheBehaviorPerPath build() {
            return new CacheBehaviorPerPath(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
