/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetBucketMetricDataRequest extends LightsailRequest implements
        ToCopyableBuilder<GetBucketMetricDataRequest.Builder, GetBucketMetricDataRequest> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketName").getter(getter(GetBucketMetricDataRequest::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketName").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricName").getter(getter(GetBucketMetricDataRequest::metricNameAsString))
            .setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricName").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(GetBucketMetricDataRequest::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(GetBucketMetricDataRequest::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("period").getter(getter(GetBucketMetricDataRequest::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("period").build()).build();

    private static final SdkField<List<String>> STATISTICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("statistics")
            .getter(getter(GetBucketMetricDataRequest::statisticsAsStrings))
            .setter(setter(Builder::statisticsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statistics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("unit")
            .getter(getter(GetBucketMetricDataRequest::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            METRIC_NAME_FIELD, START_TIME_FIELD, END_TIME_FIELD, PERIOD_FIELD, STATISTICS_FIELD, UNIT_FIELD));

    private final String bucketName;

    private final String metricName;

    private final Instant startTime;

    private final Instant endTime;

    private final Integer period;

    private final List<String> statistics;

    private final String unit;

    private GetBucketMetricDataRequest(BuilderImpl builder) {
        super(builder);
        this.bucketName = builder.bucketName;
        this.metricName = builder.metricName;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.period = builder.period;
        this.statistics = builder.statistics;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The name of the bucket for which to get metric data.
     * </p>
     * 
     * @return The name of the bucket for which to get metric data.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The metric for which you want to return information.
     * </p>
     * <p>
     * Valid bucket metric names are listed below, along with the most useful statistics to include in your request, and
     * the published unit value.
     * </p>
     * <note>
     * <p>
     * These bucket metrics are reported once per day.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is calculated
     * by summing the size of all objects in the bucket (including object versions), including the size of all parts for
     * all incomplete multipart uploads to the bucket.
     * </p>
     * <p>
     * Statistics: The most useful statistic is <code>Maximum</code>.
     * </p>
     * <p>
     * Unit: The published unit is <code>Bytes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is calculated
     * by counting all objects in the bucket (including object versions) and the total number of parts for all
     * incomplete multipart uploads to the bucket.
     * </p>
     * <p>
     * Statistics: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * Unit: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link BucketMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The metric for which you want to return information.</p>
     *         <p>
     *         Valid bucket metric names are listed below, along with the most useful statistics to include in your
     *         request, and the published unit value.
     *         </p>
     *         <note>
     *         <p>
     *         These bucket metrics are reported once per day.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is
     *         calculated by summing the size of all objects in the bucket (including object versions), including the
     *         size of all parts for all incomplete multipart uploads to the bucket.
     *         </p>
     *         <p>
     *         Statistics: The most useful statistic is <code>Maximum</code>.
     *         </p>
     *         <p>
     *         Unit: The published unit is <code>Bytes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is
     *         calculated by counting all objects in the bucket (including object versions) and the total number of
     *         parts for all incomplete multipart uploads to the bucket.
     *         </p>
     *         <p>
     *         Statistics: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         Unit: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     * @see BucketMetricName
     */
    public final BucketMetricName metricName() {
        return BucketMetricName.fromValue(metricName);
    }

    /**
     * <p>
     * The metric for which you want to return information.
     * </p>
     * <p>
     * Valid bucket metric names are listed below, along with the most useful statistics to include in your request, and
     * the published unit value.
     * </p>
     * <note>
     * <p>
     * These bucket metrics are reported once per day.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is calculated
     * by summing the size of all objects in the bucket (including object versions), including the size of all parts for
     * all incomplete multipart uploads to the bucket.
     * </p>
     * <p>
     * Statistics: The most useful statistic is <code>Maximum</code>.
     * </p>
     * <p>
     * Unit: The published unit is <code>Bytes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is calculated
     * by counting all objects in the bucket (including object versions) and the total number of parts for all
     * incomplete multipart uploads to the bucket.
     * </p>
     * <p>
     * Statistics: The most useful statistic is <code>Average</code>.
     * </p>
     * <p>
     * Unit: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link BucketMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The metric for which you want to return information.</p>
     *         <p>
     *         Valid bucket metric names are listed below, along with the most useful statistics to include in your
     *         request, and the published unit value.
     *         </p>
     *         <note>
     *         <p>
     *         These bucket metrics are reported once per day.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is
     *         calculated by summing the size of all objects in the bucket (including object versions), including the
     *         size of all parts for all incomplete multipart uploads to the bucket.
     *         </p>
     *         <p>
     *         Statistics: The most useful statistic is <code>Maximum</code>.
     *         </p>
     *         <p>
     *         Unit: The published unit is <code>Bytes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is
     *         calculated by counting all objects in the bucket (including object versions) and the total number of
     *         parts for all incomplete multipart uploads to the bucket.
     *         </p>
     *         <p>
     *         Statistics: The most useful statistic is <code>Average</code>.
     *         </p>
     *         <p>
     *         Unit: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     * @see BucketMetricName
     */
    public final String metricNameAsString() {
        return metricName;
    }

    /**
     * <p>
     * The timestamp indicating the earliest data to be returned.
     * </p>
     * 
     * @return The timestamp indicating the earliest data to be returned.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The timestamp indicating the latest data to be returned.
     * </p>
     * 
     * @return The timestamp indicating the latest data to be returned.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The granularity, in seconds, of the returned data points.
     * </p>
     * <note>
     * <p>
     * Bucket storage metrics are reported once per day. Therefore, you should specify a period of 86400 seconds, which
     * is the number of seconds in a day.
     * </p>
     * </note>
     * 
     * @return The granularity, in seconds, of the returned data points.</p> <note>
     *         <p>
     *         Bucket storage metrics are reported once per day. Therefore, you should specify a period of 86400
     *         seconds, which is the number of seconds in a day.
     *         </p>
     */
    public final Integer period() {
        return period;
    }

    /**
     * <p>
     * The statistic for the metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified period. By
     * comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you can determine the
     * full scope of a metric and how close the average use is to the <code>Minimum</code> and <code>Maximum</code>
     * values. This comparison helps you to know when to increase or decrease your resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStatistics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The statistic for the metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
     *         determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified
     *         period. By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you
     *         can determine the full scope of a metric and how close the average use is to the <code>Minimum</code> and
     *         <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your
     *         resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     */
    public final List<MetricStatistic> statistics() {
        return MetricStatisticListCopier.copyStringToEnum(statistics);
    }

    /**
     * Returns true if the Statistics property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasStatistics() {
        return statistics != null && !(statistics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The statistic for the metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified period. By
     * comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you can determine the
     * full scope of a metric and how close the average use is to the <code>Minimum</code> and <code>Maximum</code>
     * values. This comparison helps you to know when to increase or decrease your resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStatistics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The statistic for the metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
     *         determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified
     *         period. By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you
     *         can determine the full scope of a metric and how close the average use is to the <code>Minimum</code> and
     *         <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your
     *         resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     */
    public final List<String> statisticsAsStrings() {
        return statistics;
    }

    /**
     * <p>
     * The unit for the metric data request.
     * </p>
     * <p>
     * Valid units depend on the metric data being requested. For the valid units with each available metric, see the
     * <code>metricName</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for the metric data request.</p>
     *         <p>
     *         Valid units depend on the metric data being requested. For the valid units with each available metric,
     *         see the <code>metricName</code> parameter.
     * @see MetricUnit
     */
    public final MetricUnit unit() {
        return MetricUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit for the metric data request.
     * </p>
     * <p>
     * Valid units depend on the metric data being requested. For the valid units with each available metric, see the
     * <code>metricName</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for the metric data request.</p>
     *         <p>
     *         Valid units depend on the metric data being requested. For the valid units with each available metric,
     *         see the <code>metricName</code> parameter.
     * @see MetricUnit
     */
    public final String unitAsString() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatistics() ? statisticsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetBucketMetricDataRequest)) {
            return false;
        }
        GetBucketMetricDataRequest other = (GetBucketMetricDataRequest) obj;
        return Objects.equals(bucketName(), other.bucketName())
                && Objects.equals(metricNameAsString(), other.metricNameAsString())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(period(), other.period()) && hasStatistics() == other.hasStatistics()
                && Objects.equals(statisticsAsStrings(), other.statisticsAsStrings())
                && Objects.equals(unitAsString(), other.unitAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetBucketMetricDataRequest").add("BucketName", bucketName())
                .add("MetricName", metricNameAsString()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Period", period()).add("Statistics", hasStatistics() ? statisticsAsStrings() : null)
                .add("Unit", unitAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "metricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "period":
            return Optional.ofNullable(clazz.cast(period()));
        case "statistics":
            return Optional.ofNullable(clazz.cast(statisticsAsStrings()));
        case "unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetBucketMetricDataRequest, T> g) {
        return obj -> g.apply((GetBucketMetricDataRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LightsailRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetBucketMetricDataRequest> {
        /**
         * <p>
         * The name of the bucket for which to get metric data.
         * </p>
         * 
         * @param bucketName
         *        The name of the bucket for which to get metric data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The metric for which you want to return information.
         * </p>
         * <p>
         * Valid bucket metric names are listed below, along with the most useful statistics to include in your request,
         * and the published unit value.
         * </p>
         * <note>
         * <p>
         * These bucket metrics are reported once per day.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is
         * calculated by summing the size of all objects in the bucket (including object versions), including the size
         * of all parts for all incomplete multipart uploads to the bucket.
         * </p>
         * <p>
         * Statistics: The most useful statistic is <code>Maximum</code>.
         * </p>
         * <p>
         * Unit: The published unit is <code>Bytes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is
         * calculated by counting all objects in the bucket (including object versions) and the total number of parts
         * for all incomplete multipart uploads to the bucket.
         * </p>
         * <p>
         * Statistics: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * Unit: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The metric for which you want to return information.</p>
         *        <p>
         *        Valid bucket metric names are listed below, along with the most useful statistics to include in your
         *        request, and the published unit value.
         *        </p>
         *        <note>
         *        <p>
         *        These bucket metrics are reported once per day.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is
         *        calculated by summing the size of all objects in the bucket (including object versions), including the
         *        size of all parts for all incomplete multipart uploads to the bucket.
         *        </p>
         *        <p>
         *        Statistics: The most useful statistic is <code>Maximum</code>.
         *        </p>
         *        <p>
         *        Unit: The published unit is <code>Bytes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is
         *        calculated by counting all objects in the bucket (including object versions) and the total number of
         *        parts for all incomplete multipart uploads to the bucket.
         *        </p>
         *        <p>
         *        Statistics: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        Unit: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         * @see BucketMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BucketMetricName
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The metric for which you want to return information.
         * </p>
         * <p>
         * Valid bucket metric names are listed below, along with the most useful statistics to include in your request,
         * and the published unit value.
         * </p>
         * <note>
         * <p>
         * These bucket metrics are reported once per day.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is
         * calculated by summing the size of all objects in the bucket (including object versions), including the size
         * of all parts for all incomplete multipart uploads to the bucket.
         * </p>
         * <p>
         * Statistics: The most useful statistic is <code>Maximum</code>.
         * </p>
         * <p>
         * Unit: The published unit is <code>Bytes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is
         * calculated by counting all objects in the bucket (including object versions) and the total number of parts
         * for all incomplete multipart uploads to the bucket.
         * </p>
         * <p>
         * Statistics: The most useful statistic is <code>Average</code>.
         * </p>
         * <p>
         * Unit: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The metric for which you want to return information.</p>
         *        <p>
         *        Valid bucket metric names are listed below, along with the most useful statistics to include in your
         *        request, and the published unit value.
         *        </p>
         *        <note>
         *        <p>
         *        These bucket metrics are reported once per day.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>BucketSizeBytes</code> </b> - The amount of data in bytes stored in a bucket. This value is
         *        calculated by summing the size of all objects in the bucket (including object versions), including the
         *        size of all parts for all incomplete multipart uploads to the bucket.
         *        </p>
         *        <p>
         *        Statistics: The most useful statistic is <code>Maximum</code>.
         *        </p>
         *        <p>
         *        Unit: The published unit is <code>Bytes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NumberOfObjects</code> </b> - The total number of objects stored in a bucket. This value is
         *        calculated by counting all objects in the bucket (including object versions) and the total number of
         *        parts for all incomplete multipart uploads to the bucket.
         *        </p>
         *        <p>
         *        Statistics: The most useful statistic is <code>Average</code>.
         *        </p>
         *        <p>
         *        Unit: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         * @see BucketMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BucketMetricName
         */
        Builder metricName(BucketMetricName metricName);

        /**
         * <p>
         * The timestamp indicating the earliest data to be returned.
         * </p>
         * 
         * @param startTime
         *        The timestamp indicating the earliest data to be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The timestamp indicating the latest data to be returned.
         * </p>
         * 
         * @param endTime
         *        The timestamp indicating the latest data to be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The granularity, in seconds, of the returned data points.
         * </p>
         * <note>
         * <p>
         * Bucket storage metrics are reported once per day. Therefore, you should specify a period of 86400 seconds,
         * which is the number of seconds in a day.
         * </p>
         * </note>
         * 
         * @param period
         *        The granularity, in seconds, of the returned data points.</p> <note>
         *        <p>
         *        Bucket storage metrics are reported once per day. Therefore, you should specify a period of 86400
         *        seconds, which is the number of seconds in a day.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified period.
         * By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you can determine
         * the full scope of a metric and how close the average use is to the <code>Minimum</code> and
         * <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic
         *        to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified
         *        period. By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you
         *        can determine the full scope of a metric and how close the average use is to the <code>Minimum</code>
         *        and <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your
         *        resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(Collection<String> statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified period.
         * By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you can determine
         * the full scope of a metric and how close the average use is to the <code>Minimum</code> and
         * <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic
         *        to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified
         *        period. By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you
         *        can determine the full scope of a metric and how close the average use is to the <code>Minimum</code>
         *        and <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your
         *        resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(String... statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified period.
         * By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you can determine
         * the full scope of a metric and how close the average use is to the <code>Minimum</code> and
         * <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic
         *        to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified
         *        period. By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you
         *        can determine the full scope of a metric and how close the average use is to the <code>Minimum</code>
         *        and <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your
         *        resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(Collection<MetricStatistic> statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified period.
         * By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you can determine
         * the full scope of a metric and how close the average use is to the <code>Minimum</code> and
         * <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - The sum of all values submitted for the matching metric. You can use this statistic
         *        to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of <code>Sum</code> / <code>SampleCount</code> during the specified
         *        period. By comparing this statistic with the <code>Minimum</code> and <code>Maximum</code> values, you
         *        can determine the full scope of a metric and how close the average use is to the <code>Minimum</code>
         *        and <code>Maximum</code> values. This comparison helps you to know when to increase or decrease your
         *        resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(MetricStatistic... statistics);

        /**
         * <p>
         * The unit for the metric data request.
         * </p>
         * <p>
         * Valid units depend on the metric data being requested. For the valid units with each available metric, see
         * the <code>metricName</code> parameter.
         * </p>
         * 
         * @param unit
         *        The unit for the metric data request.</p>
         *        <p>
         *        Valid units depend on the metric data being requested. For the valid units with each available metric,
         *        see the <code>metricName</code> parameter.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit for the metric data request.
         * </p>
         * <p>
         * Valid units depend on the metric data being requested. For the valid units with each available metric, see
         * the <code>metricName</code> parameter.
         * </p>
         * 
         * @param unit
         *        The unit for the metric data request.</p>
         *        <p>
         *        Valid units depend on the metric data being requested. For the valid units with each available metric,
         *        see the <code>metricName</code> parameter.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(MetricUnit unit);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LightsailRequest.BuilderImpl implements Builder {
        private String bucketName;

        private String metricName;

        private Instant startTime;

        private Instant endTime;

        private Integer period;

        private List<String> statistics = DefaultSdkAutoConstructList.getInstance();

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(GetBucketMetricDataRequest model) {
            super(model);
            bucketName(model.bucketName);
            metricName(model.metricName);
            startTime(model.startTime);
            endTime(model.endTime);
            period(model.period);
            statisticsWithStrings(model.statistics);
            unit(model.unit);
        }

        public final String getBucketName() {
            return bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public final Builder metricName(BucketMetricName metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final Integer getPeriod() {
            return period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        public final Collection<String> getStatistics() {
            if (statistics instanceof SdkAutoConstructList) {
                return null;
            }
            return statistics;
        }

        @Override
        public final Builder statisticsWithStrings(Collection<String> statistics) {
            this.statistics = MetricStatisticListCopier.copy(statistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statisticsWithStrings(String... statistics) {
            statisticsWithStrings(Arrays.asList(statistics));
            return this;
        }

        @Override
        public final Builder statistics(Collection<MetricStatistic> statistics) {
            this.statistics = MetricStatisticListCopier.copyEnumToString(statistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statistics(MetricStatistic... statistics) {
            statistics(Arrays.asList(statistics));
            return this;
        }

        public final void setStatistics(Collection<String> statistics) {
            this.statistics = MetricStatisticListCopier.copy(statistics);
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(MetricUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetBucketMetricDataRequest build() {
            return new GetBucketMetricDataRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
