/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an access key for an Amazon Lightsail bucket.
 * </p>
 * <p>
 * Access keys grant full programmatic access to the specified bucket and its objects. You can have a maximum of two
 * access keys per bucket. Use the <a>CreateBucketAccessKey</a> action to create an access key for a specific bucket.
 * For more information about access keys, see <a
 * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-creating-bucket-access-keys">Creating
 * access keys for a bucket in Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
 * </p>
 * <important>
 * <p>
 * The <code>secretAccessKey</code> value is returned only in response to the <code>CreateBucketAccessKey</code> action.
 * You can get a secret access key only when you first create an access key; you cannot get the secret access key later.
 * If you lose the secret access key, you must create a new access key.
 * </p>
 * </important>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccessKey implements SdkPojo, Serializable, ToCopyableBuilder<AccessKey.Builder, AccessKey> {
    private static final SdkField<String> ACCESS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accessKeyId").getter(getter(AccessKey::accessKeyId)).setter(setter(Builder::accessKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessKeyId").build()).build();

    private static final SdkField<String> SECRET_ACCESS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("secretAccessKey").getter(getter(AccessKey::secretAccessKey)).setter(setter(Builder::secretAccessKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secretAccessKey").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AccessKey::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(AccessKey::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCESS_KEY_ID_FIELD,
            SECRET_ACCESS_KEY_FIELD, STATUS_FIELD, CREATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accessKeyId;

    private final String secretAccessKey;

    private final String status;

    private final Instant createdAt;

    private AccessKey(BuilderImpl builder) {
        this.accessKeyId = builder.accessKeyId;
        this.secretAccessKey = builder.secretAccessKey;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
    }

    /**
     * <p>
     * The ID of the access key.
     * </p>
     * 
     * @return The ID of the access key.
     */
    public final String accessKeyId() {
        return accessKeyId;
    }

    /**
     * <p>
     * The secret access key used to sign requests.
     * </p>
     * <p>
     * You should store the secret access key in a safe location. We recommend that you delete the access key if the
     * secret access key is compromised.
     * </p>
     * 
     * @return The secret access key used to sign requests.</p>
     *         <p>
     *         You should store the secret access key in a safe location. We recommend that you delete the access key if
     *         the secret access key is compromised.
     */
    public final String secretAccessKey() {
        return secretAccessKey;
    }

    /**
     * <p>
     * The status of the access key.
     * </p>
     * <p>
     * A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is not.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the access key.</p>
     *         <p>
     *         A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is not.
     * @see StatusType
     */
    public final StatusType status() {
        return StatusType.fromValue(status);
    }

    /**
     * <p>
     * The status of the access key.
     * </p>
     * <p>
     * A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is not.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the access key.</p>
     *         <p>
     *         A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is not.
     * @see StatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The timestamp when the access key was created.
     * </p>
     * 
     * @return The timestamp when the access key was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accessKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(secretAccessKey());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccessKey)) {
            return false;
        }
        AccessKey other = (AccessKey) obj;
        return Objects.equals(accessKeyId(), other.accessKeyId()) && Objects.equals(secretAccessKey(), other.secretAccessKey())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdAt(), other.createdAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AccessKey").add("AccessKeyId", accessKeyId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretAccessKey", secretAccessKey()).add("Status", statusAsString()).add("CreatedAt", createdAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accessKeyId":
            return Optional.ofNullable(clazz.cast(accessKeyId()));
        case "secretAccessKey":
            return Optional.ofNullable(clazz.cast(secretAccessKey()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AccessKey, T> g) {
        return obj -> g.apply((AccessKey) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccessKey> {
        /**
         * <p>
         * The ID of the access key.
         * </p>
         * 
         * @param accessKeyId
         *        The ID of the access key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessKeyId(String accessKeyId);

        /**
         * <p>
         * The secret access key used to sign requests.
         * </p>
         * <p>
         * You should store the secret access key in a safe location. We recommend that you delete the access key if the
         * secret access key is compromised.
         * </p>
         * 
         * @param secretAccessKey
         *        The secret access key used to sign requests.</p>
         *        <p>
         *        You should store the secret access key in a safe location. We recommend that you delete the access key
         *        if the secret access key is compromised.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretAccessKey(String secretAccessKey);

        /**
         * <p>
         * The status of the access key.
         * </p>
         * <p>
         * A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is not.
         * </p>
         * 
         * @param status
         *        The status of the access key.</p>
         *        <p>
         *        A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is
         *        not.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the access key.
         * </p>
         * <p>
         * A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is not.
         * </p>
         * 
         * @param status
         *        The status of the access key.</p>
         *        <p>
         *        A status of <code>Active</code> means that the key is valid, while <code>Inactive</code> means it is
         *        not.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder status(StatusType status);

        /**
         * <p>
         * The timestamp when the access key was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the access key was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);
    }

    static final class BuilderImpl implements Builder {
        private String accessKeyId;

        private String secretAccessKey;

        private String status;

        private Instant createdAt;

        private BuilderImpl() {
        }

        private BuilderImpl(AccessKey model) {
            accessKeyId(model.accessKeyId);
            secretAccessKey(model.secretAccessKey);
            status(model.status);
            createdAt(model.createdAt);
        }

        public final String getAccessKeyId() {
            return accessKeyId;
        }

        public final void setAccessKeyId(String accessKeyId) {
            this.accessKeyId = accessKeyId;
        }

        @Override
        @Transient
        public final Builder accessKeyId(String accessKeyId) {
            this.accessKeyId = accessKeyId;
            return this;
        }

        public final String getSecretAccessKey() {
            return secretAccessKey;
        }

        public final void setSecretAccessKey(String secretAccessKey) {
            this.secretAccessKey = secretAccessKey;
        }

        @Override
        @Transient
        public final Builder secretAccessKey(String secretAccessKey) {
            this.secretAccessKey = secretAccessKey;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(StatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        @Transient
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        @Override
        public AccessKey build() {
            return new AccessKey(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
