/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the settings of a container that will be launched, or that is launched, to an Amazon Lightsail container
 * service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Container implements SdkPojo, Serializable, ToCopyableBuilder<Container.Builder, Container> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(Container::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(Container::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("environment")
            .getter(getter(Container::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> PORTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ports")
            .getter(getter(Container::portsAsStrings))
            .setter(setter(Builder::portsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ports").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD, COMMAND_FIELD,
            ENVIRONMENT_FIELD, PORTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String image;

    private final List<String> command;

    private final Map<String, String> environment;

    private final Map<String, String> ports;

    private Container(BuilderImpl builder) {
        this.image = builder.image;
        this.command = builder.command;
        this.environment = builder.environment;
        this.ports = builder.ports;
    }

    /**
     * <p>
     * The name of the image used for the container.
     * </p>
     * <p>
     * Container images sourced from your Lightsail container service, that are registered and stored on your service,
     * start with a colon (<code>:</code>). For example, <code>:container-service-1.mystaticwebsite.1</code>. Container
     * images sourced from a public registry like Docker Hub don't start with a colon. For example,
     * <code>nginx:latest</code> or <code>nginx</code>.
     * </p>
     * 
     * @return The name of the image used for the container.</p>
     *         <p>
     *         Container images sourced from your Lightsail container service, that are registered and stored on your
     *         service, start with a colon (<code>:</code>). For example,
     *         <code>:container-service-1.mystaticwebsite.1</code>. Container images sourced from a public registry like
     *         Docker Hub don't start with a colon. For example, <code>nginx:latest</code> or <code>nginx</code>.
     */
    public final String image() {
        return image;
    }

    /**
     * Returns true if the Command property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The launch command for the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCommand()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The launch command for the container.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * Returns true if the Environment property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables of the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The environment variables of the container.
     */
    public final Map<String, String> environment() {
        return environment;
    }

    /**
     * <p>
     * The open firewall ports of the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPorts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The open firewall ports of the container.
     */
    public final Map<String, ContainerServiceProtocol> ports() {
        return PortMapCopier.copyStringToEnum(ports);
    }

    /**
     * Returns true if the Ports property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasPorts() {
        return ports != null && !(ports instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The open firewall ports of the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPorts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The open firewall ports of the container.
     */
    public final Map<String, String> portsAsStrings() {
        return ports;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPorts() ? portsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Container)) {
            return false;
        }
        Container other = (Container) obj;
        return Objects.equals(image(), other.image()) && hasCommand() == other.hasCommand()
                && Objects.equals(command(), other.command()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && hasPorts() == other.hasPorts()
                && Objects.equals(portsAsStrings(), other.portsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Container").add("Image", image()).add("Command", hasCommand() ? command() : null)
                .add("Environment", hasEnvironment() ? environment() : null).add("Ports", hasPorts() ? portsAsStrings() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "ports":
            return Optional.ofNullable(clazz.cast(portsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Container, T> g) {
        return obj -> g.apply((Container) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Container> {
        /**
         * <p>
         * The name of the image used for the container.
         * </p>
         * <p>
         * Container images sourced from your Lightsail container service, that are registered and stored on your
         * service, start with a colon (<code>:</code>). For example,
         * <code>:container-service-1.mystaticwebsite.1</code>. Container images sourced from a public registry like
         * Docker Hub don't start with a colon. For example, <code>nginx:latest</code> or <code>nginx</code>.
         * </p>
         * 
         * @param image
         *        The name of the image used for the container.</p>
         *        <p>
         *        Container images sourced from your Lightsail container service, that are registered and stored on your
         *        service, start with a colon (<code>:</code>). For example,
         *        <code>:container-service-1.mystaticwebsite.1</code>. Container images sourced from a public registry
         *        like Docker Hub don't start with a colon. For example, <code>nginx:latest</code> or <code>nginx</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The launch command for the container.
         * </p>
         * 
         * @param command
         *        The launch command for the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The launch command for the container.
         * </p>
         * 
         * @param command
         *        The launch command for the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The environment variables of the container.
         * </p>
         * 
         * @param environment
         *        The environment variables of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);

        /**
         * <p>
         * The open firewall ports of the container.
         * </p>
         * 
         * @param ports
         *        The open firewall ports of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portsWithStrings(Map<String, String> ports);

        /**
         * <p>
         * The open firewall ports of the container.
         * </p>
         * 
         * @param ports
         *        The open firewall ports of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ports(Map<String, ContainerServiceProtocol> ports);
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> ports = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Container model) {
            image(model.image);
            command(model.command);
            environment(model.environment);
            portsWithStrings(model.ports);
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        @Transient
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        @Transient
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = EnvironmentCopier.copy(environment);
        }

        @Override
        @Transient
        public final Builder environment(Map<String, String> environment) {
            this.environment = EnvironmentCopier.copy(environment);
            return this;
        }

        public final Map<String, String> getPorts() {
            if (ports instanceof SdkAutoConstructMap) {
                return null;
            }
            return ports;
        }

        public final void setPorts(Map<String, String> ports) {
            this.ports = PortMapCopier.copy(ports);
        }

        @Override
        @Transient
        public final Builder portsWithStrings(Map<String, String> ports) {
            this.ports = PortMapCopier.copy(ports);
            return this;
        }

        @Override
        @Transient
        public final Builder ports(Map<String, ContainerServiceProtocol> ports) {
            this.ports = PortMapCopier.copyEnumToString(ports);
            return this;
        }

        @Override
        public Container build() {
            return new Container(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
