/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon Lightsail bucket.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Bucket implements SdkPojo, Serializable, ToCopyableBuilder<Bucket.Builder, Bucket> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(Bucket::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<AccessRules> ACCESS_RULES_FIELD = SdkField.<AccessRules> builder(MarshallingType.SDK_POJO)
            .memberName("accessRules").getter(getter(Bucket::accessRules)).setter(setter(Builder::accessRules))
            .constructor(AccessRules::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessRules").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Bucket::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> BUNDLE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bundleId").getter(getter(Bucket::bundleId)).setter(setter(Builder::bundleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bundleId").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Bucket::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("url")
            .getter(getter(Bucket::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).memberName("location").getter(getter(Bucket::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Bucket::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("supportCode").getter(getter(Bucket::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(Bucket::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OBJECT_VERSIONING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("objectVersioning").getter(getter(Bucket::objectVersioning)).setter(setter(Builder::objectVersioning))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("objectVersioning").build()).build();

    private static final SdkField<Boolean> ABLE_TO_UPDATE_BUNDLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ableToUpdateBundle").getter(getter(Bucket::ableToUpdateBundle))
            .setter(setter(Builder::ableToUpdateBundle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ableToUpdateBundle").build())
            .build();

    private static final SdkField<List<String>> READONLY_ACCESS_ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("readonlyAccessAccounts")
            .getter(getter(Bucket::readonlyAccessAccounts))
            .setter(setter(Builder::readonlyAccessAccounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("readonlyAccessAccounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceReceivingAccess>> RESOURCES_RECEIVING_ACCESS_FIELD = SdkField
            .<List<ResourceReceivingAccess>> builder(MarshallingType.LIST)
            .memberName("resourcesReceivingAccess")
            .getter(getter(Bucket::resourcesReceivingAccess))
            .setter(setter(Builder::resourcesReceivingAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourcesReceivingAccess").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceReceivingAccess> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceReceivingAccess::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<BucketState> STATE_FIELD = SdkField.<BucketState> builder(MarshallingType.SDK_POJO)
            .memberName("state").getter(getter(Bucket::state)).setter(setter(Builder::state)).constructor(BucketState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<BucketAccessLogConfig> ACCESS_LOG_CONFIG_FIELD = SdkField
            .<BucketAccessLogConfig> builder(MarshallingType.SDK_POJO).memberName("accessLogConfig")
            .getter(getter(Bucket::accessLogConfig)).setter(setter(Builder::accessLogConfig))
            .constructor(BucketAccessLogConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessLogConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            ACCESS_RULES_FIELD, ARN_FIELD, BUNDLE_ID_FIELD, CREATED_AT_FIELD, URL_FIELD, LOCATION_FIELD, NAME_FIELD,
            SUPPORT_CODE_FIELD, TAGS_FIELD, OBJECT_VERSIONING_FIELD, ABLE_TO_UPDATE_BUNDLE_FIELD, READONLY_ACCESS_ACCOUNTS_FIELD,
            RESOURCES_RECEIVING_ACCESS_FIELD, STATE_FIELD, ACCESS_LOG_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final AccessRules accessRules;

    private final String arn;

    private final String bundleId;

    private final Instant createdAt;

    private final String url;

    private final ResourceLocation location;

    private final String name;

    private final String supportCode;

    private final List<Tag> tags;

    private final String objectVersioning;

    private final Boolean ableToUpdateBundle;

    private final List<String> readonlyAccessAccounts;

    private final List<ResourceReceivingAccess> resourcesReceivingAccess;

    private final BucketState state;

    private final BucketAccessLogConfig accessLogConfig;

    private Bucket(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.accessRules = builder.accessRules;
        this.arn = builder.arn;
        this.bundleId = builder.bundleId;
        this.createdAt = builder.createdAt;
        this.url = builder.url;
        this.location = builder.location;
        this.name = builder.name;
        this.supportCode = builder.supportCode;
        this.tags = builder.tags;
        this.objectVersioning = builder.objectVersioning;
        this.ableToUpdateBundle = builder.ableToUpdateBundle;
        this.readonlyAccessAccounts = builder.readonlyAccessAccounts;
        this.resourcesReceivingAccess = builder.resourcesReceivingAccess;
        this.state = builder.state;
        this.accessLogConfig = builder.accessLogConfig;
    }

    /**
     * <p>
     * The Lightsail resource type of the bucket (for example, <code>Bucket</code>).
     * </p>
     * 
     * @return The Lightsail resource type of the bucket (for example, <code>Bucket</code>).
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * An object that describes the access rules of the bucket.
     * </p>
     * 
     * @return An object that describes the access rules of the bucket.
     */
    public final AccessRules accessRules() {
        return accessRules;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the bucket.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the bucket.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The ID of the bundle currently applied to the bucket.
     * </p>
     * <p>
     * A bucket bundle specifies the monthly cost, storage space, and data transfer quota for a bucket.
     * </p>
     * <p>
     * Use the <a>UpdateBucketBundle</a> action to change the bundle of a bucket.
     * </p>
     * 
     * @return The ID of the bundle currently applied to the bucket.</p>
     *         <p>
     *         A bucket bundle specifies the monthly cost, storage space, and data transfer quota for a bucket.
     *         </p>
     *         <p>
     *         Use the <a>UpdateBucketBundle</a> action to change the bundle of a bucket.
     */
    public final String bundleId() {
        return bundleId;
    }

    /**
     * <p>
     * The timestamp when the distribution was created.
     * </p>
     * 
     * @return The timestamp when the distribution was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The URL of the bucket.
     * </p>
     * 
     * @return The URL of the bucket.
     */
    public final String url() {
        return url;
    }

    /**
     * Returns the value of the Location property for this object.
     * 
     * @return The value of the Location property for this object.
     */
    public final ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The name of the bucket.
     * </p>
     * 
     * @return The name of the bucket.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The support code for a bucket. Include this code in your email to support when you have questions about a
     * Lightsail bucket. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code for a bucket. Include this code in your email to support when you have questions about a
     *         Lightsail bucket. This code enables our support team to look up your Lightsail information more easily.
     */
    public final String supportCode() {
        return supportCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the bucket. For more information, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon Lightsail</a> in
     * the <i>Amazon Lightsail Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tag keys and optional values for the bucket. For more information, see <a
     *         href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon
     *         Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Indicates whether object versioning is enabled for the bucket.
     * </p>
     * <p>
     * The following options can be configured:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Enabled</code> - Object versioning is enabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Suspended</code> - Object versioning was previously enabled but is currently suspended. Existing object
     * versions are retained.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NeverEnabled</code> - Object versioning has never been enabled.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Indicates whether object versioning is enabled for the bucket.</p>
     *         <p>
     *         The following options can be configured:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Enabled</code> - Object versioning is enabled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Suspended</code> - Object versioning was previously enabled but is currently suspended. Existing
     *         object versions are retained.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NeverEnabled</code> - Object versioning has never been enabled.
     *         </p>
     *         </li>
     */
    public final String objectVersioning() {
        return objectVersioning;
    }

    /**
     * <p>
     * Indicates whether the bundle that is currently applied to a bucket can be changed to another bundle.
     * </p>
     * <p>
     * You can update a bucket's bundle only one time within a monthly AWS billing cycle.
     * </p>
     * <p>
     * Use the <a>UpdateBucketBundle</a> action to change a bucket's bundle.
     * </p>
     * 
     * @return Indicates whether the bundle that is currently applied to a bucket can be changed to another bundle.</p>
     *         <p>
     *         You can update a bucket's bundle only one time within a monthly AWS billing cycle.
     *         </p>
     *         <p>
     *         Use the <a>UpdateBucketBundle</a> action to change a bucket's bundle.
     */
    public final Boolean ableToUpdateBundle() {
        return ableToUpdateBundle;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReadonlyAccessAccounts property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasReadonlyAccessAccounts() {
        return readonlyAccessAccounts != null && !(readonlyAccessAccounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings that specify the AWS account IDs that have read-only access to the bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReadonlyAccessAccounts} method.
     * </p>
     * 
     * @return An array of strings that specify the AWS account IDs that have read-only access to the bucket.
     */
    public final List<String> readonlyAccessAccounts() {
        return readonlyAccessAccounts;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourcesReceivingAccess property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasResourcesReceivingAccess() {
        return resourcesReceivingAccess != null && !(resourcesReceivingAccess instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe Lightsail instances that have access to the bucket.
     * </p>
     * <p>
     * Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourcesReceivingAccess} method.
     * </p>
     * 
     * @return An array of objects that describe Lightsail instances that have access to the bucket.</p>
     *         <p>
     *         Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
     */
    public final List<ResourceReceivingAccess> resourcesReceivingAccess() {
        return resourcesReceivingAccess;
    }

    /**
     * <p>
     * An object that describes the state of the bucket.
     * </p>
     * 
     * @return An object that describes the state of the bucket.
     */
    public final BucketState state() {
        return state;
    }

    /**
     * <p>
     * An object that describes the access log configuration for the bucket.
     * </p>
     * 
     * @return An object that describes the access log configuration for the bucket.
     */
    public final BucketAccessLogConfig accessLogConfig() {
        return accessLogConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(accessRules());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(bundleId());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(url());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(objectVersioning());
        hashCode = 31 * hashCode + Objects.hashCode(ableToUpdateBundle());
        hashCode = 31 * hashCode + Objects.hashCode(hasReadonlyAccessAccounts() ? readonlyAccessAccounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourcesReceivingAccess() ? resourcesReceivingAccess() : null);
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(accessLogConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Bucket)) {
            return false;
        }
        Bucket other = (Bucket) obj;
        return Objects.equals(resourceType(), other.resourceType()) && Objects.equals(accessRules(), other.accessRules())
                && Objects.equals(arn(), other.arn()) && Objects.equals(bundleId(), other.bundleId())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(url(), other.url())
                && Objects.equals(location(), other.location()) && Objects.equals(name(), other.name())
                && Objects.equals(supportCode(), other.supportCode()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(objectVersioning(), other.objectVersioning())
                && Objects.equals(ableToUpdateBundle(), other.ableToUpdateBundle())
                && hasReadonlyAccessAccounts() == other.hasReadonlyAccessAccounts()
                && Objects.equals(readonlyAccessAccounts(), other.readonlyAccessAccounts())
                && hasResourcesReceivingAccess() == other.hasResourcesReceivingAccess()
                && Objects.equals(resourcesReceivingAccess(), other.resourcesReceivingAccess())
                && Objects.equals(state(), other.state()) && Objects.equals(accessLogConfig(), other.accessLogConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Bucket").add("ResourceType", resourceType()).add("AccessRules", accessRules()).add("Arn", arn())
                .add("BundleId", bundleId()).add("CreatedAt", createdAt()).add("Url", url()).add("Location", location())
                .add("Name", name()).add("SupportCode", supportCode()).add("Tags", hasTags() ? tags() : null)
                .add("ObjectVersioning", objectVersioning()).add("AbleToUpdateBundle", ableToUpdateBundle())
                .add("ReadonlyAccessAccounts", hasReadonlyAccessAccounts() ? readonlyAccessAccounts() : null)
                .add("ResourcesReceivingAccess", hasResourcesReceivingAccess() ? resourcesReceivingAccess() : null)
                .add("State", state()).add("AccessLogConfig", accessLogConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "accessRules":
            return Optional.ofNullable(clazz.cast(accessRules()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "bundleId":
            return Optional.ofNullable(clazz.cast(bundleId()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "url":
            return Optional.ofNullable(clazz.cast(url()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "objectVersioning":
            return Optional.ofNullable(clazz.cast(objectVersioning()));
        case "ableToUpdateBundle":
            return Optional.ofNullable(clazz.cast(ableToUpdateBundle()));
        case "readonlyAccessAccounts":
            return Optional.ofNullable(clazz.cast(readonlyAccessAccounts()));
        case "resourcesReceivingAccess":
            return Optional.ofNullable(clazz.cast(resourcesReceivingAccess()));
        case "state":
            return Optional.ofNullable(clazz.cast(state()));
        case "accessLogConfig":
            return Optional.ofNullable(clazz.cast(accessLogConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Bucket, T> g) {
        return obj -> g.apply((Bucket) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Bucket> {
        /**
         * <p>
         * The Lightsail resource type of the bucket (for example, <code>Bucket</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type of the bucket (for example, <code>Bucket</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * An object that describes the access rules of the bucket.
         * </p>
         * 
         * @param accessRules
         *        An object that describes the access rules of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessRules(AccessRules accessRules);

        /**
         * <p>
         * An object that describes the access rules of the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link AccessRules.Builder} avoiding the need to create
         * one manually via {@link AccessRules#builder()}.
         *
         * When the {@link Consumer} completes, {@link AccessRules.Builder#build()} is called immediately and its result
         * is passed to {@link #accessRules(AccessRules)}.
         * 
         * @param accessRules
         *        a consumer that will call methods on {@link AccessRules.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessRules(AccessRules)
         */
        default Builder accessRules(Consumer<AccessRules.Builder> accessRules) {
            return accessRules(AccessRules.builder().applyMutation(accessRules).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the bucket.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ID of the bundle currently applied to the bucket.
         * </p>
         * <p>
         * A bucket bundle specifies the monthly cost, storage space, and data transfer quota for a bucket.
         * </p>
         * <p>
         * Use the <a>UpdateBucketBundle</a> action to change the bundle of a bucket.
         * </p>
         * 
         * @param bundleId
         *        The ID of the bundle currently applied to the bucket.</p>
         *        <p>
         *        A bucket bundle specifies the monthly cost, storage space, and data transfer quota for a bucket.
         *        </p>
         *        <p>
         *        Use the <a>UpdateBucketBundle</a> action to change the bundle of a bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bundleId(String bundleId);

        /**
         * <p>
         * The timestamp when the distribution was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the distribution was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The URL of the bucket.
         * </p>
         * 
         * @param url
         *        The URL of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);

        /**
         * Sets the value of the Location property for this object.
         *
         * @param location
         *        The new value for the Location property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * Sets the value of the Location property for this object.
         *
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The name of the bucket.
         * </p>
         * 
         * @param name
         *        The name of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The support code for a bucket. Include this code in your email to support when you have questions about a
         * Lightsail bucket. This code enables our support team to look up your Lightsail information more easily.
         * </p>
         * 
         * @param supportCode
         *        The support code for a bucket. Include this code in your email to support when you have questions
         *        about a Lightsail bucket. This code enables our support team to look up your Lightsail information
         *        more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The tag keys and optional values for the bucket. For more information, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon
         * Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the bucket. For more information, see <a
         *        href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon
         *        Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the bucket. For more information, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon
         * Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the bucket. For more information, see <a
         *        href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon
         *        Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the bucket. For more information, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Tags in Amazon
         * Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Indicates whether object versioning is enabled for the bucket.
         * </p>
         * <p>
         * The following options can be configured:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Enabled</code> - Object versioning is enabled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Suspended</code> - Object versioning was previously enabled but is currently suspended. Existing object
         * versions are retained.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NeverEnabled</code> - Object versioning has never been enabled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param objectVersioning
         *        Indicates whether object versioning is enabled for the bucket.</p>
         *        <p>
         *        The following options can be configured:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Enabled</code> - Object versioning is enabled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Suspended</code> - Object versioning was previously enabled but is currently suspended. Existing
         *        object versions are retained.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NeverEnabled</code> - Object versioning has never been enabled.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectVersioning(String objectVersioning);

        /**
         * <p>
         * Indicates whether the bundle that is currently applied to a bucket can be changed to another bundle.
         * </p>
         * <p>
         * You can update a bucket's bundle only one time within a monthly AWS billing cycle.
         * </p>
         * <p>
         * Use the <a>UpdateBucketBundle</a> action to change a bucket's bundle.
         * </p>
         * 
         * @param ableToUpdateBundle
         *        Indicates whether the bundle that is currently applied to a bucket can be changed to another
         *        bundle.</p>
         *        <p>
         *        You can update a bucket's bundle only one time within a monthly AWS billing cycle.
         *        </p>
         *        <p>
         *        Use the <a>UpdateBucketBundle</a> action to change a bucket's bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ableToUpdateBundle(Boolean ableToUpdateBundle);

        /**
         * <p>
         * An array of strings that specify the AWS account IDs that have read-only access to the bucket.
         * </p>
         * 
         * @param readonlyAccessAccounts
         *        An array of strings that specify the AWS account IDs that have read-only access to the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyAccessAccounts(Collection<String> readonlyAccessAccounts);

        /**
         * <p>
         * An array of strings that specify the AWS account IDs that have read-only access to the bucket.
         * </p>
         * 
         * @param readonlyAccessAccounts
         *        An array of strings that specify the AWS account IDs that have read-only access to the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyAccessAccounts(String... readonlyAccessAccounts);

        /**
         * <p>
         * An array of objects that describe Lightsail instances that have access to the bucket.
         * </p>
         * <p>
         * Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
         * </p>
         * 
         * @param resourcesReceivingAccess
         *        An array of objects that describe Lightsail instances that have access to the bucket.</p>
         *        <p>
         *        Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesReceivingAccess(Collection<ResourceReceivingAccess> resourcesReceivingAccess);

        /**
         * <p>
         * An array of objects that describe Lightsail instances that have access to the bucket.
         * </p>
         * <p>
         * Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
         * </p>
         * 
         * @param resourcesReceivingAccess
         *        An array of objects that describe Lightsail instances that have access to the bucket.</p>
         *        <p>
         *        Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesReceivingAccess(ResourceReceivingAccess... resourcesReceivingAccess);

        /**
         * <p>
         * An array of objects that describe Lightsail instances that have access to the bucket.
         * </p>
         * <p>
         * Use the <a>SetResourceAccessForBucket</a> action to update the instances that have access to a bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceReceivingAccess>.Builder} avoiding
         * the need to create one manually via {@link List<ResourceReceivingAccess>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceReceivingAccess>.Builder#build()} is called
         * immediately and its result is passed to {@link #resourcesReceivingAccess(List<ResourceReceivingAccess>)}.
         * 
         * @param resourcesReceivingAccess
         *        a consumer that will call methods on {@link List<ResourceReceivingAccess>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourcesReceivingAccess(List<ResourceReceivingAccess>)
         */
        Builder resourcesReceivingAccess(Consumer<ResourceReceivingAccess.Builder>... resourcesReceivingAccess);

        /**
         * <p>
         * An object that describes the state of the bucket.
         * </p>
         * 
         * @param state
         *        An object that describes the state of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(BucketState state);

        /**
         * <p>
         * An object that describes the state of the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link BucketState.Builder} avoiding the need to create
         * one manually via {@link BucketState#builder()}.
         *
         * When the {@link Consumer} completes, {@link BucketState.Builder#build()} is called immediately and its result
         * is passed to {@link #state(BucketState)}.
         * 
         * @param state
         *        a consumer that will call methods on {@link BucketState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #state(BucketState)
         */
        default Builder state(Consumer<BucketState.Builder> state) {
            return state(BucketState.builder().applyMutation(state).build());
        }

        /**
         * <p>
         * An object that describes the access log configuration for the bucket.
         * </p>
         * 
         * @param accessLogConfig
         *        An object that describes the access log configuration for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessLogConfig(BucketAccessLogConfig accessLogConfig);

        /**
         * <p>
         * An object that describes the access log configuration for the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link BucketAccessLogConfig.Builder} avoiding the need
         * to create one manually via {@link BucketAccessLogConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link BucketAccessLogConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #accessLogConfig(BucketAccessLogConfig)}.
         * 
         * @param accessLogConfig
         *        a consumer that will call methods on {@link BucketAccessLogConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessLogConfig(BucketAccessLogConfig)
         */
        default Builder accessLogConfig(Consumer<BucketAccessLogConfig.Builder> accessLogConfig) {
            return accessLogConfig(BucketAccessLogConfig.builder().applyMutation(accessLogConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private AccessRules accessRules;

        private String arn;

        private String bundleId;

        private Instant createdAt;

        private String url;

        private ResourceLocation location;

        private String name;

        private String supportCode;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String objectVersioning;

        private Boolean ableToUpdateBundle;

        private List<String> readonlyAccessAccounts = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceReceivingAccess> resourcesReceivingAccess = DefaultSdkAutoConstructList.getInstance();

        private BucketState state;

        private BucketAccessLogConfig accessLogConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Bucket model) {
            resourceType(model.resourceType);
            accessRules(model.accessRules);
            arn(model.arn);
            bundleId(model.bundleId);
            createdAt(model.createdAt);
            url(model.url);
            location(model.location);
            name(model.name);
            supportCode(model.supportCode);
            tags(model.tags);
            objectVersioning(model.objectVersioning);
            ableToUpdateBundle(model.ableToUpdateBundle);
            readonlyAccessAccounts(model.readonlyAccessAccounts);
            resourcesReceivingAccess(model.resourcesReceivingAccess);
            state(model.state);
            accessLogConfig(model.accessLogConfig);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        @Transient
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final AccessRules.Builder getAccessRules() {
            return accessRules != null ? accessRules.toBuilder() : null;
        }

        public final void setAccessRules(AccessRules.BuilderImpl accessRules) {
            this.accessRules = accessRules != null ? accessRules.build() : null;
        }

        @Override
        @Transient
        public final Builder accessRules(AccessRules accessRules) {
            this.accessRules = accessRules;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getBundleId() {
            return bundleId;
        }

        public final void setBundleId(String bundleId) {
            this.bundleId = bundleId;
        }

        @Override
        @Transient
        public final Builder bundleId(String bundleId) {
            this.bundleId = bundleId;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        @Transient
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        @Transient
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        @Override
        @Transient
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        @Override
        @Transient
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getObjectVersioning() {
            return objectVersioning;
        }

        public final void setObjectVersioning(String objectVersioning) {
            this.objectVersioning = objectVersioning;
        }

        @Override
        @Transient
        public final Builder objectVersioning(String objectVersioning) {
            this.objectVersioning = objectVersioning;
            return this;
        }

        public final Boolean getAbleToUpdateBundle() {
            return ableToUpdateBundle;
        }

        public final void setAbleToUpdateBundle(Boolean ableToUpdateBundle) {
            this.ableToUpdateBundle = ableToUpdateBundle;
        }

        @Override
        @Transient
        public final Builder ableToUpdateBundle(Boolean ableToUpdateBundle) {
            this.ableToUpdateBundle = ableToUpdateBundle;
            return this;
        }

        public final Collection<String> getReadonlyAccessAccounts() {
            if (readonlyAccessAccounts instanceof SdkAutoConstructList) {
                return null;
            }
            return readonlyAccessAccounts;
        }

        public final void setReadonlyAccessAccounts(Collection<String> readonlyAccessAccounts) {
            this.readonlyAccessAccounts = PartnerIdListCopier.copy(readonlyAccessAccounts);
        }

        @Override
        @Transient
        public final Builder readonlyAccessAccounts(Collection<String> readonlyAccessAccounts) {
            this.readonlyAccessAccounts = PartnerIdListCopier.copy(readonlyAccessAccounts);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder readonlyAccessAccounts(String... readonlyAccessAccounts) {
            readonlyAccessAccounts(Arrays.asList(readonlyAccessAccounts));
            return this;
        }

        public final List<ResourceReceivingAccess.Builder> getResourcesReceivingAccess() {
            List<ResourceReceivingAccess.Builder> result = AccessReceiverListCopier.copyToBuilder(this.resourcesReceivingAccess);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourcesReceivingAccess(Collection<ResourceReceivingAccess.BuilderImpl> resourcesReceivingAccess) {
            this.resourcesReceivingAccess = AccessReceiverListCopier.copyFromBuilder(resourcesReceivingAccess);
        }

        @Override
        @Transient
        public final Builder resourcesReceivingAccess(Collection<ResourceReceivingAccess> resourcesReceivingAccess) {
            this.resourcesReceivingAccess = AccessReceiverListCopier.copy(resourcesReceivingAccess);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder resourcesReceivingAccess(ResourceReceivingAccess... resourcesReceivingAccess) {
            resourcesReceivingAccess(Arrays.asList(resourcesReceivingAccess));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder resourcesReceivingAccess(Consumer<ResourceReceivingAccess.Builder>... resourcesReceivingAccess) {
            resourcesReceivingAccess(Stream.of(resourcesReceivingAccess)
                    .map(c -> ResourceReceivingAccess.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final BucketState.Builder getState() {
            return state != null ? state.toBuilder() : null;
        }

        public final void setState(BucketState.BuilderImpl state) {
            this.state = state != null ? state.build() : null;
        }

        @Override
        @Transient
        public final Builder state(BucketState state) {
            this.state = state;
            return this;
        }

        public final BucketAccessLogConfig.Builder getAccessLogConfig() {
            return accessLogConfig != null ? accessLogConfig.toBuilder() : null;
        }

        public final void setAccessLogConfig(BucketAccessLogConfig.BuilderImpl accessLogConfig) {
            this.accessLogConfig = accessLogConfig != null ? accessLogConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder accessLogConfig(BucketAccessLogConfig accessLogConfig) {
            this.accessLogConfig = accessLogConfig;
            return this;
        }

        @Override
        public Bucket build() {
            return new Bucket(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
